! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> This module contains scheme property type and module related to scheme prop
module mus_scheme_header_module

  ! include treelm modules
  use env_module,         only: LabelLen
  use tem_tools_module,   only: tem_horizontalSpacer
  use tem_aux_module,     only: tem_abort
  use tem_logging_module, only: logUnit

  ! include aotus modules
  use aotus_module,     only: flu_State, aot_get_val, aoterr_NonExistent
  use aot_table_module, only: aot_table_open, aot_table_close
  use aot_out_module,   only: aot_out_type, aot_out_val, aot_out_open_table,   &
    &                         aot_out_close_table

  implicit none

  private

  public :: mus_scheme_header_type, mus_scheme_header_out
  public :: mus_load_scheme_header

  !> Datatype containing information to identify the scheme
  !!
  !! Combination of scheme kind, relaxation and layout%stencilKind
  !! are used to choose the correct compute kernel for the
  !! scheme
  type mus_scheme_header_type
    !> scheme kind, Ex: lbm, lbm_incomp, lbm_nNwtn
    character(len=labelLen) :: kind
    !> scheme layout, Ex: d3q19
    character(len=labelLen) :: layout
    !> scheme relaxation type Ex: BGK, MRT, bgk_pl, bgk_cy, bgk_cs...
    character(len=labelLen) :: relaxation
  end type mus_scheme_header_type

contains

! ****************************************************************************** !
  !> load scheme header info from lua file identify table or from scheme table
  !! or from config
  !!
  !! Load scheme label, kind, layoutKind and relaxation
  !!```lua
  !! identify = { kind = 'simType',
  !!              layout = 'stencilLayout',
  !!              relaxation = 'relaxationType' }
  !!```
  !! For a possible selection of the other parameters
  !! - simType: lbm, lbm_incomp, lbm_nNwtn, multi-species-liquid
  !! - [[mus_scheme_layout_module]]: d2q9, d3q19, ...
  !! - relaxationType: bgk, mrt, ...
  !!
  subroutine mus_load_scheme_header( me, conf, parent, scaling )
  ! -----------------------------------------------------------------------------
    !> returns scheme identify information
    type( mus_scheme_header_type ), intent(out) :: me
    type( flu_State ) :: conf !< flu state
    !> parent handle if scheme table is defined
    integer, intent(in), optional :: parent
    !> scaling: diffusive or acoustic?
    character(len=*), intent(in) :: scaling
    ! ---------------------------------------------------------------------------
    integer :: thandle !< handle for scheme identify table
    integer :: iError
    ! ---------------------------------------------------------------------------

    call tem_horizontalSpacer(fUnit = logUnit(1))
    write(logUnit(1),*) 'Loading Scheme identify table: '
    !default values
    me%kind = 'lbm'
    me%layout = 'd3q19'
    me%relaxation = 'bgk'

    call aot_table_open( L       = conf,      &
      &                  parent  = parent,    &
      &                  thandle = thandle,   &
      &                  key     = 'identify' )

    if( thandle .gt. 0 ) then
      ! get schemekind
      call aot_get_val( L       = conf,                                        &
        &               thandle = thandle,                                     &
        &               key     = 'kind',                                      &
        &               val     = me%kind,                                     &
        &               default = 'lbm',                                       &
        &               ErrCode = iError )

      ! get layoutkind
      call aot_get_val( L       = conf,                                        &
        &               thandle = thandle,                                     &
        &               key     = 'layout',                                    &
        &               val     = me%layout,                                   &
        &               default = 'd3q19',                                     &
        &               ErrCode = iError )

      ! get relaxation
      call aot_get_val( L       = conf,                                        &
        &               thandle = thandle,                                     &
        &               key     = 'relaxation',                                &
        &               val     = me%relaxation,                               &
        &               default = 'bgk',                                       &
        &               ErrCode = iError )

    else
      write(logUnit(1),*) 'Scheme Identify table not defined.'
      write(logUnit(1),*) 'Setting default values for scheme..'
    endif

    call aot_table_close( L=conf, thandle=thandle )

    write(logUnit(1),*) 'kind: '// trim( me%kind )
    write(logUnit(1),*) 'Layout: '// trim( me%layout )
    write(logUnit(1),*) 'relaxation: '// trim( me%relaxation )
    call tem_horizontalSpacer(fUnit = logUnit(1))

    ! Both multispeciees and poisson equation must have diffusive scaling
    ! since diffusive scaling is used to recover macroscopic equations
    ! from asymptotic analysis
    select case(trim(me%kind))
    case ('multi-species_gas', 'multi-species_liquid', 'poisson',  &
      &   'poisson_boltzmann_linear', 'poisson_boltzmann_nonlinear')
      if(trim(scaling) /= 'diffusive') then
         write(logUnit(1),*)'ERROR: Choose scaling = "diffusive" for '//     &
           &                'multi-species scheme'
         write(logUnit(1),*)'Aborting'
         call tem_abort()
      end if
    end select

  end subroutine mus_load_scheme_header
! ****************************************************************************** !

! ****************************************************************************** !
  !> Dumps scheme header
  subroutine mus_scheme_header_out( me, conf )
  ! -----------------------------------------------------------------------------
    !> returns scheme identify information
    type( mus_scheme_header_type ), intent(in) :: me
    type(aot_out_type) :: conf
    ! ---------------------------------------------------------------------------

    ! the label does not have to be outputted.
    ! because it is the name of the outputted varSys
    call aot_out_open_table( put_conf = conf, tname = 'identify' )
    call aot_out_val( put_conf = conf,                                         &
      &               vname    = 'kind',                                       &
      &               val      = trim( me%kind ))
    call aot_out_val( put_conf = conf,                                         &
      &               vname    = 'relaxation',                                 &
      &               val      = trim( me%relaxation ))
    call aot_out_val( put_conf = conf,                                         &
      &               vname    = 'layout',                                     &
      &               val      = trim( me%layout ))
    call aot_out_close_table( put_conf = conf )
  end subroutine mus_scheme_header_out
! ****************************************************************************** !

end module mus_scheme_header_module
! ****************************************************************************** !
