! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> This module contains data type of MRT. Also functions and routines to set
!! relaxation parameters for single-component and multi-species
module mus_relaxationParam_module

  ! include treelm modules
  use env_module,               only: rk, io_buffer_size
  use tem_param_module,         only: cs2inv
  use tem_time_module,          only: tem_time_type
  use tem_temporal_module,      only: tem_temporal_type, tem_temporal_for
  use tem_varSys_module,        only: tem_varSys_type
  use tem_spacetime_fun_module, only: tem_spacetime_fun_type, tem_spacetime_for
  use tem_logging_module,       only: logUnit
  use tem_aux_module,           only: tem_abort
  use tem_grow_array_module,    only: grw_realArray_type, init

  use mus_scheme_header_module, only: mus_scheme_header_type  
  use mus_scheme_layout_module, only: mus_scheme_layout_type
  use mus_moments_type_module,  only: mus_moment_type
  use mus_derVarPos_type_module, only: mus_derVarPos_type
  use mus_turbulence_module,    only: mus_turbulence_type, mus_turb_calcVisc
  use mus_mrtRelaxation_module, only: mus_mrt_type, mus_set_mrtRelaxation

  implicit none

  private

  public :: mus_viscosity_type
  public :: mus_update_viscKine
  public :: mus_update_relaxation
  public :: mus_update_relaxParamKine
  public :: mus_update_relaxParamFromViscSTfun
  public :: mus_init_relaxParam
  public :: mus_calcOmegaFromVisc

  !> Contains relaxation parameter for a level
  type mus_relaxationParam_type
    !> Relaxation parameter computed from viscosity
    !! For kinematic viscosity, if turbulence is active, this omega refers to
    !! effective omega which is omega_bg + omega_turb
    !! size: nElems_solve
    real(kind=rk), allocatable :: val(:)
  end type mus_relaxationParam_type

  !> Contains STfun of viscosity variable and relaxation parameter for each
  !! level
  type mus_viscosity_type
    !> space-time function
    type(tem_spacetime_fun_type) :: STfun
    !> viscosity value evaluated from STfun
    type(grw_realArray_type), allocatable :: dataOnLvl(:)
    !> relaxation paramter omega for each level
    type(mus_relaxationParam_type), allocatable :: omLvl(:)
  end type mus_viscosity_type

contains


! ****************************************************************************** !
  !> This routine initialize relaxation parameter
  subroutine mus_init_relaxParam( omLvl, minLevel, maxLevel, nSolve )
    ! --------------------------------------------------------------------------
    !> relaxation paramter
    type(mus_relaxationParam_type), allocatable, intent(out) :: omLvl(:)
    !> minlevel and maxLevel
    integer, intent(in) :: minLevel, maxLevel
    !> number of elements to solve per level
    integer, intent(in) :: nSolve(minLevel:maxLevel)
    ! --------------------------------------------------------------------------
    integer :: iLevel
    ! --------------------------------------------------------------------------
    allocate(omLvl(minLevel:maxLevel))
    do iLevel = minLevel, maxLevel
      allocate( omLvl(iLevel)%val( nSolve(iLevel) ) )
      omLvl(iLevel)%val = -100.0_rk
    end do  

  end subroutine mus_init_relaxParam
! ****************************************************************************** !

! ****************************************************************************** !
  !> Update omega by temporal factor
  !! This routine is called every iteration to update relaxation parameters
  !! including omega and mrt.\n
  !! In Jonas Toelke paper (2006) about MRT, the following notions are used:\n
  !!  s(a) = s(2)
  !!  s(b) = s(3)
  !!  s(c) = s(5) = s(7) = s(9)
  !!  s(d) = s(11) = s(13
  !!  s(e) = s(17) = s(18) = s(19)
  !!  s(w) = s(10) = s(12) = s(14) = s(15) = s(16)
  !! It is suggested that, for D3Q19,
  !!  s(a) = s(b) = s(c) = s(d) = s(e) = max( s(w), -1.0 )
  !! Notice that the collision matrix S used in this papar corresponds to
  !! -omega in BGK model, because it express the LB equation is slightly
  !! different way.
  !!
  subroutine mus_update_relaxation( omegaKine, omegaBulk, viscKine, viscBulk, &
    &                               visc_ramping, mrt, moment, schemeHeader, &
    &                               tNow )
    ! --------------------------------------------------------------------------
    !> viscosity omega on current level
    real(kind=rk), intent(out) :: omegaKine
    !> bulk viscosity omega on current level
    real(kind=rk), intent(out) :: omegaBulk
    !> kinematic viscosity
    real(kind=rk), intent(in) :: viscKine
    !> bulk viscosity
    real(kind=rk), intent(in) :: viscBulk
    !> viscosity ramping
    type(tem_temporal_type), intent(in) :: visc_ramping
    !> MRT relaxation parameter 
    type( mus_mrt_type ), intent(inout) :: mrt
    type(mus_moment_type), intent(in) :: moment
    type(mus_scheme_header_type), intent(in) :: schemeHeader
    type( tem_time_type ),   intent(in) :: tNow
    ! --------------------------------------------------------------------------
    !> Relaxation parameter
    real(kind=rk) :: fac
    ! --------------------------------------------------------------------------
    ! calculate ramping factor 
    fac = tem_temporal_for( temporal = visc_ramping, &
      &                     time     = tNow )

    ! apply temporal scaling to viscosity
    ! visc = cs2 * ( 1/w - 1/2 )
    omegaKine = 1.0 / ( cs2inv * fac * viscKine + 0.5_rk )

    omegaBulk = 2.0_rk / (9.0_rk * fac * viscBulk + 1.0_rk )

    write(logUnit(10),"(A,F10.5)") "Omega factor: ", fac
    write(logUnit(10),"(A,F10.5)") "Omega ramped: ", omegaKine
    call mus_set_mrtRelaxation(mrt          = mrt,           &
      &                        omegaKine    = omegaKine,     &
      &                        omegaBulk    = omegaBulk,     &
      &                        schemeHeader = schemeHeader,  &
      &                        moment       = moment         )

  end subroutine mus_update_relaxation
! ****************************************************************************** !

!! ****************************************************************************** !
!  !> Apply viscosity ramping to omega state
!  subroutine mus_update_stateOmega(state, omegaPos, varSys, nElems, nSize, &
!    &                              visc_ramping, tNow)
!    ! --------------------------------------------------------------------------
!    !> current state, update only omega
!    real(kind=rk), intent(inout) :: state(:)
!    !> Position of omega variable in varSys
!    integer, intent(in) :: omegaPos
!    !> variable system
!    type(tem_varSys_type), intent(in) :: varSys
!    integer, intent(in) :: nElems !< nElems in local partition
!    integer, intent(in) :: nSize !< size of state array
!    !> viscosity ramping
!    type(tem_temporal_type), intent(in) :: visc_ramping
!    !> Current simultion time
!    type( tem_time_type ),   intent(in) :: tNow
!    ! --------------------------------------------------------------------------
!    integer :: iElem, omegaPosInState
!    real(kind=rk) :: fac
!    ! --------------------------------------------------------------------------
!    ! calculate ramping factor 
!    fac = tem_temporal_for( temporal = visc_ramping, &
!      &                     time     = tNow )
!
!    omegaPosInState = varSys%method%val(omegaPos)%state_varPos(1)
!    ! update omega in state array according to ramping
!    do iElem = 1, nElems 
!      ! local omega might be different from reference omega due to turbulence
!      ! or non-newtonian fluid
!      omega = state(?IDX?(omegaPosInState, iElem, nScalars, nSize))
!
!      ! apply temporal scaling to viscosity
!      state(?IDX?(omegaPos, iElem, nScalars, nSize)) = &
!        & 1.0_rk / ( fac * (1.0_rk/omega - 0.5_rk) + 0.5_rk )
!    end do
!
!  end subroutine mus_update_stateOmega
!! ****************************************************************************** !

! ****************************************************************************** !
  !> Update kinematic viscosity from STfun and calculate turbulent viscosity
  !! from velocity gradient or nonEqPDF
  !! Viscosity obtained from this routine are normalized to the level
  subroutine mus_update_viscKine(viscKine, state, neigh, auxField, layout,   &
    & baryOfTotal, tNow, nSize, nFluids, nGhostFromCoarser, nGhostFromFiner, &
    & varSys, iLevel, viscRef, dxL, dtL, derVarPos, turb)
    ! --------------------------------------------------------------------------
    !> Kinematic viscosity
    type(mus_viscosity_type), intent(inout) :: viscKine
    !> bary of treeID in total list
    real(kind=rk), intent(in) :: baryOfTotal(:,:)
    !> number of elements in state array
    integer, intent(in) :: nSize
    !> number of fluid elements in state array
    integer, intent(in) :: nFluids
    !> Number of ghostFromCoarser element in state array
    integer, intent(in) :: nGhostFromCoarser
    !> Number of ghostFromFiner element in state array
    integer, intent(in) :: nGhostFromFiner
    !> state array
    real(kind=rk), intent(in) :: state(:)
    !> neighbor connectivity array
    integer, intent(in) :: neigh(:)
    !> Auxiliary field variable array
    real(kind=rk), intent(in) :: auxField(:)
    !> stencil layout
    type(mus_scheme_layout_type), intent(in) :: layout
    !> current level
    integer, intent(in) :: iLevel
    !> current simulation time
    type(tem_time_type), intent(in) :: tNow
    !> reference viscosity on current level i.e vL_c/dtL 
    real(kind=rk), intent(in) :: viscRef
    !> lattice element size in current level
    real(kind=rk), intent(in) :: dxL
    !> lattice time step size in current level
    real(kind=rk), intent(in) :: dtL
    !> variable system
    type(tem_varSys_type), intent(in) :: varSys
    !> contains position of in varSys
    type(mus_derVarPos_type), intent(in) :: derVarPos
    !> turbulence type
    type(mus_turbulence_type), intent(inout) :: turb
    ! --------------------------------------------------------------------------
    integer :: nScalars, nAuxScalars, nSolve, velPos(3)
    ! --------------------------------------------------------------------------
    nSolve = nFluids + nGhostFromCoarser + nGhostFromFiner
    nScalars = varSys%nScalars
    nAuxScalars = varSys%nAuxScalars 
    velPos = varSys%method%val(derVarPos%velocity)%auxField_varPos(:)

    ! background fluid kinematic viscosity for nFluid+nGhost
    viscKine%dataOnLvl(iLevel)%val(:) = tem_spacetime_for(                     &
      &                                     me      = viscKine%STfun,          &
      &                                     coord   = baryOfTotal(1:nSolve,:), &
      &                                     time    = tNow,                    &
      &                                     n       = nSolve                   )

    ! convert viscosity to lattice unit
    viscKine%dataOnLvl(iLevel)%val(:) = viscKine%dataOnLvl(iLevel)%val(:) &
      &                               / viscRef

    if (turb%active) then
      !> calculate turbulence viscosity for nFluids and ghostFromCoarser, 
      !! for ghostFromFiner elements, turb visc is interpolated
      nSolve = nFluids + nGhostFromCoarser
      call mus_turb_calcVisc(turbData     = turb%dataOnLvl(iLevel),    &
        &                    turbConfig   = turb%config,               &
        &                    turbViscFunc = turb%viscFunc,             &
        &                    state        = state,                     &
        &                    neigh        = neigh,                     &
        &                    auxField     = auxField,                  &
        &                    velPos       = velPos,                    &
        &                    nSize        = nSize,                     &
        &                    nSolve       = nSolve,                    &
        &                    nScalars     = nScalars,                  &
        &                    nAuxScalars  = nAuxScalars,               &
        &                    layout       = layout,                    &
        &                    viscKine     = viscKine%dataOnLvl(iLevel) &
        &                                           %val(:),           &
        &                    dxL          = dxL,                       & 
        &                    dtL          = dtL                        )
    end if

  end subroutine mus_update_viscKine
! ****************************************************************************** !

! ****************************************************************************** !
  !> Update kinematic relaxation parameter from viscosity
  pure subroutine mus_update_relaxParamKine(viscKine, turb, nSolve, iLevel)
    ! --------------------------------------------------------------------------
    !> Kinematic viscosity
    type(mus_viscosity_type), intent(inout) :: viscKine
    !> Number of elements to solve in compute kernel
    integer, intent(in) :: nSolve
    !> turbulence type
    type(mus_turbulence_type), intent(in) :: turb
    !> current level 
    integer, intent(in) :: iLevel
    ! --------------------------------------------------------------------------
    integer :: iElem
    real(kind=rk) :: tot_visc
    ! --------------------------------------------------------------------------

    if (turb%active) then
      do iElem = 1, nSolve
        ! total normalized viscosity
        tot_visc = viscKine%dataOnLvl(iLevel)%val(iElem) &
          &      + turb%dataOnLvl(iLevel)%visc(iElem)
        ! compute omega  
        viscKine%omLvl(iLevel)%val(iElem) = mus_calcOmegaFromVisc(tot_visc)
      end do 
    else
      ! compute omega from kinematic viscosity
      viscKine%omLvl(iLevel)%val(1:nSolve) &
        & = mus_calcOmegaFromVisc( viscKine%dataOnLvl(iLevel)%val(1:nSolve) )
    end if

  end subroutine mus_update_relaxParamKine
! ****************************************************************************** !

! ****************************************************************************** !
  !> This routine is used to initialize relaxation paramter and update
  !! bulk viscosity at every time step
  !! Bulk visocisty is defined as space-time function to apply
  !! ramping and spatial sponge in bulk viscosity
  subroutine mus_update_relaxParamFromViscSTfun(omega, visc, viscSTfun, nSolve,&
    &                                           baryOfTotal, tNow, viscRef, dtL)
    !--------------------------------------------------------------------------
    !> relaxation parameter
    real(kind=rk), intent(inout) :: omega(:)
    real(kind=rk), intent(inout) :: visc(:)
    !> viscosity space-time function
    type(tem_spacetime_fun_type), intent(in) :: viscSTfun
    !> Number of elements to solve (no halos)
    integer, intent(in) :: nSolve
    !> baryID of total list
    real(kind=rk), intent(in) :: baryOfTotal(:,:)
    !> current simulation time
    type(tem_time_type), intent(in) :: tNow
    !> reference viscosity on current level i.e vL_c/dtL 
    real(kind=rk), intent(in) :: viscRef
    !> lattice time step size in current level
    real(kind=rk), intent(in) :: dtL
    !--------------------------------------------------------------------------
    ! viscosity values from space-time function
    integer :: iChunk, nChunks, nChunkElems, elemoff
    integer :: minBuf, maxBuf
    !--------------------------------------------------------------------------
    !\todo KM: 'Optimize this routine for constant viscosity'
    ! find chunksize and number of chunks required for initialzation
    nChunks = ceiling( real(nSolve, rk) / real(io_buffer_size, rk) )

    do iChunk = 1, nChunks
      ! Number of elements read so far in previous chunks.
      elemOff = ( (iChunk-1)*io_buffer_size )
      nChunkElems = min(io_buffer_size, nSolve - elemOff)
      minBuf = elemOff+1
      maxBuf = elemOff+nChunkElems

      ! background viscosity
      visc(1:nChunkElems) = tem_spacetime_for(                           &
        &                       me      = viscSTfun,                     &
        &                       coord   = baryOfTotal(minBuf:maxBuf, :), &
        &                       time    = tNow,                          &
        &                       n       = nChunkElems                    )

      ! convert viscosity to lattice unit
      ! viscRef is scaled with current level dtL 
      visc(minBuf:maxBuf) = visc(minBuf:maxBuf) / viscRef
  
      ! compute omega 
      !do iElem = 1, nChunkElems
      !  omega(elemOff+iElem) = mus_calcOmegaFromVisc(visc(iElem))
      !end do  
      omega(minBuf:maxBuf) = mus_calcOmegaFromVisc(visc(minBuf:maxBuf))
    end do  

  end subroutine mus_update_relaxParamFromViscSTfun
! ****************************************************************************** !

! ****************************************************************************** !
  !> This function compute relaxation paramter omega from viscosity
  elemental function mus_calcOmegaFromVisc(visc) result(omega)
    !---------------------------------------------------------------------------
    !> scaled lattice viscosity i.e vL_c/dtL
    real(kind=rk), intent(in) :: visc
    !> lattice time step size in current level
    !!real(kind=rk), intent(in) :: dtL
    !> output: relaxation parameter omega
    real(kind=rk) :: omega
    !---------------------------------------------------------------------------
    !omega = 1.0_rk / ( cs2inv * visc / dtL + 0.5_rk )
    omega = 1.0_rk / ( cs2inv * visc + 0.5_rk )
    !---------------------------------------------------------------------------
  end function mus_calcOmegaFromVisc
! ****************************************************************************** !


end module mus_relaxationParam_module
