! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> This module contains routines which initiliaze advection relaxation and
!! flow field for lbm incompressible model.
module mus_initPoisson_module
  ! include treelm modules
  use env_module,         only: labelLen
  use tem_aux_module,     only: tem_abort
  use tem_logging_module, only: logUnit

  ! include musubi modules
  use mus_compute_Poisson_module,  only: mus_Poisson_advRel_explicit, &
    &                                    mus_Poisson_advRel_d2q9,     &
    &                                    mus_PBLinear_advRel_explicit, &
    &                                    mus_PBnonLinear_advRel_explicit
  use mus_scheme_type_module, only: kernel

  implicit none

  private

  public :: mus_init_advRel_Poisson
  public :: mus_init_advRel_PBLinear
  public :: mus_init_advRel_PBnonLinear

contains

  ! ************************************************************************** !
  !> Initialize the relaxation model for lbm poisson equation
  subroutine mus_init_advRel_Poisson( relaxation, layout, compute )
    ! ---------------------------------------------------------------------------
    character(len=labelLen), intent(inout) :: relaxation
    character(len=labelLen), intent(in) :: layout
    procedure( kernel ), pointer, intent(out) :: compute
    ! ---------------------------------------------------------------------------
    write(logUnit(1),*) 'Choosing relaxation model for Poisson: ' &
      &                 //trim( relaxation )

    select case(trim(relaxation))
    case( 'bgk' )
      select case( trim(layout) )
      case('d2q9')
        compute => mus_Poisson_advRel_d2q9
      case default
        compute => mus_Poisson_advRel_explicit
      end select   
    case default
      write(logUnit(1),*) 'Relaxation '//trim(relaxation)//' is not supported!'
      call tem_abort()
    end select

  end subroutine mus_init_advRel_Poisson
  ! ************************************************************************** !

  ! ************************************************************************** !
  !> Initialize the relaxation model for lbm poisson equation
  subroutine mus_init_advRel_PBLinear( relaxation, layout, compute )
    ! ---------------------------------------------------------------------------
    character(len=labelLen), intent(inout) :: relaxation
    character(len=labelLen), intent(in) :: layout
    procedure( kernel ), pointer, intent(out) :: compute
    ! ---------------------------------------------------------------------------
    write(logUnit(1),*) 'Choosing relaxation model for Poisson-Boltzmann '&
      &               //'linear: '//trim( relaxation )

    select case(trim(relaxation))
    case( 'bgk' )
      compute => mus_PBLinear_advRel_explicit
    case default
      write(logUnit(1),*) 'Relaxation '//trim(relaxation)//' is not supported!'
      call tem_abort()
    end select

  end subroutine mus_init_advRel_PBLinear
  ! ************************************************************************** !

  ! ************************************************************************** !
  !> Initialize the relaxation model for lbm poisson equation
  subroutine mus_init_advRel_PBnonLinear( relaxation, layout, compute )
    ! ---------------------------------------------------------------------------
    character(len=labelLen), intent(inout) :: relaxation
    character(len=labelLen), intent(in) :: layout
    procedure( kernel ), pointer, intent(out) :: compute
    ! ---------------------------------------------------------------------------
    write(logUnit(1),*) 'Choosing relaxation model for Poisson-Boltzmann ' &
      &               //'nonLinear: '//trim( relaxation )

    select case(trim(relaxation))
    case( 'bgk' )
      compute => mus_PBnonLinear_advRel_explicit
    case default
      write(logUnit(1),*) 'Relaxation '//trim(relaxation)//' is not supported!'
      call tem_abort()
    end select

  end subroutine mus_init_advRel_PBnonLinear
  ! ************************************************************************** !
end module mus_initPoisson_module
