! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> author: Philipp Otte
!! Routines and parameter definitions for the isothermal acoustic Eq D3Q19 model
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'header/lbm_d3q19Macros.inc'
module mus_isotherm_acEq_module

  ! include treelm modules
  use env_module,            only: rk
  use tem_param_module,      only: rho0
  use tem_varSys_module,     only: tem_varSys_type, tem_varSys_op_type
  use tem_dyn_array_module,  only: PositionOfVal
  use tem_aux_module,        only: tem_abort

  ! include musubi modules
  use mus_field_prop_module,    only: mus_field_prop_type
  use mus_scheme_layout_module, only: mus_scheme_layout_type
  use mus_scheme_type_module,   only: mus_scheme_type
  use mus_param_module,         only: mus_param_type

  ! Just for debugging
  ! use tem_logging_module, only: logUnit

  implicit none

  private

  public :: bgk_advRel_isotherm_acEq_d3q19

  ! =============================================================================
  ! D3Q19 flow model
  ! =============================================================================
  !> Definition of the discrete velocity set

  ! integer,parameter :: block = 32
  integer,parameter :: QQ   = 19  !< number of pdf directions

  integer,parameter :: qN00 = 1   !< west             x-
  integer,parameter :: q0N0 = 2   !< south            y-
  integer,parameter :: q00N = 3   !< bottom           z-
  integer,parameter :: q100 = 4   !< east             x+
  integer,parameter :: q010 = 5   !< north            y+
  integer,parameter :: q001 = 6   !< top              z+
  integer,parameter :: q0NN = 7   !<                  z-,y-
  integer,parameter :: q0N1 = 8   !<                  z+,y-
  integer,parameter :: q01N = 9   !<                  z-,y+
  integer,parameter :: q011 = 10  !<                  z+,y+
  integer,parameter :: qN0N = 11  !<                  x-,z-
  integer,parameter :: q10N = 12  !<                  x+,z-
  integer,parameter :: qN01 = 13  !<                  x-,z+
  integer,parameter :: q101 = 14  !<                  x+,z+
  integer,parameter :: qNN0 = 15  !<                  y-,x-
  integer,parameter :: qN10 = 16  !<                  y+,x-
  integer,parameter :: q1N0 = 17  !<                  y-,x+
  integer,parameter :: q110 = 18  !<                  y+,x+
  integer,parameter :: q000 = 19  !< rest density is last

  real(kind=rk), parameter :: f1 = 2.0_rk / 5.0_rk
  real(kind=rk), parameter :: f2 = 1.0_rk / 30.0_rk
  real(kind=rk), parameter :: f8 = 1.0_rk / 30.0_rk


contains

! ****************************************************************************** !
  !> Advection relaxation routine for the D3Q19 model with BGK for the
  !> isothermal acoustic equation.
?? copy :: compute_routineHeader( bgk_advRel_isotherm_acEq_d3q19 )
?? copy :: compute_routineParams
?? IF (SOA) THEN
?? ELSE
    ! ---------------------------------------------------------------------------
    integer :: iElem, nScalars
?? copy :: pdfTmp19( f )
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: omega, cmpl_o
    real(kind=rk) :: inv_rho0

    nScalars = varSys%nScalars
    ! read the relaxation parameter omega for the current level
    omega   = fieldProp(1)%fluid%omLvl( level )
    cmpl_o  = 1 - omega
    inv_rho0 = 1.0_rk / rho0

!$omp do schedule(static)

?? copy :: dir_novec
    nodeloop: do iElem = 1, nSolve
      ! First load all local values into temp array
      fN00 = inState(?FETCH?( qN00, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0N0 = inState(?FETCH?( q0N0, 1, iElem, QQ, nScalars, nElems,neigh ))
      f00N = inState(?FETCH?( q00N, 1, iElem, QQ, nScalars, nElems,neigh ))
      f100 = inState(?FETCH?( q100, 1, iElem, QQ, nScalars, nElems,neigh ))
      f010 = inState(?FETCH?( q010, 1, iElem, QQ, nScalars, nElems,neigh ))
      f001 = inState(?FETCH?( q001, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0NN = inState(?FETCH?( q0NN, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0N1 = inState(?FETCH?( q0N1, 1, iElem, QQ, nScalars, nElems,neigh ))
      f01N = inState(?FETCH?( q01N, 1, iElem, QQ, nScalars, nElems,neigh ))
      f011 = inState(?FETCH?( q011, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN0N = inState(?FETCH?( qN0N, 1, iElem, QQ, nScalars, nElems,neigh ))
      f10N = inState(?FETCH?( q10N, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN01 = inState(?FETCH?( qN01, 1, iElem, QQ, nScalars, nElems,neigh ))
      f101 = inState(?FETCH?( q101, 1, iElem, QQ, nScalars, nElems,neigh ))
      fNN0 = inState(?FETCH?( qNN0, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN10 = inState(?FETCH?( qN10, 1, iElem, QQ, nScalars, nElems,neigh ))
      f1N0 = inState(?FETCH?( q1N0, 1, iElem, QQ, nScalars, nElems,neigh ))
      f110 = inState(?FETCH?( q110, 1, iElem, QQ, nScalars, nElems,neigh ))
      f000 = inState(?FETCH?( q000, 1, iElem, QQ, nScalars, nElems,neigh ))

      ! compute density and velocity
      rho = f000 &
          + f100 + f010 + f001 &
          + fN00 + f0N0 + f00N &
          + f110 + f101 + f011 &
          + fN10 + fN01 + f0N1 &
          + f1N0 + f10N + f01N &
          + fNN0 + fN0N + f0NN
      u_x = ( f100 - fN00 &
          + f110 + f101 + f1N0 + f10N &
          - fN10 - fN01 - fNN0 - fN0N ) * inv_rho0
      u_y = ( f010 - f0N0 &
          + f110 + f011 + fN10 + f01N &
          - f1N0 - f0N1 - fNN0 - f0NN ) * inv_rho0
      u_z = ( f001 - f00N &
          + f101 + f011 + fN01 + f0N1 &
          - f10N - f01N - fN0N - f0NN ) * inv_rho0

      ! set output
      outState( ?SAVE?( qN00, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * fN00 + omega * ( rho - 3 * rho0 * u_x ) * f2
      outState( ?SAVE?( q0N0, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f0N0 + omega * ( rho - 3 * rho0 * u_y ) * f2
      outState( ?SAVE?( q00N, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f00N + omega * ( rho - 3 * rho0 * u_z ) * f2
      outState( ?SAVE?( q100, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f100 + omega * ( rho + 3 * rho0 * u_x ) * f2
      outState( ?SAVE?( q010, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f010 + omega * ( rho + 3 * rho0 * u_y ) * f2
      outState( ?SAVE?( q001, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f001 + omega * ( rho + 3 * rho0 * u_z ) * f2
      outState( ?SAVE?( q0NN, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f0NN + omega * ( rho - 3 * rho0 * ( u_y + u_z ) ) * f8
      outState( ?SAVE?( q0N1, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f0N1 + omega * ( rho - 3 * rho0 * ( u_y - u_z ) ) * f8
      outState( ?SAVE?( q01N, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f01N + omega * ( rho + 3 * rho0 * ( u_y - u_z ) ) * f8
      outState( ?SAVE?( q011, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f011 + omega * ( rho + 3 * rho0 * ( u_y + u_z ) ) * f8
      outState( ?SAVE?( qN0N, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * fN0N + omega * ( rho - 3 * rho0 * ( u_x + u_z ) ) * f8
      outState( ?SAVE?( qN01, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * fN01 + omega * ( rho - 3 * rho0 * ( u_x - u_z ) ) * f8
      outState( ?SAVE?( q10N, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f10N + omega * ( rho + 3 * rho0 * ( u_x - u_z ) ) * f8
      outState( ?SAVE?( q101, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f101 + omega * ( rho + 3 * rho0 * ( u_x + u_z ) ) * f8
      outState( ?SAVE?( qNN0, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * fNN0 + omega * ( rho - 3 * rho0 * ( u_x + u_y ) ) * f8
      outState( ?SAVE?( qN10, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * fN10 + omega * ( rho - 3 * rho0 * ( u_x - u_y ) ) * f8
      outState( ?SAVE?( q1N0, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f1N0 + omega * ( rho + 3 * rho0 * ( u_x - u_y ) ) * f8
      outState( ?SAVE?( q110, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f110 + omega * ( rho + 3 * rho0 * ( u_x + u_y ) ) * f8
      outState( ?SAVE?( q000, 1, iElem, QQ, nScalars, nElems, neigh )) = &
        & cmpl_o * f000 + omega * ( rho ) * f1
    enddo nodeloop
!$omp end do nowait
?? END IF

  end subroutine bgk_advRel_isotherm_acEq_d3q19
! ****************************************************************************** !

end module mus_isotherm_acEq_module
! ****************************************************************************** !
