-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of Gaussian pulse in density. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). 
-- To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the 
-- level variable. An increment of one will half the radius of your elements.

timestep_info = 1

logging = {level=10}

-- Check for Nans and unphysical values
check =  {
           interval = 1,
         }

-- global simulation options
simulation_name='ateles_left' 
degree = 1
tmax = 1.0
track_dt = 0.05
sim_control = {
             time_control = {
                  min = 0,
                  max = tmax,
                  interval = {iter = 10}, -- final simulation time
                }
}

-- table for preCICE
precice = {
           accessor = 'Ateles_left',
           configFile ='precice_config.xml',
          }

---- Restart settings
NOrestart = { 
  write = './restart/left/',                                        
  time_control = { 
    min = 0, 
    max = tmax, 
    interval = {iter=1} 
  }
}

-- Tracking
tracking = {
  { -- GLobal VTK
    label = 'global',
    folder = './track_nofilter/left/',
    variable = {'density', 'momentum', 'energy', 'velocity', 'pressure', 
                'grad_density', 'grad_velocity', 'grad_pressure' },
    shape = {
      { kind = 'all',}
    },
    time_control = {min = 0, max = tmax, interval = track_dt},
    output={format = 'vtk'},
  },
}

-- Variable system definintion--
characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  -- This is the global material for Euler 3D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with three
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  },
  {
     name = 'grad_velocity',
     ncomponents = 4,
     vartype = 'operation',
     operation = {
       kind = 'gradient',
       input_varname = 'velocity',
     }          
  },
  {
     name = 'coupling',
     ncomponents = 8,
     vartype = 'st_fun',
     st_fun = { 
       predefined = 'precice',
       precice_mesh = 'leftSurface',
       write_varname = {'density_left', 
                        'velX_left', 
                        'velY_left', 
                        'pressure_left',
                        'gradX_density_left',
                        'gradX_velX_left',
                        'gradX_velY_left',
                        'gradX_pressure_left'},
       read_varname = {'density_right', 
                       'velX_right', 
                       'velY_right', 
                       'pressure_right',
                       'gradX_density_right',
                       'gradX_velX_right',
                       'gradX_velY_right',
                       'gradX_pressure_right'},
     }          
   },
  -- write to precice
  {
     name = 'density_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'density',
       input_varindex = {1}
     }          
  },
  {
     name = 'grad_density',
     ncomponents = 2,
     vartype = 'operation',
     operation = {
       kind = 'gradient',
       input_varname = 'density',
     }          
  },
  {
     name = 'gradX_density_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'grad_density',
       input_varindex = {1}
     }          
  },
  {
     name = 'velX_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'velocity',
       input_varindex = {1}
     }          
  },
  {
     name = 'gradX_velX_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'grad_velocity',
       input_varindex = {1}
     }          
  },
  {
     name = 'velY_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'velocity',
       input_varindex = {2}
     }          
  },
  {
     name = 'gradX_velY_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'grad_velocity',
       input_varindex = {3}
     }          
  },
  {
     name = 'pressure_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'pressure',
       input_varindex = {1}
     }          
  },
  {
     name = 'grad_pressure',
     ncomponents = 2,
     vartype = 'operation',
     operation = {
       kind = 'gradient',
       input_varname = 'pressure',
     }          
  },
  {
     name = 'gradX_pressure_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'grad_pressure',
       input_varindex = {1}
     }          
  },
  -- read from precice
  {
     name = 'density_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {1}
     }          
  },
  {
     name = 'velX_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {2}
     }          
  },
  {
     name = 'velY_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {3}
     }          
  },
  {
     name = 'pressure_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {4}
     }          
  },
  {
     name = 'gradX_density_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {5}
     }          
  },
  {
     name = 'gradX_velX_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {6}
     }          
  },
  {
     name = 'gradX_velY_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {7}
     }          
  },
  {
     name = 'gradX_pressure_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {8}
     }          
  },
}

-- Mesh definitions --
mesh = 'mesh_left/'


-- timing settings (i.e. output for performance measurements, this table is otional)
timing_file = 'timing_left.res'         -- the filename of the timing results

-- Equation definitions --
penalization_eps = 8.0/(degree+1)
penalization_alpha = 1.0
equation = {
  penalization = {
    global = {
      kind = 'const',
      const = {0.0, 0.0, 0.0, 0.0},
    },
  },
  name       = 'navier_stokes_2d',
  -- general fluid's parameter
  isen_coef  = 1.4,
  r          = 280.0,
  -- viscous parameters
  therm_cond = 0.5, 
  mu         = 1.0e-4,  
  ip_param   = 4.0, -- The stabilization parameter for the Interior Penalty discretization
  -- Parameters of the penalization
  porosity             = penalization_eps,
  viscous_permeability = penalization_alpha*penalization_eps,
  thermal_permeability = penalization_alpha*penalization_eps,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_2d',        -- we use the modal discontinuous Galerkin scheme 
    modg_space = 'Q',        -- the polynomial space Q or P
    m = degree,                   -- the maximal polynomial degree for each spatial direction
  },
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitSSPRungeKutta',  --'explicitEuler',
    steps = 2,
    -- how to control the timestep
    control = {
      name = 'cfl',   -- the name of the timestep control mechanism
      cfl  = 0.2,     -- CourantFriedrichsLewy number
      cfl_visc  = 0.2,     -- CourantFriedrichsLewy number
    },
  },
}

-- ...the general projection table
projection = {
  kind = 'l2p',  -- 'fpt' or 'l2p', default 'l2p'
  factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
}

-- This is a very simple example to define constant boundary condtions.
-- Transport velocity of the pulse in x direction.
dens = 1.0
press = 8.0
velocityX = 2.0
function ic_gauss (x,y,z)
d= (x+1)*(x+1)+y*y 
return( dens + 0.1* math.exp(-d/0.09*math.log(2)) )
end 

initial_condition = {
  density = ic_gauss, 
  pressure = press, 
  velocityX = velocityX,
  velocityY = 0.0,
}

 -- Boundary definitions
boundary_condition = {
  {
    label = 'west',
    kind = 'inflow_normal',
    density = dens,
    v_norm = velocityX,
  }
  ,
  {
    label = 'east',
    kind = 'grad_primitives',
    density = 'density_right',
    velocityX = 'velX_right', 
    velocityY = 'velY_right',
    pressure = 'pressure_right',
    grad_density   = 'gradX_density_right',
    grad_velocityX = 'gradX_velX_right', 
    grad_velocityY = 'gradX_velY_right',
    grad_pressure  = 'gradX_pressure_right',
  }
  ,
  {
    label = 'north',
    kind = 'outflow',
    pressure = press,
  }
  ,
  {
    label = 'south',
    kind = 'outflow',
    pressure = press,
  }
  ,
  {
    label = 'top',
    kind = 'outflow',
    pressure = press,
  }
  ,
  {
    label = 'bottom',
    kind = 'outflow',
    pressure = press,
  }

}
