-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of Gaussian pulse in density. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). 
-- To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the 
-- level variable. An increment of one will half the radius of your elements.

timestep_info = 1

logging = {level=3}

-- Check for Nans and unphysical values
check =  {
           interval = 1,
         }

-- global simulation options
simulation_name='ateles_left' 
degree= 1
tmax = 1.0
dt = 0.05
sim_control = {
             time_control = {
                  min = 0,
                  max = tmax,
                  interval = {iter = 10}, -- final simulation time
                }
}

-- table for preCICE
precice = {
           accessor = 'Ateles_left',
           configFile ='precice_config.xml',
          }

---- Restart settings
NOrestart = { 
  write = './restart/left/',                                        
  time_control = { 
    min = 0, 
    max = tmax, 
    interval = {iter=1} 
  }
}

-- Tracking
tracking = {
  { -- GLobal VTK
    label = 'global',
    folder = './track/inner/',
    variable = {'density', 'momentum', 'energy'},
    shape = {
      { kind = 'all',}
    },
    time_control = {min = 0, max = tmax, interval = dt},
    output={format = 'vtk'},
  },
}

-- Variable system definintion--
characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  -- This is the global material for Euler 3D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with three
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  },
  {
     name = 'coupling',
     ncomponents = 4,
     vartype = 'st_fun',
     st_fun = { 
       predefined = 'precice',
       precice_mesh = 'leftSurface',
       write_varname = {'Density_left', 
                        'VelX_left', 
                        'VelY_left', 
                        'Pressure_left'},
       read_varname = {'Density_right', 
                        'VelX_right', 
                        'VelY_right', 
                        'Pressure_right'},
     }          
   },
  -- write to precice
  {
     name = 'Density_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'density',
       input_varindex = {1}
     }          
  },
  {
     name = 'VelX_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'velocity',
       input_varindex = {1}
     }          
  },
  {
     name = 'VelY_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'velocity',
       input_varindex = {2}
     }          
  },
  {
     name = 'Pressure_left',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'pressure',
       input_varindex = {1}
     }          
  },
  -- read from precice
  {
     name = 'Density_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {1}
     }          
  },
  {
     name = 'VelX_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {2}
     }          
  },
  {
     name = 'VelY_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {3}
     }          
  },
  {
     name = 'Pressure_right',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
       kind = 'extract',
       input_varname = 'coupling',
       input_varindex = {4}
     }          
  }
}

-- Mesh definitions --
mesh = 'mesh_inner/'


-- timing settings (i.e. output for performance measurements, this table is otional)
timing_file = 'timing_left.res'         -- the filename of the timing results

-- Equation definitions --
equation = {
  name = 'euler_2d',
  therm_cond = 2.555e-02,
  isen_coef = 1.4,
  r = 296.0,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_2d',        -- we use the modal discontinuous Galerkin scheme 
    modg_space = 'Q',        -- the polynomial space Q or P
    m = degree,                   -- the maximal polynomial degree for each spatial direction
  },
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitSSPRungeKutta',  --'explicitEuler',
    steps = 2,
    -- how to control the timestep
    control = {
      name = 'cfl',   -- the name of the timestep control mechanism
      cfl  = 0.8,     -- CourantFriedrichsLewy number
    },
  },
}

-- ...the general projection table
projection = {
  kind = 'l2p',  -- 'fpt' or 'l2p', default 'l2p'
  factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
}

-- This is a very simple example to define constant boundary condtions.
-- Transport velocity of the pulse in x direction.
dens = 1.0
press = 10.0
velocityX = 0.0
function gauss_dens (x,y,z)
d= (x)*(x)+y*y 
return( dens + 1.0* math.exp(-d/0.02*math.log(2)) )
end 
function gauss_press (x,y,z)
d= (x)*(x)+y*y 
return( press + 5.0* math.exp(-d/0.02*math.log(2)) )
end 

initial_condition = {
  density = dens, 
  pressure = gauss_press, 
  pressure = press, 
  velocityX = velocityX,
  velocityY = 0.0,
}

 -- Boundary definitions
boundary_condition = {
  {
    label = 'west',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }
  ,
  {
    label = 'east',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }
  ,
  {
    label = 'north',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }
  ,
  {
    label = 'south',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }
  ,
  {
    label = 'top',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }
  ,
  {
    label = 'bottom',
    kind = 'primitives',
    density = 'Density_right',
    velocityX = 'VelX_right', 
    velocityY = 'VelY_right',
    pressure = 'Pressure_right',
  }

}
