-- Simple setup to test sponge layer in linearized Euler equations.
-- It uses a 1D gaussian pulse in pressure, with a line of elements
-- and a zero_gradient boundary on the left and sponge with Dirichlet
-- zero boundaries on the right.
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
degree = 15
poly_space = 'Q'
gamma = 1.4
pressure = 1
speed_of_sound = 1
sponge_length = 0.001

logging = { level = 2 }


--...Configuration of simulation time
sim_control = { 
  time_control = {
    max      = 0.006/speed_of_sound,
    min      = 0.0,
    interval = {iter = 10} 
  }
}
--------------------------------------------------------------------------------
-- Definition of the test-case.

-- Mesh definitions --
mesh = {
  predefined = 'line_bounded', -- use the predefined line with boundaries
  origin = { -0.001 },         -- origin of the line
  length = 0.003,              -- length of all elements
  element_count = 3            -- number of elements
}

variable = {
  { name = "spongelayer_var", -- >>>>>>>>>>>>>
    ncomponents = 6,
    vartype = "st_fun",
    st_fun = {
      predefined = 'combined',
      spatial = {
        predefined = 'spongelayer',
        plane_origin = {.001, 0, 0},
        plane_normal = {sponge_length, 0, 0},
        damp_factor = 0.5*(degree+1)*speed_of_sound/sponge_length,
        damp_exponent = 2*degree/(degree+math.log(degree+1,2)),
        target_state = {
          density   = 0.0,
          velocityX = 0.0,
          velocityY = 0.0,
          velocityZ = 0.0,
          pressure =  0.0
        }
      },
      --temporal = {1},
      shape = {
        kind = 'canoND',
        object= {
          origin = { 0.001001, -0.000001, -0.000001 },
          vec = {
            {0.001,0,0},
            {0,0.001002,0},
            {0,0,0.001002}
          },
          segments = {5,5,5}
        }
      }
    }
  } -- <<<<<<<<<<< spongelayer_var
}

-- Tracking              
tracking = {
  {
    label = 'press_east',
    folder = 'track_sphere_',
    variable = {'pressure'},
    shape = { kind = 'canoND',
              object= { origin = {0.0019,0.0,0.0} }
    },
    time_control = {
      max = sim_control.time_control.max,
      min = 0,
      interval = {iter=1},
    },
    output = { format = 'ascii', use_get_point = true }
  },
  {
    label = 'press',
    folder = 'vis_sphere_',
    variable = {'pressure'},
    shape = { kind = 'global'},
    time_control = {
      max = sim_control.time_control.max,
      min = 0,
      interval = 0.0002/speed_of_sound,
    },
    output = { format = 'vtk'}
  }
}
ply_sampling = {
  nlevels = 3,
  adaptiveDofReduction = true,
  tolerance = 0.01
}

-- Equation definitions --
density = gamma*pressure/speed_of_sound^2
equation = {
  name      = 'linearEuler',
  isen_coef = 1.4,
  background = {
    density = density,
    velocityX = 0.0,
    velocityY = 0.0,
    velocityZ = 0.0,
    pressure = pressure
  }
}

source = {
  -- Outlet sponge
  spongelayer = "spongelayer_var"
}


-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',
               m =  degree,
               modg_space = poly_space
               }, 
    -- the temporal discretization scheme
    temporal = {
                 name = 'explicitRungeKutta', 
                 steps = 4,
              -- how to control the timestep
                 control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.95,     -- CourantÐFriedrichsÐLewy number
                         },
               },
}

ampl_pressure = 0.0001*equation.background.pressure

function ic_gauss_pressure(x,y,z)
  r = ((x-0.0005)^2+(y-0.0005)^2+(z-0.0005)^2)/0.00025^2
  return( ampl_pressure * math.exp(r*math.log(0.5)) )
end

-- Initial Condition definitions --
initial_condition = { density   = 0.0,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      pressure  = ic_gauss_pressure
                    }

-- Boundary conditions
boundary_condition = {
  {
    label = 'west',
    kind = 'zero_gradient',
    enforce_zero_grad = true
  },
  {
    label = 'east',
    kind = 'primitives',
    density = 0.0,
    velocityX = 0.0,
    velocityY = 0.0,
    velocityZ = 0.0,
    pressure = 0.0
  }
}

-- ...the general projection table
projection = {
  kind   = 'fpt',
  factor = 1.0
}
