--Configuration for the acoustic pulse.
--
-- The setup simulates an acoustic wave by an initial Gaussian pulse in
-- pressure at the origin.
--
-- Setup taken from Tam: Computational Aeroacoustics, Appendix G.3.
--------------------------------------------------------------------------------

-- Parameters to vary --
degree = 3
poly_space = 'Q'
meshlevel = 6
--------------------------------------------------------------------------------

-- Equation definitions --
--  * Ratio of specific heats
gamma = 1.4
equation = {
             name   = 'linearEuler_2d',
             isen_coef = gamma,
             background = {
             -- Mean field values (see Tam, implementation p. 441)
                 density = 1.0,
                 velocityX = 0.0,
                 velocityY = 0.0,
                 velocityZ = 0.0,
                 pressure = 1.0/gamma 
             }
}

-- Setup (gaussian pulse parameters)
gp_center = {0.0, 0.0, 0.0}
gp_halfwidth = 3.0
gp_amplitude = 0.001
gp_background = 0.0
gp_c = math.sqrt( gamma * equation.background.pressure
                        / equation.background.density )


function exact_2d(x,y,z,t)                        
  R= math.sqrt(x*x+y*y)
  p = gp_background + gp_amplitude /2
      * (  (R-gp_c*t) /(2*R) * math.exp( -math.log(2) * ( (R-gp_c*t)/gp_halfwidth )^2 )
         + (R+gp_c*t) /(2*R) * math.exp( -math.log(2) * ( (R+gp_c*t)/gp_halfwidth )^2 ) )
  return(p)
end


variable = {
  {
    name = 'exact_p',
    ncomponents = 1,
    vartype = 'st_fun',
    st_fun = {
        predefined = 'acoustic_pulse',
        center         = gp_center,
        halfwidth      = gp_halfwidth,
        amplitude      = gp_amplitude,
        background     = gp_background,
        speed_of_sound = gp_c
    },
  },
  {
    name = 'exact_2d_p',
    ncomponents = 1,
    vartype = 'st_fun',
    st_fun = exact_2d
  },
  {
    name = 'error',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'difference',
      input_varname = {'pressure', 'exact_p'},
    },
  },
  {
    name = 'error_2d',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'difference',
      input_varname = {'pressure', 'exact_2d_p'},
    },
  },
}
--------------------------------------------------------------------------------


-- Initial Condition --
initial_condition = { density = 0.0,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      pressure = { predefined = 'gausspulse',
                                   center     = gp_center,
                                   halfwidth  = gp_halfwidth,
                                   amplitude  = gp_amplitude,
                                   background = gp_background
                      }
}

tmax = 5.0
-- Configuration of simulation time
sim_control = { time_control = { max = tmax, -- final Simulated time
                                 min = 0.0, 
                                 interval = {iter = 1} 
                }
}

--------------------------------------------------------------------------------

logging = { level = 2 }

-- Check for Nans and unphysical values
check =  { interval = 1 }

-- ...the general projection table
projection = {
    kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
    -- for fpt the  nodes are automatically 'chebyshev'
    -- for l2p the  nodes are automatically 'gauss-legendre'
    factor = 1.0   -- dealising factor
    -- blocksize = 32,
}

ply_sampling = { nlevels = 2}    -- maximal level to use in subsampling
tracking = {
  {
    label = 'center',
    folder = './track_2d/',
    variable = {'pressure', 'exact_p', 'error', 'exact_2d_p', 'error_2d'},
    shape = {
      kind = 'canoND',
      object = {
        -- close to center
        { origin = { 0.1, 0.0, 0.0 }},
      },
    },
    time_control = {min = 0, max = tmax, interval = {iter=1}},
    output = { format = 'ascii', use_get_point = true},
  },
}

rest_prefix = 'restart_'..poly_space..degree..'_L'..meshlevel..'_'

--...Configuration for the restart file
NOrestart = { 
            -- file to restart from (if existing)
            read = rest_prefix..'simulation_lastHeader.lua',
            write = rest_prefix,
            init_on_missing = true,
            -- temporal definition of restart write
            time_control = { max = sim_control.time_control.max,
                             min = 0.0,
                             interval = {iter = 25}
            }
}
--------------------------------------------------------------------------------
-- Definition of the test-case.

---- Mesh definition --
mesh = {
        predefined = 'slice',
        origin = { -32.0, -32.0, -0.5 },
        length = 64.0,
        refinementLevel = meshlevel
}


-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
        name = 'modg_2d',
        m =  degree,
        modg_space = poly_space
    }, 
    -- the temporal discretization scheme
    temporal = {
        name = 'explicitRungeKutta', 
        steps = 4,
        control = {
            name = 'cfl',
            cfl  = 0.95
        }
    }
}
