-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin
-- Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous
-- media. The simulation domain is a periodic cube with edge length 2.0.
-- Therefore this is a very good way to verify your algorithmic implementations,
-- since this testcase does not involve any boundary conditions.
-- The testcase simulates the temporal development of Gaussian pulse in density.
-- Since we are considering a very simple domain, an analytic solution is well
-- known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel
-- (with multiple mpi ranks).
-- To specify the number of ranks please modify nprocs variable. To calculate a
-- grid convergence behavior please modify the level variable. An increment of
-- one will half the radius of your elements.

-- ...the length of the cube
cubeLength = 2.0*math.pi

-- the refinement level of the octree
level = 2

-- global simulation options
sim_name = 'tgv_nvrstk_modg_2d'
logging = {level=10}
sim_control = {
  time_control = {
    max =  {iter = 10},
  }
}

check = {
  interval = 1,
}

-- Mesh definitions --
mesh = { 
  predefined = 'cube',
  origin = { 0.0, 0.0, 0.0 },
  length = cubeLength,
  refinementLevel = level
}

-- Now, define the source term to create analytical solution
function source_fun_dens(x,y)
  return 8.0*math.cos(x)*math.cos(y)
end
function source_fun_mom_x(x,y)
  return 16.0*math.cos(x)*math.sin(x)*(math.cos(y)^2) + (16.0/3.0)*math.sin(x)*math.cos(y) - 2.5*math.sin(2.0*x)
end
function source_fun_mom_y(x,y)
  return 16.0*math.cos(y)*math.sin(y)*(math.cos(x)^2) + (16.0/3.0)*math.sin(y)*math.cos(x) - 2.5*math.sin(2.0*y)
end
function source_fun_en(x,y)
  return 0.0004464285714*math.cos(2.0*x) + 0.0004464285714*math.cos(2.0*y) + 48.0*math.cos(x)*math.cos(y) + (40.0/3.0)*(math.cos(x)^2) + (40.0/3.0)*(math.cos(y)^2) + 16.0*math.cos(x)*(math.cos(y)^3) + 16.0*(math.cos(x)^3)*math.cos(y) - (64.0/3.0)*(math.cos(x)^2)*(math.cos(y)^2) - 24.0*(math.cos(x)^3)*(math.cos(y)^3) + 1.75*math.cos(2.0*x)*math.cos(x)*math.cos(y) + 1.75*math.cos(2.0*y)*math.cos(x)*math.cos(y) - 1.75*math.sin(2.0*x)*math.cos(y)*math.sin(x) - 1.75*math.sin(2.0*y)*math.cos(x)*math.sin(y) - 8.0
end
function source_fun(x,y,z,t)
  src_dens = source_fun_dens(x,y)
  src_mom_x = source_fun_mom_x(x,y)
  src_mom_y = source_fun_mom_y(x,y)
  src_ener = source_fun_en(x,y)
  return {src_dens, src_mom_x, src_mom_y, src_ener}
end

characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0,0.0}
end
relax_temperature = 0.0

variable = {
  -- This is the global material for Euler 2D. It consists of three different 
  -- components, characteristics with one scalar, relax_velocity with two 
  -- scalars and relax_temperature with one scalar, thus we need four scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a 
  -- neutral term, which in this case is 0.

  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     --st_fun = relax_velocity
     st_fun = {const = {0.0,0.0}}
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  },
  {
    name = "arbitrary_var",
    ncomponents = 4,
    vartype = "st_fun",
    st_fun = source_fun
  }
}

source = {
  arbitrary = "arbitrary_var"
}

-- Tracking
tracking = {
  label = 'track_momentum',
  folder = './',
  variable = { 'momentum' },
  shape = {
    kind = 'canoND', 
    object= { 
      origin ={ 1e-5, 1e-5, 1e-5 }
    }
  },
  time_control = {
    min = { iter = 0 },
    max = { sim_control.time_control.max.iter },
    interval = { iter = 1 },
  },
  output = { format = 'ascii', ndofs = 1 } -- 'asciiSpatial', 'harvester', 'convergence'
}


-- the filename of the timing results i.e. output for performance measurements
timing = { filename = 'timing.res' }

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_2d',
    m = 3
  },
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitRungeKutta',
    steps = 4,
    control = {
      name = 'cfl',
      cfl = 0.1,
      cfl_visc = 0.4,
    }
  }
}

-- Equation definitions --
-- ... the smallness parameter of the penalization
penalization_eps = 8.0/(scheme.spatial.m+1)
penalization_alpha = 1.0
equation = {
  penalization = {
    global = {
      kind = 'const',
      const = {0.0, 0.0, 0.0, 0.0},
    },
  },
  name       = 'navier_stokes_2d',
  -- general fluid's parameter
  isen_coef  = 1.4,
  r          = 280.0,
  -- viscous parameters
  therm_cond = 0.5, 
  mu         = 2.0,  
  ip_param   = 4.0, -- The stabilization parameter for the Interior Penalty discretization
  -- Parameters of the penalization
  porosity             = penalization_eps,
  viscous_permeability = penalization_alpha*penalization_eps,
  thermal_permeability = penalization_alpha*penalization_eps,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)


projection = {
  kind = 'fpt',
  factor = 1.0,
}

function ini_dens(x,y,z)
  return 4.0
end 
function ini_vel_x(x,y,z)
  return math.sin(x)*math.cos(y)
end 
function ini_vel_y(x,y,z)
  return math.cos(x) * math.sin(y)
end 
function ini_press(x,y,z)
  return (1.0/4.0)*(math.cos(2.0*x)+math.cos(2.0*y)) + 8.0
end


-- Now, define the initial conditions
initial_condition = {
  density = ini_dens,
  pressure = ini_press,
  velocityX = ini_vel_x,
  velocityY = ini_vel_y,
  useFpt = true
}
