-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of Gaussian pulse in density. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). 
-- To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the 
-- level variable. An increment of one will half the radius of your elements.

-- ...the length of the cube
cubeLength = 2.0*math.pi

-- the refinement level of the octree
level = os.getenv('level')

-- global simulation options
sim_name = 'tgv_nvrstk_modg_2d'
sim_control = {
                time_control = {   
                                 min =  0, 
                                 max =  1e-3,
                                 interval = {iter = 1},
                               }
             }

check = {
          interval = 1,
        }


-- Mesh definitions --
mesh = { predefined = 'cube',
         origin = { 
                    0.0,
                    0.0,
                    0.0,
                  },
         length = cubeLength,
         refinementLevel = level
       }


---- Restart settings
restart = { 
            write = './restart/',                                        
            time_control = {
                                 min = 0, 
                                 max = sim_control.time_control.max,
                           }, 
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Equation definitions --
equation = {
    name       = 'navier_stokes_2d',
    -- general fluid's parameter
    isen_coef  = 1.4,
    r          = 280.0,
    -- viscous parameters
    therm_cond = 0.5, 
    mu         = 2.0,  
    ip_param   = 4.0, -- The stabilization parameter for the Interior Penalty discretization
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
                 name = 'modg_2d',     
                 m = os.getenv('order')-1,                
               }, 
    -- the temporal discretization scheme
    temporal = {
               --name = 'explicitEuler',
               name = 'explicitRungeKutta',
               steps = 4,
               control = {
                          name      = 'cfl',   
                          cfl       = 0.1, 
                          cfl_visc  = 0.1, 
                         },
               },
}


projection = {
              kind = 'fpt',
              factor = 1.0,
             }

function ini_dens(x,y,z)
  return 4.0
end 
function ini_vel_x(x,y,z)
  return math.sin(x)*math.cos(y)
end 
function ini_vel_y(x,y,z)
  return math.cos(x) * math.sin(y)
end 
function ini_press(x,y,z)
  return (1.0/4.0)*(math.cos(2.0*x)+math.cos(2.0*y)) + 8.0
end

---- Get minimal and maximal density
--max_dens = ini_dens(0.0,0.0,0.0)
--min_dens = ini_dens(math.pi/2.0,math.pi/2.0,0.0)
--print('max_dens: ', max_dens)
--print('min_dens: ', min_dens)
--
---- Get minimal and maximal pressure
--max_press = ini_press(0.0,0.0,0.0)
--min_press = ini_press(math.pi/2.0,math.pi/2.0,0.0)
--print('max_press: ', max_press)
--print('min_press: ', min_press)
--
---- Get minimal and maximal speed of sound
--max_speed_of_sound = math.sqrt(equation.isen_coef*max_press/min_dens)
--min_speed_of_sound = math.sqrt(equation.isen_coef*min_press/max_dens)
--print('max_speed_of_sound: ', max_speed_of_sound)
--print('min_speed_of_sound: ', min_speed_of_sound)
--
---- Get the Mach number of the TGV
--mach_number = 1.0 / min_speed_of_sound
--print('Mach number: ', mach_number)

-- Now, define the initial conditions
initial_condition = { density = ini_dens,
                      pressure = ini_press, 
                      velocityX = ini_vel_x,
                      velocityY = ini_vel_y,
                      useFpt = true, 
                    }

-- Now, define the source term to create analytical solution
function source_fun_dens(x,y)
  return 8.0*math.cos(x)*math.cos(y)
end
function source_fun_mom_x(x,y)
  return 16.0*math.cos(x)*math.sin(x)*(math.cos(y)^2) + (16.0/3.0)*math.sin(x)*math.cos(y) - 2.5*math.sin(2.0*x)
end
function source_fun_mom_y(x,y)
  return 16.0*math.cos(y)*math.sin(y)*(math.cos(x)^2) + (16.0/3.0)*math.sin(y)*math.cos(x) - 2.5*math.sin(2.0*y)
end
function source_fun_en(x,y)
  return 0.0004464285714*math.cos(2.0*x) + 0.0004464285714*math.cos(2.0*y) + 48.0*math.cos(x)*math.cos(y) + (40.0/3.0)*(math.cos(x)^2) + (40.0/3.0)*(math.cos(y)^2) + 16.0*math.cos(x)*(math.cos(y)^3) + 16.0*(math.cos(x)^3)*math.cos(y) - (64.0/3.0)*(math.cos(x)^2)*(math.cos(y)^2) - 24.0*(math.cos(x)^3)*(math.cos(y)^3) + 1.75*math.cos(2.0*x)*math.cos(x)*math.cos(y) + 1.75*math.cos(2.0*y)*math.cos(x)*math.cos(y) - 1.75*math.sin(2.0*x)*math.cos(y)*math.sin(x) - 1.75*math.sin(2.0*y)*math.cos(x)*math.sin(y) - 8.0
end
function source_fun(x,y,z,t)
  src_dens = source_fun_dens(x,y)
  src_mom_x = source_fun_mom_x(x,y)
  src_mom_y = source_fun_mom_y(x,y)
  src_ener = source_fun_en(x,y)
  return {src_dens, src_mom_x, src_mom_y, src_ener}
end
source = {
  arbitrary = {
    fun = source_fun
  }
}
--print('source term at (1,1) : ', source_fun(1,1,0,0))
