-- Shear hat configuration: --
--  A small setup with a flow in y-direction and a hat function variation
--  of the -y velocity in x-direction.
--
--  This is the common configuration for 2D and 3D.
sim_name = 'shear-parabola'
sim_control = {
    time_control = {
        max      = { iter = 400 },
        min      = 0.0,
        interval = { iter = 1 },
    }
}

degree = 5
mach = 0.1
dx = 1.0e-4
cfl = 0.8
cfl_visc = 0.4

-- Material:
isen    = 1.4
boltz   = 287.0
rho     = 1.4
p       = 1.0
conducT = 0.5
mu      = 2.0

-- Interior penalization parameter:
c2 = 8./3.

-- Equation definitions 
equation = {
    name      = 'navier_stokes',
    isen_coef = isen,
    r         = boltz,
    -- Viscous parameters
    therm_cond = conducT,
    mu         = mu,
    ip_param   = c2*3*(degree+2)/(2*(degree+3)),
    material = {
        characteristic = 0.0,
        relax_velocity = {0.0, 0.0, 0.0},
        relax_temperature = 0.0
    }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)


-- Mesh:
mesh = {
    predefined = 'cube',
    origin = { -0.5*dx, -0.5*dx, -0.5*dx },
    length = dx,
    refinementLevel = 0
}

-- Definition of velocity field.
mean_velocity = mach * math.sqrt(isen * p / rho )
max_dv = 0.5*mean_velocity

parabola = function(x)
  -- A parabola defined by the mean velocity, and the maximal
  -- deviation from that mean velocity (max_dv).
  xi = 2*x/dx
  return mean_velocity - max_dv * 0.5 * (3*xi*xi - 1)
end

-- Spatial function to describe the y-velocity profile
velX = function(x,y,z)
  return parabola(x)
end
velY = function(x,y,z)
  return parabola(y)
end
velZ = function(x,y,z)
  return parabola(z)
end

-- Scheme definitions --
scheme = {
    spatial =  {
        name = 'modg',          
        m = degree
    },
    temporal = {
        name = 'explicitRungeKutta',
        steps = 4,
        control = {
            name     = 'cfl', 
            cfl      = cfl,
            cfl_visc = cfl_visc
        }
    }
}

projection = {
    kind = 'fpt',
    factor = 1.0
}


-- Tracking              
ply_sampling = {
    nlevels = 5,
    tolerance = 0.01,
    dof_reduction = 1
}

tracking = {
    label = 'track_shearparabola',
    folder = './',
    variable = {'momentum', 'density', 'energy', 'velocity'},
    shape = {kind = 'all'},
    time_control = {
      min = 0,
      max = sim_control.time_control.max,
      interval = {iter = 1}
    },
    output = { format = 'vtk' }
}
--tracking = {
--    label = 'track_shearparabola',
--    folder = './',
--    variable = {'momentum','density','energy'},
--    shape = {kind = 'canoND', object= { origin ={0., 0., 0.} } },
--    time_control = {
--      min = 0,
--      max = sim_control.time_control.max,
--      interval = {iter = 10}
--    },
--    output = { format = 'vtu', ndofs = 1, use_get_point = true }
--}
