-- Shear hat configuration: --
--  A small setup with a flow in y-direction and a hat function variation
--  of the -y velocity in x-direction.
--
--  This is the common configuration for 2D and 3D.
sim_name = 'shear-hat'
sim_control = {
    time_control = {
        max      = { iter = 400 },
        min      = 0.0,
        interval = { iter = 1 },
    }
}

degree = 3
mach = 0.1
dx = 1.0e-4
cfl = 0.8
cfl_visc = 0.4

-- Material:
isen    = 1.4
boltz   = 287.0
rho     = 1.4
p       = 1.0
conducT = 0.5
mu      = 2.0

-- Interior penalization parameter:
c2 = 8./3.

-- Mesh:
mesh = {
    predefined = 'line',
    origin = { -dx, -0.5*dx, -0.5*dx },
    length = 2*dx,
    element_count = 2
}

-- Spatial function to describe the y-velocity profile
vel  = function (x,y,z)
  if x > 0 then
    return mach * math.sqrt(isen * p / rho ) * (1. - (x-0.5*dx)/dx)
  else
    return mach * math.sqrt(isen * p / rho ) * (1. + (x+0.5*dx)/dx)
  end
end


projection = {
    kind = 'fpt',
    factor = 1.0
}

initial_condition = {
    density   = rho,
    pressure  = p,
    velocityX = 0.0,
    velocityY = vel
}

-- Tracking              
tracking = {
    label = 'track_shearhat',
    folder = './',
    variable = {'momentum','density','energy'},
    shape = {kind = 'canoND', object= { origin ={0.01*dx, 0., 0.} } },
    time_control = {
      min = 0,
      max = sim_control.time_control.max,
      interval = {iter = 10}
    },
    output = { format = 'ascii', ndofs = 1, use_get_point = true }
}
