-- Configuration file for Ateles --
require 'seeder_3d'

timestep_info = 10

-- global simulation options
simulation = {
              name          = 'rayl_tayl_modg',             -- the name of the simualtion
              time          = {
                                useIterations = false,
                                min = 0.0,
                                max = 1.0e+01,                  -- final simulation time
                              },
             }


-- Mesh definitions --
mesh = './mesh/'


---- Debug information
--debug = {
--         debugFiles = true,
--         verbosity = 99,
--        }


-- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/',                                        
            -- temporal definition of restart write
            time = {   
                      useIterations = false,  
                      min = 0, 
                      max = simulation.time.max, 
                      interval = simulation.time.max/1e+04,
                   },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Equation definitions --
equation = {
    name   = 'euler',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m = 31,                   -- the maximal polynomial degree for each spatial direction
               dealiasFactor = 1.0,      -- factor to remove aliases: 1.0 means no additional dealiasing
               blocksize = 32,
               }, 
    -- the stabilzation of the scheme
    stabilization = {
                       -- Spectral viscosity, i.e.:
                       -- u(x) = \sum_{k=0}^N \sigma_k \hat{u}_k L_k(x)
                       -- with
                       -- \sigma_k = exp(- \alpha (k/N)^order )
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 10,
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.6,       -- CourantÐFriedrichsÐLewy number
                         },
               },
}

check = {
          interval = 10,
        }

-- Source terms, here we have only gravitational acceleration
gravAcc = 0.1
source_terms = {
                 name = 'gravitation',
                 ncomponents = 3,
                 gravitation = {
                                 kind = 'const',
                                 const = { 0.0, 0.0, (-1.0)*gravAcc },
                               }
               }

-- The initial condition for the density
width = 0.03/minlevel
dens = 2.0
function iniDens(x,y,z,t)
  return dens
         + 0.5*dens*math.tanh(z/width)/2
end 
-- The initial condition for the pressure
upper = bounding_cube.length/2-elemsize
function hydroPress(x,y,z)
  return 2.5 + gravAcc*dens*(upper-z) 
                + (0.5*gravAcc*dens/2.0)*( 
                                       width*math.log(math.cosh(upper/width)) 
                                        - width*math.log(math.cosh(z/width)) 
                                      ) 
                         

end 
-- The initial condition for the z component of the velocity
domLengthZ = bounding_cube.length - 2*elemsize
domLengthY = domLengthZ/2
domLengthX = domLengthZ/2
function iniVelZ(x,y,z)
  return 0.005 * ( 1 + math.cos(2.0*math.pi*z/domLengthZ) ) * ( 1 + math.cos(2.0*math.pi*(y-domLengthY/2.0)/domLengthY) ) * ( 1 + math.cos(2.0*math.pi*(x-domLengthX/2.0)/domLengthX) )
end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = iniDens, 
                      pressure = hydroPress, 
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = iniVelZ,
                    }


-- ...the general projection table
projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }

-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'slipTop', 
                       kind = 'slipwall',       
                       }
                       ,
                       { 
                       label = 'slipBottom', 
                       kind = 'slipwall',       
                       }
                       ,
                     }  
