-- Configuration file for Ateles --
require('S834_definition')

-- Simulation settings, cord length of the airfoil, angle of attack
-- spatial order of the scheme, and level of the mesh
airfoil_length = 1.0
angle_of_attack = math.rad(12.7)
spatial_order = 8
level = 4

-- global simulation options
sim_name = 'S834'
sim_control = {
                time_control = {
                                 max = 75.0,
                                 interval = { iter = 100 },
                               }
             }
logging = { level = 5 }

--print('Physical simulation time: ', sim_control.time_control.max)

-- Mesh definitions --
mesh = { predefined = 'slice',
         origin = {-1, -1, 0},
         refinementLevel = level,
         length = 4.0
       }

elemsize = mesh.length * 2^(-level)
eps = elemsize * 2^(-level)

-- Restart settings
io_buffer_size = 10
restart = {
            --read  = './restart/simulation_lastHeader.lua',
            write = './restart/',
            time_control = {
                            min = 0,
                            max = sim_control.time_control.max,
                            interval = sim_control.time_control.max/100,
                           },
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'

-- ...the initial condition table.
isen = 1.4
boltz= 296
rho  = 1.0
p    = 1.0
-- ... define background velocity (ATTENTION: change boundaries accordingly)
vel  = 0.1 * math.sqrt(isen * p / rho )
-- ... define the temperature in the penalization area
temp = p / ( rho * boltz )

-- ... the smallness parameter of the penalization
penalization_eps = 8.0/spatial_order
penalization_alpha = 0.5

-- ... the scaling and rotation of the profile
scale = airfoil_length
rotation = {
            { math.cos(angle_of_attack), math.sin(angle_of_attack)},
            {-math.sin(angle_of_attack), math.cos(angle_of_attack)}
           }
profile = {}
for i, v in ipairs(S834_vertex) do
  point_X = scale*(   rotation[1][1] * S834_vertex[i][1]
                    + rotation[1][2] * S834_vertex[i][2] )
  point_Y = scale*(   rotation[2][1] * S834_vertex[i][1]
                    + rotation[2][2] * S834_vertex[i][2] )
  table.insert(profile, {point_X, point_Y})
end

initial_condition = {
                      density  = rho,
                      pressure = p,
                      velocityX = vel,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                    }

-- Check for Nans and unphysical values
check =  {
           interval = 1
         }

-- Scheme definitions --
scheme = {
    spatial =  {
                  name = 'modg_2d',
                  m = spatial_order-1,
               },
    temporal = {
               name = 'explicitRungeKutta',
               steps = 4,
               control = {
                          name     = 'cfl',
                          cfl      = 0.1,
                          cfl_visc = 0.1,
                         },
               },
}


-- ...the general projection table
projection = {
              kind = 'fpt',
              lobattoPoints = true,
              factor = 1.0,
              blocksize = 32
             }

-- Equation definitions --
equation = {
    name   = 'euler_2d',
    -- The definition of an obstacle by penalization
    penalization = {
                     {
                        kind = 'combined',
                        spatial = { predefined = 'polygon_material',
                                    inval      = {1.0, 0.0, 0.0, temp},
                                    vertex     = profile
                                  },
                        temporal = 1.0,
                        shape={
                                  kind = 'canoND',
                                  object = {
                                             origin = {
                                                       (-0.5)*scale,
                                                       (-0.5)*scale,
                                                       -eps,
                                                      },
                                             vec = {
                                                     {2.0*scale,0.0,0.0},
                                                     {0.0,2.0*scale,0.0},
                                                     {0.0,0.0,elemsize+2*eps},
                                                   },
                                             segments = {100,100,10},
                                           },
                            },
                     },
                     global = {
                                kind = 'const',
                                const = {0.0, 0.0, 0.0, 0.0},
                              },
                   },
    -- Physical parameters of the gas phase
    isen_coef            = isen,
    r                    = boltz,
    -- Parameters of the penalization
    porosity             = penalization_eps,
    viscous_permeability = penalization_alpha*penalization_eps,
    thermal_permeability = penalization_alpha*penalization_eps,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

