-- Configuration file for Ateles --
-- This is the setup for the 1d benchmark testcase from the 
-- paper from Liu & Vasilyev
-- This is a pulse with perturbation in density, velocity and
-- pressure. 
-- The domain is of size = 1 and the material occupies half
-- of the domain. i.e in the interval (0.5,1)
-- The solution after t=0.5 should be the same as the initial
-- condition (t=0)

require ('seeder')

time_to_dump = 0.5

wall_x = 0.5
wall_thickness = 0.5

-- ... the smallness parameter of the penalization
penalization_eps = 0.1 --8.0/(scheme.spatial.m+1)
alpha_v = 1e-2
alpha_t = 1e-2

--Simulation
spatial_degree = 3
cfl = 0.1


--  parameters for IC
ini_vel_x = 0.0
ini_vel_y = 0.0
ini_dens  = 1.0
ini_press = 1.0

-- equation parameters
isen  = 1.4
boltz = 1.0
sphere_diameter = 1.0
-- background state
back_press = ini_press/isen                   --equation.background.pressure
back_dens = ini_dens                          --equation.background.density
ini_temp = back_press/back_dens/boltz
c = math.sqrt((isen*back_press)/back_dens)    --speed of sound
Re = 5e5                                      --Reynolds number
Pr = 0.5
viscosity = (back_dens*c*sphere_diameter)/Re  --mu (dynamic viscosity)
cp = boltz*isen/(isen-1) 
th_conductivity = cp*viscosity/Pr



--GaussPulse parameter
--Amplitude
amp_Dens = 0.001
amp_Press =0.001
eps = 0.001
--Center
axis_x = 0.25
--Width Control
width_con = 0.004	
-- global simulation options



-- global simulation options
sim_name = 'gp_nvrstk'
logging = {level=3}

sim_control = {
  time_control = {
   -- max =  {sim = 0.6},
    max =  {sim = time_to_dump, clock = 2*3600 - 150},
    interval = {iter=5000},       
  }
}

check = {
  interval = 1,
}

-- Mesh definitions --
mesh = './mesh/'

---- Restart settings
restart = {

  --read  = './restart/simulation_lastHeader.lua',
  write = './restart/',

  -- temporal definition of restart write
  time_control = {
    min = 0.0,
    max = time_to_dump,

    interval = sim_control.time_control.max.sim/5
  }
}



function p_prime(x,y,z)
d = ((x-axis_x)^2) --+ ((y-axis_y)^2) 
arg = (-d/width_con)* math.log(2)
res =  eps*math.exp(arg)
return res
end

function pulse_press (x,y,z)
res = back_press + p_prime(x,y,z)
return res
end

function pulse_dens (x,y,z)
res = back_dens + p_prime(x,y,z)
return res
end

function pulse_velx (x,y,z)
res =  p_prime(x,y,z)
return res
end



-- Now, define the initial conditions
initial_condition = { 
                      density = pulse_dens,
                      pressure = pulse_press,
                      velocityX = pulse_velx,
                      velocityY = 0.0,
                    }

function inside_wall(x_phys,y_phys,z_phys)
  if x_phys > wall_x then 
    return 1.0
  else
    return 0.0
  end
end


function characteristic (x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return 1.0
  else
    return 0.0
  end
end

function relax_velocity(x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return {0.0,0.0}
  else
    return {ini_vel_x, ini_vel_y}
  end
end
function relax_temperature (x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return ini_temp
  else
    return 0.0
  end
end

tiny_eps = dx/8.0

variable = {
{
   name = 'characteristic',
   ncomponents = 1,
   vartype = "st_fun",
   st_fun = 
   {
     {
       const = { 0.0 }
     },
     {
       fun = characteristic,
       shape = { 
                 kind = 'canoND', 
                 object = {
                              origin = 
                              { 
                                wall_x +  tiny_eps, 
                                -eps, 
                                -eps,
                              },
                              vec = 
                              {
                                { wall_thickness - 2*tiny_eps,0.0,0.0},
                                {0.0,dx,0.0},
                                {0.0,0.0,dx},
                              },
                              segments = { 100, 100, 100 }
                          }
               }
     }
   }
},

{
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     st_fun = 
     {
       {
         const = { 0.0 , 0.0}
       },
       {
         fun = relax_velocity,
         shape = { 
                   kind = 'canoND', 
                 object = {
                              origin = 
                              { 
                                wall_x +  tiny_eps, 
                                -eps, 
                                -eps,
                              },
                              vec = 
                              {
                                { wall_thickness - 2*tiny_eps,0.0,0.0},
                                {0.0,dx,0.0},
                                {0.0,0.0,dx},
                              },
                              segments = { 100, 100, 100 }
                          }
                 }
       }
     }
},

{
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = 
     {
       {
         const = { 0.0 }
       },
       {
         fun = relax_temperature,
         shape = { 
                   kind = 'canoND', 
                 object = {
                              origin = 
                              { 
                                wall_x +  tiny_eps, 
                                -eps, 
                                -eps,
                              },
                              vec = 
                              {
                                { wall_thickness - 2*tiny_eps,0.0,0.0},
                                {0.0,dx,0.0},
                                {0.0,0.0,dx},
                              },
                              segments = { 100, 100, 100 }
                          }
                 }
       }
     }
},
  -- the sponge layer
}

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
                name = 'modg_2d',
                m = spatial_degree
             },
  -- the temporal discretization scheme
  temporal = {
                 --name = 'explicitRungeKutta',
                 name = 'imexRungeKutta',
                 steps = 4,
                 control = {
                              name = 'cfl',
                              cfl = cfl,
                              cfl_visc = cfl,
                           }
             },
}

-- Equation definitions --
equation = {
  name       = 'navier_stokes_2d',
  --name       = 'euler_2d',
  -- general fluid's parameter
  isen_coef  = isen,
  r          = boltz,
  -- viscous parameters
  therm_cond = th_conductivity, 
  mu         = viscosity,  
  ip_param   = 4.0, -- The stabilization parameter -Interior Penalty discretization
  -- Parameters of the penalization
  porosity             = penalization_eps,
  viscous_permeability = alpha_v*penalization_eps,
  thermal_permeability = alpha_t*penalization_eps,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature= 'relax_temperature'
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)


-- Tracking
--tracking = {
--	    {
--             label = 'track',
--             folder = 'tracking/',
--             variable = {'pressure', 'density'},
--             shape = {kind = 'canoND', object= { origin ={trackx, H/2, dx/2}}},
--             time_control = {
--                      min = 0,
--                      max = sim_control.time_control.max.sim,
--                      interval = {iter = 5}
--                    },
--            -- format = 'harvester',
--             output = { format = 'ascii', use_get_point = true },
--           },
--	}
--


projection = {
  kind = 'fpt',
  --factor = 2.0,
}



-- Boundary definitions
boundary_condition = {
                       {
                       label = 'east',
                       kind = 'isothermal_wall',
                       --kind = 'wall',
		                   temperature = ini_temp,
                       v_tan = 0.0,
                       }
                       ,
                       {
                       label = 'west',
                       kind = 'outflow',
                       pressure = back_press,
                       }
                     }



