-- Configuration file for Ateles --
--
-- This the very basic setup of a constant state with a 1D Euler simulation
-- for debugging the material.
-- The domain is of size = 1 and the material occupies half
-- of the domain. i.e in the interval (0.5,1)
-- The solution should just remain constant, no matter the material properties.

--Simulation
spatial_degree = 1
cfl = 0.1
nElems = 4

time_to_dump = 0.5

wall_x = 0.5
wall_thickness = 0.5

-- ... the smallness parameter of the penalization
penalization_eps = 0.1 --8.0/(scheme.spatial.m+1)
alpha_v = 1e-2
alpha_t = 1e-2



--  parameters for IC
ini_vel_x = 0.0
ini_vel_y = 0.0
ini_dens  = 1.0
ini_press = 1.0

-- equation parameters
isen  = 1.4
boltz = 1.0
sphere_diameter = 1.0
-- background state
back_press = ini_press/isen                   --equation.background.pressure
back_dens = ini_dens                          --equation.background.density
ini_temp = back_press/back_dens/boltz
c = math.sqrt((isen*back_press)/back_dens)    --speed of sound
Re = 5e5                                      --Reynolds number
Pr = 0.5
viscosity = (back_dens*c*sphere_diameter)/Re  --mu (dynamic viscosity)
cp = boltz*isen/(isen-1) 
th_conductivity = cp*viscosity/Pr



--GaussPulse parameter
--Amplitude
amp_Dens = 0.001
amp_Press =0.001
eps = 0.001
--Center
axis_x = 0.25
--Width Control
width_con = 0.004	
-- global simulation options



-- global simulation options
sim_name = 'gp_nvrstk'
logging = {level=10}

sim_control = {
  time_control = {
    max =  { sim = time_to_dump, iter=20 },
    interval = {iter=1},       
  }
}

check = {
  interval = 1,
}

-- Mesh definitions --
mesh = {
  predefined = 'line_bounded',
  origin = {0.0, 0.0, 0.0},
  length = 1,
  element_count = nElems
}

---- Restart settings
--restart = {
--
--  --read  = './restart/simulation_lastHeader.lua',
--  write = 'restart/',
--
--  -- temporal definition of restart write
--  time_control = {
--    min = 0.0,
--    max = time_to_dump,
--
--    interval = {iter = 1}
--  }
--}



function p_prime(x,y,z)
d = ((x-axis_x)^2) --+ ((y-axis_y)^2) 
arg = (-d/width_con)* math.log(2)
res =  eps*math.exp(arg)
return res
end

function pulse_press (x,y,z)
res = back_press + p_prime(x,y,z)
return res
end

function pulse_dens (x,y,z)
res = back_dens + p_prime(x,y,z)
return res
end

function pulse_velx (x,y,z)
res =  p_prime(x,y,z)
return res
end



-- Now, define the initial conditions
initial_condition = { 
  density  = back_dens, --pulse_dens,
  pressure = back_press, --pulse_press,
  velocity = 0.0, --pulse_velx
}

function inside_wall(x_phys,y_phys,z_phys)
  if x_phys > wall_x then 
    return 1.0
  else
    return 0.0
  end
end


function characteristic (x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return 1.0
  else
    return 0.0
  end
end

function relax_velocity(x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return {0.0,0.0}
  else
    return {ini_vel_x, ini_vel_y}
  end
end
function relax_temperature (x,y,z,t)
  if inside_wall(x,y,z) > 0.0 then
    return ini_temp
  else
    return 0.0
  end
end

dx = 1.0/nElems
tiny_eps = dx/128.0

variable = {
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = 
     {
       {
         const = { 0.0 }
       },
       {
         fun = 1.0, --characteristic,
         shape = { 
                   kind = 'canoND', 
                   object = {
                                origin = 
                                { 
                                  wall_x + tiny_eps, 
                                  -eps, 
                                  -eps,
                                },
                                vec = 
                                {
                                  { wall_thickness - 2*tiny_eps,0.0,0.0},
                                  {0.0,dx,0.0},
                                  {0.0,0.0,dx},
                                },
                                segments = { 100, 100, 100 }
                            }
                 }
       }
     }
  },
  
  {
       name = 'relax_velocity',
       ncomponents = 1,
       vartype = "st_fun",
       st_fun = 
       {
         {
           const = { 0.0 }
         }--,
         --{
         --  fun = relax_velocity,
         --  shape = { 
         --            kind = 'canoND', 
         --          object = {
         --                       origin = 
         --                       { 
         --                         wall_x +  tiny_eps, 
         --                         -eps, 
         --                         -eps,
         --                       },
         --                       vec = 
         --                       {
         --                         { wall_thickness - 2*tiny_eps,0.0,0.0},
         --                         {0.0,dx,0.0},
         --                         {0.0,0.0,dx},
         --                       },
         --                       segments = { 100, 100, 100 }
         --                   }
         --          }
         --}
       }
  },
  
  {
       name = 'relax_temperature',
       ncomponents = 1,
       vartype = "st_fun",
       st_fun = 
       {
         {
           const = { 0.0 }
         },
         {
           fun = ini_temp, --relax_temperature,
           shape = { 
                     kind = 'canoND', 
                   object = {
                                origin = 
                                { 
                                  wall_x +  tiny_eps, 
                                  -eps, 
                                  -eps,
                                },
                                vec = 
                                {
                                  { wall_thickness - 2*tiny_eps,0.0,0.0},
                                  {0.0,dx,0.0},
                                  {0.0,0.0,dx},
                                },
                                segments = { 100, 100, 100 }
                            }
                   }
         }
       }
  }
}

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_1d',
    m    = spatial_degree
  },
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitRungeKutta',
    --name = 'imexRungeKutta',
    steps = 4,
    control = {
                 name     = 'cfl',
                 cfl      = cfl,
                 cfl_visc = cfl
              }
  }
}

-- Equation definitions --
equation = {
  name       = 'euler_1d',
  -- general fluid's parameter
  isen_coef  = isen,
  r          = boltz,
  -- viscous parameters
  therm_cond = th_conductivity, 
  mu         = viscosity,  
  ip_param   = 4.0, -- The stabilization parameter -Interior Penalty discretization
  -- Parameters of the penalization
  porosity             = penalization_eps,
  viscous_permeability = alpha_v*penalization_eps,
  thermal_permeability = alpha_t*penalization_eps,
  material = {
    characteristic    = 'characteristic',
    relax_velocity    = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

ply_sampling = { nlevels = 3 }

-- Tracking
tracking = {
  {
     label = 'track',
     folder = 'tracking/',
     variable = {'density', 'momentum', 'energy'},
     shape = {kind = 'global'},
     --shape = { kind = 'canoND',
     --  object = {
     --    origin   = {0.0, 0.5*dx, 0.5*dx},
     --    vec      = { {1.0, 0.0, 0.0} },
     --    segments = {100}
     --  }
     --},
     time_control = {
              min = 0,
              max = sim_control.time_control.max,
              interval = {iter = 1}
     },
     output = { format = 'vtk' }
  }
}

projection = {
  kind = 'fpt',
  --factor = 2.0,
}

-- Boundary definitions
boundary_condition = {
  {
    label = 'east',
    kind = 'wall'
  },
  {
    label = 'west',
    kind = 'outflow',
    pressure = back_press,
  }
}
