-- Configuration for the simulation of 1D shock tubes with Riemann problems.
--
-- Define the Riemann problem itself in riemann.lua, this includes left and
-- right state and the fintime up to which the simulation should be done.
require('riemann')

order      = tonumber(os.getenv('ATL_ORDER'))   -- spatial scheme order
nelems     = tonumber(os.getenv('ATL_ELEMS'))   -- number of elements in mesh
dealiasing = tonumber(os.getenv('ATL_DEALIAS')) -- dealiasing factor

-- Filtering:
use_covolume = (os.getenv('ATL_COVOLUME') ~= nil)

-- Exponent of the spectral filter,
-- set it to 0 to deactivate spectral filtering.
filter_order = tonumber(os.getenv('ATL_FILTER'))



-- global simulation options
if use_covolume then
  coname = 'co'
else
  coname = 'nc'
end
simulation_name = 'tube_Q'..order..'-E'..nelems..'-F'..filter_order..coname..'-D'..dealiasing
sim_control = {
                time_control = {
                  min = 0,
                  max = fintime
                }
              }


-- Mesh definitions --
mesh = {
        predefined = 'line_bounded', -- use the predefined line with boundaries
        origin = { -0.5 },           -- origin of the line
        length = 1.0,                -- length of all elements
        element_count = nelems       -- number of elements
       }

-- Equation definitions --
equation = {
    name   = 'euler_1d',
    isen_coef = isen_coef,
    r      = 1.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

---- Restart settings
restart = {
            write = 'restart_',
            time_control = {
                             min = 0,
                             max = fintime,
                             interval = fintime
                          },
          }

-- The initial conditions for the Riemann problem
-- ... left state
rho_l = left[1]
u_l = left[2]
p_l = left[3]
-- ... right state
rho_r = right[1]
u_r = right[2]
p_r = right[3]

function rho(x,y,z)
    if ( x < 0.0 ) then
        return rho_l
    else
        return rho_r
    end
end

function p(x,y,z)
    if ( x < 0.0 ) then
        return p_l
    else
        return p_r
    end
end

function u(x,y,z)
    if ( x < 0.0 ) then
        return u_l
    else
        return u_r
    end
end

projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              factor = dealiasing   -- dealising factor
             }

initial_condition = {
                      density = rho,
                      pressure = p,
                      velocityX = u,
             }

-- Construct the stabilization table:
stab_tab = {}
if use_covolume then
  table.insert(stab_tab, {name = 'covolume', beta = 1.0})
end

if filter_order > 0 then
  table.insert(stab_tab, {name = 'spectral_viscosity'})
  -- Add the spectral filter to all defined stabilizations:
  for f in pairs(stab_tab) do
    stab_tab[f].alpha = 36
    stab_tab[f].order = filter_order
  end
end

-- Scheme definitions --
scheme = {
           -- the spatial discretization scheme
           spatial =  {
                      name = 'modg_1d',
                      m =  order-1,
                      },
           ---- the stabilzation of the scheme
           stabilization = stab_tab,

           -- the temporal discretization scheme
           temporal = {
                      name = 'explicitRungeKutta',
                      steps = 4,
                      control = {
                                 name = 'cfl',
                                 cfl  = 0.5,
                                },
                      },
}

-- Boundary conditions
boundary_condition = {
        {
         label = 'west',
         kind = 'inflow_normal',
         density = rho_l,
         v_norm = u_l,
	}
         ,
        {
         label = 'east',
         kind = 'outflow',
         pressure = p_r,
         }
}
