-- Configuration file for Ateles --
require 'seeder_3d'


-- global simulation options
simulation_name = 'jet_euler_modg'
sim_control = {
                time_control  = {
                                  min = 0.0,
                                  max = 40,
                                  interval = { iter= 1000 },
                                },
             }

check = {
          interval = 300,
        }

-- Mesh definitions --
mesh = './mesh_3d/'


-- Restart settings
restart = { 
            --read = './restart_3d/jet_euler_modg_lastHeader.lua',
            write = './restart_3d/',                                        
            time_control = {   
                             useIterations = false,  
                             min = 0, 
                             max = sim_control.time_control.max, 
                             interval = sim_control.time_control.max/40.0,
                           },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
                  name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
                  m = 15,                   -- the maximal polynomial degree for each spatial direction
               }, 
    stabilization = {
                      {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 10
                      },
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitSSPRungeKutta',  --'explicitEuler',
               steps = 2,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.6,
                         },
               },
}

-- The ambient medium
press = 1
dens = 1

-- the Mach number of the jet
mach = 0.8

isen_coef = 1.4
therm_cond = 2.555e-02
r = 296.0
-- the velocity amplitude in z direction
velAmpl = mach*math.sqrt(isen_coef*press/dens) -- 0.1
densAmpl = 2.0


-- the radius of the jet
momentum_thickness = jet_radius/20

function weight(r)
  return (1+ math.tanh((4*jet_radius-r)/(2*momentum_thickness)))*0.5
end 

function velX_inlet(x,y,z,t)
  r = math.sqrt( (y-jet_center_y)*(y-jet_center_y) + (z-jet_center_z)*(z-jet_center_z)  )
  return velAmpl * (1/2) * ( 1 +  math.tanh((jet_radius-r)/(2*momentum_thickness) ) )
end 

function dens_inlet(x,y,z,t)
  tmpVel = velX_inlet(x,y,z,t)/velAmpl
  r = math.sqrt( (y-jet_center_y)*(y-jet_center_y) + (z-jet_center_z)*(z-jet_center_z)  )
  w = weight(r)
  return w*(densAmpl / ( 1 + (isen_coef-1)/2*mach*mach*tmpVel*(1-tmpVel)  )) + (1-w)*dens
end


-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = dens,
                      pressure = press,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                    }

-- ...the general projection table
projection = {
              kind = 'fpt',  
              factor = 1.0,          
              lobattoPoints=true,
             }

damp_factor = 0.25*scheme.spatial.m
characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 3,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  },
  {
    name = 'sponge',
    ncomponents = 6,
    vartype = "st_fun",
    st_fun = {
      {
        -- Sponge (east outflow)
        predefined = 'combined',
        spatial = { 
          predefined = 'spongelayer', 
          plane_origin = { bounding_cube.length - 3 * elemsize, 0.0, 0 },
          plane_normal = { 1.0, 0, 0 },
          damp_factor = damp_factor,
          damp_exponent = 2,
          target_state = {
            density = dens,
            velocityX = 0.0,
            velocityY = 0.0,
            velocityZ = 0.0,
            pressure = press
          } 
        },
        shape = {   
          kind = 'canoND', 
          object= {
            origin = {
              bounding_cube.length - 3 * elemsize + eps ,
              -cubeLength / 2,
              -cubeLength / 2
            },
            vec = {
              { 2 * elemsize, 0, 0 },
              { 0, cubeLength, 0 },
              { 0, 0, cubeLength }
            },
            segments = { 100, 100, 100 },
          }
        },
      },
    },
    ---- Sponge (south)
    --st_fun = {
    --  predefined = 'combined',
    --  spatial = { 
    --    predefined = 'spongeLayer', 
    --    plane_origin = { 0.0, -bounding_cube.length / 2 + 3 * elemsize - eps, 0.0 },
    --    plane_normal = { 0, -1.0, 0 },
    --     damp_factor = damp_factor,
    --    damp_exponent = 2,
    --    target_state = {
    --      density = dens,
    --      velx = 0.0,
    --      vely = 0.0,
    --      velz = 0.0,
    --      pressure = press
    --    },
    --  },
    --  shape = {
    --    kind = 'canoND', 
    --    object= {
    --      origin = {
    --        eps ,
    --        -bounding_cube.length/2+3*elemsize-eps,
    --        -cubeLength/2
    --      },
    --      vec = {
    --        { cubeLength, 0, 0 },
    --        { 0, -2 * elemsize, 0 },
    --        { 0, 0, cubeLength }
    --      },
    --      segments = {100,100,100}
    --    },
    --  },
    --},
    ---- Sponge (north)
    --st_fun = {
    --  predefined = 'combined',
    --  spatial = { 
    --    predefined = 'spongeLayer', 
    --    plane_origin = { 0.0, bounding_cube.length/2-3*elemsize+eps, 0.0 },
    --    plane_normal = { 0, 1.0, 0 },
    --    damp_factor = damp_factor,
    --    damp_exponent = 2,
    --    target_state = {
    --      density = dens,
    --      velx = 0.0,
    --      vely = 0.0,
    --      velz = 0.0,
    --      pressure = press
    --    },
    --  },
    --  shape = {
    --    kind = 'canoND', 
    --    object= {
    --      origin = {
    --        eps ,
    --        bounding_cube.length/2-3*elemsize+eps,
    --        -cubeLength/2
    --      },
    --      vec = {
    --        { cubeLength, 0, 0 },
    --        { 0, 2 * elemsize, 0 },
    --        { 0, 0, cubeLength }
    --      },
    --      segments = { 100, 100, 100 }
    --    },
    --  },
    --},
    ---- Sponge (top)
    --st_fun = {
    --  kind = 'combined',
    --  spatial = { 
    --    predefined = 'spongeLayer', 
    --    plane_origin = { 0.0, 0.0, bounding_cube.length/2-3*elemsize+eps },
    --    plane_normal = { 0, 0, 1.0 },
    --    damp_factor = damp_factor,
    --    damp_exponent = 2,
    --    target_state = {
    --      density = dens,
    --      velx = 0.0,
    --      vely = 0.0,
    --      velz = 0.0,
    --      pressure = press
    --    },
    --  },
    --  shape = {
    --    kind = 'canoND', 
    --    object= {
    --      origin = {
    --        eps ,
    --        -cubeLength/2 ,
    --        bounding_cube.length/2-3*elemsize+eps,
    --      },
    --      vec = {
    --        { cubeLength, 0, 0 },
    --        { 0, cubeLength, 0 },
    --        { 0, 0, 2*elemsize }
    --      },
    --      segments = {100,100,100}
    --    },
    --  },
    --},
    --st_fun = {
    --  kind = 'combined',
    --  spatial = { 
    --    predefined = 'spongeLayer',
    --    plane_origin = { 0.0, 0.0, -bounding_cube.length/2+3*elemsize-eps },
    --    plane_normal = { 0, 0.0, -1.0 },
    --    damp_factor = damp_factor,
    --    damp_exponent = 2,
    --    target_state = {
    --      density = dens,
    --      velx = 0.0,
    --      vely = 0.0,
    --      velz = 0.0,
    --      pressure = press
    --    },
    --  },
    --  shape = {
    --    kind = 'canoND',
    --    object= {
    --      origin = {
    --        eps,
    --        -cubeLength/2,
    --        -bounding_cube.length/2+3*elemsize-eps
    --      },
    --      vec = {
    --        { cubeLength, 0, 0 },
    --        { 0, cubeLength, 0 },
    --        { 0, 0, -2*elemsize }
    --      },
    --      segments = { 100, 100, 100 }
    --    },
    --  },
    --},
  }
}

-- Sponge definition
source = {
  spongelayer = 'sponge'
}

-- Boundary definitions
boundary_condition = {
  { 
    label = 'inflowWest',  
    kind = 'inflow_normal',      
    density = dens_inlet,
    v_norm = velX_inlet,
    v_tan = 0.0,
  }
  ,
  { 
    label = 'outflowEast',  
    kind = 'outflow',       
    pressure = press,
  }
  ,
  { 
    label = 'outflowNorth', 
    --kind = 'slipwall',       
    kind = 'outflow',       
    pressure = press,
  }
  ,
  { 
    label = 'outflowSouth', 
    --kind = 'slipwall',       
    kind = 'outflow',       
    pressure = press,
  }
  ,
  { 
    label = 'outflowTop', 
    --kind = 'slipwall',       
    kind = 'outflow',       
    pressure = press,
  }
  ,
  { 
    label = 'outflowBottom', 
    --kind = 'slipwall',       
    kind = 'outflow',       
    pressure = press,
  }
  ,
}  

