-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of Gaussian pulse in density. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). 
-- To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the 
-- level variable. An increment of one will half the radius of your elements.

timestep_info = 1

-- ...the length of the cube
cubeLength = 2.0

-- the refinement level of the octree
level = 2

-- Transport velocity of the pulse in x direction.
velocityX = 100

-- global simulation options
sim_control = {
              name          = 'gPulseDens_euler_modg',             -- the name of the simualtion
              time_control = {   
                               min = 0, 
                               max = cubeLength/velocityX/4,              -- final simulation time
                               interval = {iter = 10}
                     }
             }


-- Mesh definitions --
mesh = { predefined = 'slice',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }


---- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/',                                        
            -- temporal definition of restart write
            time_control = {   
                              min = 0, 
                              max = sim_control.time_control.max, 
                              interval = sim_control.time_control.max/8.0,
                   },  
          }

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
          folder = './',                  -- the folder for the timing results
          filename = 'timing.res'         -- the filename of the timing results
         }

-- Equation definitions --
equation = {
    name   = 'euler_2d',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
               m = 7,                   -- the maximal polynomial degree for each spatial direction
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.1,     -- CourantFriedrichsLewy number
                         },
               },
}

projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattoPoints = false  -- if lobatto points should be used, default = false
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           --   blocksize = 32,        -- for fpt, default -1
           --   fftMultiThread = true -- for fpt, logical, default false
             }


function ic_gauss (x,y,z)
d= x*x+y*y 
return( 1.225 + 2* math.exp(-d/0.2*math.log(2)) )
end 

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = ic_gauss, 
-- {
--                                 predefined='gausspulse',
--                                 center={0.0, 0.0, 0.0},
--                                 halfwidth=0.20,
--                                 amplitude=2.0,
--                                 background=1.225
--                                },
                      -- pressure = {
                      --              predefined='gausspulse',
                      --              center={0.0, 0.0, 0.0},
                      --              halfwidth=0.2,
                      --              amplitude=2.0,
                      --              background=100000
                      --            }, 
                      pressure = 100000,
                      velocityY = velocityX ,
                   --   velocityX = velocityX ,
                   --   velocityY = 0.0,
                   --   projOverSampling =  1,    -- oversampling for quadrature projeciton of ini cond (per direction)
                   --   useFpt = true,            -- use Fast polynomial transformation to convert to modal representation
                    }


-- Tracking              
--tracking = {
--             label = 'track_momentum_2d',
--             folder = './',
--             variable = { 'momentum' },
--       --      variable = { 'density' },
--      --       variable = { {'momentum'}, {'pressure'}, {'density'} },
--             shape = {kind = 'canoND', object= { origin ={0., 0., 0.} } },
--             time = {
--                      useIterations = false,
--                      min = 0,
--                      max = simulation.time.max,
--                      interval = simulation.time.max/8.0,
--                    },
--             format = 'ascii'
--           }
