-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Maxwell equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for displacement and magnetic fields. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n. Please notice,
-- this testcase is correct only for homogenous media with epsi = mu = 1 (see equations table).
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the level variable. An increment of one will half the radius of your elements.


-- Some global parameters for the T_{nm} mode testcase
-- ...the number of mpi ranks
nprocs = 1
-- ...the uniform refinement level for the periodic cube
level = 6
-- ...the length of the cube
cubeLength = 2.0
-- ...the integer number of the mode in x direction
m = 1
-- ...the integer number of the mode in y direction
n = 1

-- The analytic soultion for this testcase is given by the following functions (only correct for epsi = mu = 1):
-- ... definition of temporal angular frequency
w = math.sqrt(m^2 + n^2)
-- ... E_x = 0.0
function displacementX(x,y,z,t)
  return 0.0
end 
-- ... E_y = 0.0
function displacementY(x,y,z,t)
  return 0.0
end 
-- ... E_z(x,y,z,t) = sin(m \pi x) sin(n \pi y) cos(w t)
function displacementZ(x,y,z,t)
  return math.sin(m*math.pi*x)*math.sin(n*math.pi*y)*math.cos(w*t)
end 
-- ... B_x(x,y,z,t) = -\frac{\pi n}{w} sin(m \pi x) cos(n \pi y) sin(w t)
function magneticX(x,y,z,t)
  return -(math.pi*n/w)*math.sin(m*math.pi*x)*math.cos(n*math.pi*y)*math.sin(w*t)
end 
-- ... B_y(x,y,z,t) = \frac{\pi m}{w} cos(m \pi x) sin(n \pi y) sin(w t)
function magneticY(x,y,z,t)
  return (math.pi*m/w)*math.cos(m*math.pi*x)*math.sin(n*math.pi*y)*math.sin(w*t)
end 
-- ... B_z = 0.0
function magneticZ(x,y,z,t)
  return 0.0
end


-- global simulation options
simulation_name = 'per_osci_maxwell_weno' -- the name of the simualtion
sim_control = {
                time_control = { 
                  max = 0.5 -- final simulation time
                }
              }

--commpattern = 'gathered_type'

-- VTK output (this is still a dummy implementation)
--output = { 
--          time      = 0.01/(math.sqrt(2)^3),                               -- time interval for VTK output
--          iteration = '',                                              -- iteration interval for VTK output
--          output_folder = './output/maxwell/'                          -- the folder the output is written to 
--         }

-- Mesh definitions --
-- mesh = './'
mesh = { predefined = 'cube',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }


-- Restart settings
--restart = { 
--            -- If this table is available it will read the restart file
--            -- given below.
--            read =  {
--                     folder    = './restart/maxwell/',                           -- folder of the restart file
--                     file_name = 'lastHeader.lua'                                -- the restart file to read
--                    },
--            -- If this table is available it will write a restart file
--            -- with the specified options.
--            write = {
--                     folder    = './restart/maxwell/',                           -- the folder the restart files are written to
--                     interval  = 5.0,                                            -- dump restart file interval
--                     tmin      = 5.0,                                            -- first timestep to output
--                     tmax      = 20.0                                            -- last timestep to output
--                    }
--          }

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
          folder = './',                  -- the folder for the timing results
          filename = 'timing.res'         -- the filename of the timing results
         }

-- Tracking of a certain variable at a certain point in space over time.           
--tracking = {
--            -- we track the displacement field
--            { 
--              label = 'probe_displacementX', 
--              variables = {'1'}, --{'displacement_fieldX'}, 
--              type='point', 
--              point ={cubeLength/2.0,cubeLength/2.0,cubeLength/2.0},
--              interval = 0.001, tmin = 0, tmax = -1 
--            }, 
--            -- TODO add tracking for magnetic field components
--  }

-- Equation definitions --
equation = {
    name   = 'maxwell',                   -- we solve maxwell's equations
    material     = {
                     -- The background permitivity and permeability
                     global = {
                                kind = 'const',
                                const = {1.0, 1.0, 0.0},
                              },
                   },
}

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'weno',            -- we use the weno scheme for reconstruction
               m = 1,                     -- the reconstructed polynomial is of degree 1
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.3,     -- CourantFriedrichsLewy number
                         },
               },
}

-- ...the initial condition table. 
-- ...initial condition function for displacement field (z component)
function ic_displacementZ(x,y,z)
  return displacementZ(x,y,z,0.0)
end
-- ...initial condition function for magnetic field (x component)
function ic_magneticX(x,y,z)
  return magneticX(x,y,z,0.0)
end
-- ...initial condition function for magnetic field (y component)
function ic_magneticY(x,y,z)
  return magneticY(x,y,z,0.0)
end
initial_condition = { 
                      displacement_fieldX = 0.0,           -- displacement field , x component
                      displacement_fieldY = 0.0,           -- displacement field , y component
                      displacement_fieldZ = ic_displacementZ,  -- displacement field , z component
                      magnetic_fieldX = ic_magneticX,  -- magnetic induction , x component
                      magnetic_fieldY = ic_magneticY,  -- magnetic induction , y component
                      magnetic_fieldZ = 0.0,           -- magnetic induction , z component
                    }

