-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Maxwell equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for displacement and magnetic fields. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n. Please notice,
-- this testcase is correct only for homogenous media with epsi = mu = 1 (see equations table).
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). To calculate a grid convergence behavior please modify the level variable. An increment of one will half the radius of your elements.


-- Some global parameters for the T_{nm} mode testcase
-- ...the uniform refinement level for the periodic cube
level = 3
-- ...the length of the cube
cubeLength = 2.0
-- ...the integer number of the mode in x direction
amplX = 1.0
-- ...the integer number of the mode in y direction
amplY = 1.0

w = math.sqrt(2)*math.pi
v = 1.0/math.sqrt(2)
-- ... E_x = 0.0
function displacementX(x,y,z,t)
  return 0.0
end 
-- ... E_y = 0.0
function displacementY(x,y,z,t)
  return 0.0 
end 
-- ... E_z(x,y,z,t) = sin(\pi x) sin(\pi y) cos(w t)
function displacementZ(x,y,z,t)
  x = x-0.25
  y = y-0.25
  z = z-0.25
  return math.sin(math.pi*x)*math.sin(math.pi*y)*math.cos(w*t)
end 
-- ... B_x(x,y,z,t) = -v sin(\pi x) cos(\pi y) sin(w t)
function magneticX(x,y,z,t)
  x = x-0.25
  y = y-0.25
  z = z-0.25
  return -v*math.sin(math.pi*x)*math.cos(math.pi*y)*math.sin(w*t)
end 
-- ... B_y(x,y,z,t) = v cos(\pi x) sin(\pi y) sin(w t)
function magneticY(x,y,z,t)
  x = x-0.25
  y = y-0.25
  z = z-0.25
   return v*math.cos(math.pi*x)*math.sin(math.pi*y)*math.sin(w*t)
end 
-- ... B_z = 0.0
function magneticZ(x,y,z,t)
  return 0.0 
end


-- global simulation options
simulation_name = 'per_osci_max_modg' -- the name of the simualtion
sim_control = {
                time_control = { 
                  min = 0, 
                  max = 0.0025, 
                  interval = 1.0 
                }
              }

--commpattern = 'gathered_type'

-- Mesh definitions --
mesh = { predefined = 'cube',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }

-- DEBUG OPTIONS
ebug = { 
         debugMode = true, 
         debugFiles = true,
         verbose = true,
         dumpTreeIDs = true,
         dumpDependencies = true,
         checkSteps = true,
         dumpState = true,
--         debugStates = { 
--                        write = {
--                                 folder    = './debug/',  -- the folder for the debug mesh 
--                                 interval  = 1,           -- dump restart file interval
--                                 tmin      = 1,           -- first timestep to output
--                                 tmax      = tmax+1       -- last timestep to output
--                                }
--                       }
        }


-- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/maxwell/',                                        
            -- temporal definition of restart write
            time_control = {   
              min = 0, 
              max = sim_control.time_control.max, 
              interval = sim_control.time_control.max/5
            }
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Tracking of a certain variable at a certain point in space over time.           
--tracking = { 
             ---- Give a name for the tracked object.
             --label = 'trackIt', 
             ---- the variable of the solver you track
             --variable = {{'displacement_field'},{'magnetic_field'}},
             ---- we track a point at the origin
             ----shape = { kind = 'global' },
             --shape={  
                      --kind = 'canoND', 
                      --object = {
                                --origin ={cubeLength/4.0,cubeLength/4.0,cubeLength/4.0}, 
                                --length = cubeLength/8.0, 
                                --segments = { 10, 10, 10 },
                               --}
                   --},
             ---- the temporal definition of the tracking: when do we start?
             --time = {
                      --useIterations = false,
                      --min = 0, 
                      ---- the maximum time: max = -1 assigns max=tmax
                      --max = simulation.time.max, 
                      --interval = simulation.time.max/10.0,
                    --},     
             --format = 'harvester',
             --folder = './tracking/maxwell/'
         --}


variable = {
  {
    name = "global_maxwell_permeability",
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = { const = { 1.0 } }
  },
  {
    name = "global_maxwell_permittivity",
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = { const = { 1.0 } }
  },
  {
    name = "global_maxwell_conductivity",
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = { const = { 0.0 } }
  }
}

-- Equation definitions --
equation = {
  name = 'maxwell',                   -- we solve maxwell's equations
  material = {
    permeability = 'global_permeability',
    permittivity = 'global_permittivity',
    conductivity = 'global_conductivity'
  }
}

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m =  7,                    -- the maximal polynomial degree for each spatial direction
               modg_space = 'P' 
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = --'explicitEuler',
               'explicitRungeKutta', 
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.01,     -- CourantÐFriedrichsÐLewy number
                         },
               },
}

-- ...the initial condition table. 
-- ...initial condition function for displacement field (z component)
function ic_displacementZ(x,y,z)
  return displacementZ(x,y,z,0.0)
end
function ic_displacementY(x,y,z)
  return displacementY(x,y,z,0.0)
end

-- ...initial condition function for magnetic field (x component)
function ic_magneticX(x,y,z)
  return magneticX(x,y,z,0.0)
end
-- ...initial condition function for magnetic field (y component)
function ic_magneticY(x,y,z)
  return magneticY(x,y,z,0.0)
end
function ic_magneticZ(x,y,z)
  return magneticY(x,y,z,0.0)
end

initial_condition = { 
                      displacement_fieldX = 0.0,           -- displacement field , x component
                      displacement_fieldY = ic_displacementY,  -- displacement field , z component
                      displacement_fieldZ = ic_displacementZ,  -- displacement field , z component
                      magnetic_fieldX = ic_magneticX,  -- magnetic induction , x component
                      magnetic_fieldY = ic_magneticY,  -- magnetic induction , y component
                      magnetic_fieldZ = ic_magneticZ,           -- magnetic induction , z component
                    }

-- ...the general projection table
projection = {
           --    kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 5.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }
