#!/bin/bash
#SBATCH --time=225:00:00
#SBATCH -N 1
#SBATCH -p all

# The maximal number of procs for the simulations (in case of smaller meshes, smaller number
# may be used)
NPROCS_TOTAL=8

# The number of OpenMP threads per rank
export OMP_NUM_THREADS=1

# load the relevant modules
module load PrgEnv/gcc-ompi/4.7.1-1.6.0

# remove old directories
rm -rf mesh_order* restart_order* harvest_order* 
rm -rf plots_order*
rm -rf plots

# the orders under consideration
order_set=(3 7 15 31) #63)

# the levels under consideration
level_set=(5 6 7)

# the spectral filter orders for post-processing
spectral_filter_set=(2 4 6 8 10 12 14 16)

# loop over all orders under condsideration
for((i=0;i<${#order_set[@]};i++))
do

  # loop over all mesh levels under condsideration
  for((j=0;j<${#level_set[@]};j++))
  do

    # the particular mesh level 
    export MESHLEVEL=${level_set[$j]}

    # calculate the number of element in the mesh
    for ((k=1, pow=4; k<MESHLEVEL-1; k++)); do ((pow *= 4)); done
    nelems=$pow

    # the particular order
    export ORDER=${order_set[$i]}

    # the number of ranks for the current mesh
    NPROCS=$(($NPROCS_TOTAL<$nelems?$NPROCS_TOTAL:$nelems))
  
    echo 'processing order ' $ORDER ' on level ' $MESHLEVEL ' with ' $nelems ' elements'
  
    SUFFIX=order"$ORDER"_level"$MESHLEVEL"
    export MESH_FOLDER=mesh_"$SUFFIX/"
    export RESTART_FOLDER=restart_"$SUFFIX/"
    export HARVEST_FOLDER=harvest_"$SUFFIX/"
    export PLOTS_FOLDER=plots_"$SUFFIX/"

    mkdir $MESH_FOLDER    
    mkdir $RESTART_FOLDER 
    mkdir $HARVEST_FOLDER 
    mkdir $PLOTS_FOLDER
    
    # run the solver
    ../../../../seeder/build/seeder seeder.lua
    mpiexec -n $NPROCS ../../../build/ateles ateles.lua
    mpiexec -n $NPROCS ../../../../harvester/build/harvester harvester.lua

    # produce plots along lines (including error plots)
    ../../../../harvester/gleaner/gleaner.py params_plot.py

    # run the post-processing by spectral filtering
    # loop over all spectral filter orders under consideration
    for((a=0;a<${#spectral_filter_set[@]};a++))
    do
      export SPECTRAL_FILTER_ORDER=${spectral_filter_set[$a]}

      FILTER_SUFFIX=order"$ORDER"_level"$MESHLEVEL"_filter"$SPECTRAL_FILTER_ORDER"
      export FILTER_RESTART_FOLDER=restart_"$FILTER_SUFFIX"
      export FILTER_HARVEST_FOLDER=harvest_"$FILTER_SUFFIX"
      export FILTER_PLOTS_FOLDER=plots_"$FILTER_SUFFIX"

      mkdir $FILTER_RESTART_FOLDER 
      mkdir $FILTER_HARVEST_FOLDER 
      mkdir $FILTER_PLOTS_FOLDER

      echo 'post-processing order ' $ORDER ' on level ' $MESHLEVEL ' with ' $nelems ' elements and spectral filter order ' $SPECTRAL_FILTER_ORDER
     
      # execute the post-processing by spectral filtering
      mpiexec -n $NPROCS ../../../build/ateles ateles_spectral_filter.lua

      # post-process the spectrally filtered solution
      mpiexec -n $NPROCS ../../../../harvester/build/harvester harvester_spectral_filter.lua
  
      # produce plots along lines (including error plots)
      ../../../../harvester/gleaner/gleaner.py params_plot_spectral_filter.py

    done
  
  done 

done
  
# create folder for plots of different orders
mkdir plots

# produce plots along lines (including error plots)
../../../../harvester/gleaner/gleaner.py params_plot_orders.py
  
