! Copyright (c) 2014 Verena Krupp <verena.krupp@uni-siegen.de>
! Copyright (c) 2014-2015 Nikhil Anand <nikhil.anand@uni-siegen.de>
! Copyright (c) 2014, 2018 Harald Klimach <harald.klimach@uni-siegen.de>
! Copyright (c) 2016 Tobias Girresser <tobias.girresser@student.uni-siegen.de>
! Copyright (c) 2016-2017, 2019 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2017 Daniel Petró <daniel.petro@student.uni-siegen.de>
!
! Permission to use, copy, modify, and distribute this software for any
! purpose with or without fee is hereby granted, provided that the above
! copyright notice and this permission notice appear in all copies.
!
! THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
! WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
! MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
! ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
! WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
! ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
! OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
! **************************************************************************** !

?? include "ply_dof_module.inc"
!> Module for routines and datatypes of Modal Discontinuous Galerkin (MODG)
!! scheme for the acoustic equation. This scheme is a spectral scheme for linear, purley hyperbolic
!! partial differential equation systems.
module atl_modg_2d_acoustic_kernel_module
  use env_module,               only: rk

  use ply_poly_project_module,  only: ply_poly_project_type, assignment(=)
  use ply_dof_module,           only: Q_space, P_space

  use atl_equation_module,      only: atl_equations_type
  use atl_facedata_module,      only: atl_facedata_type
  use atl_cube_elem_module,     only: atl_cube_elem_type
  use atl_modg_2d_scheme_module, only: atl_modg_2d_scheme_type
  use atl_acoustic_2d_flux_module, only: atl_acoustic_2d_numflux, &
   &                              atl_acoustic_2d_physFlux
  use atl_penalization_module,  only: atl_penalizationData_type
  use atl_materialPrp_module,   only: atl_material_type
  use atl_scheme_module,        only: atl_scheme_type

  implicit none

  private

  public :: atl_modg_2d_acoustic_numflux, atl_modg_2d_acoustic_physFlux


contains


  ! ****************************************************************************
  !> Calculate the physical flux for the MODG scheme and
  !! Acoustic equation.
  subroutine atl_modg_2d_acoustic_physFlux (  equation, res, state, iElem,     &
                                &  iDir,penalizationData, poly_proj, material, &
                                &  nodal_data,nodal_gradData, nodal_res,       &
                                &  elemLength,  scheme_min, scheme_current     )
    ! --------------------------------------------------------------------------
    !> The equation you solve.
    type(atl_equations_type), intent(in) :: equation
    !> To store the resulting phy flux in modal form
    real(kind=rk), intent(inout)     :: res(:,:)
    !> The state of the equation
    real(kind=rk), intent(in), optional :: state(:,:)
    !> The current Element
    integer, intent(in) :: iElem
    !> The current Direction
    integer, intent(in) :: iDir
    !> The penalization data
    type(atl_penalizationData_type), intent(inout) :: penalizationData
    !> Poly project
    type(ply_poly_project_type), intent(inout) :: poly_proj
    !> Material description for the faces on the current level
    type(atl_material_type), intent(inout) :: material
    !> The state in nodal form
    real(kind=rk), intent(in), optional :: nodal_data(:,:)
    real(kind=rk), intent(in), optional :: nodal_GradData(:,:,:)
    real(kind=rk), intent(inout) :: nodal_res(:,:)
    !> Length of the element
    real(kind=rk), intent(in) :: ElemLength
    !> The scheme information
    type(atl_scheme_type), intent(inout) :: scheme_min
    type(atl_scheme_type), intent(inout) :: scheme_current
    ! --------------------------------------------------------------------------!
    ! Loop var for all the fluid elements
    !integer :: iElem
    ! Loop var for all the dof in an element
    integer :: iDof, nDofs
    ! Rotation indices for physical flux calculation in y and z direction
    !integer :: rotX(3), rotY(3), rotZ(3)
    integer :: rot(3)
    ! --------------------------------------------------------------------------!

    ! get the rotation for the physical flux calculation in y and z direction.
    rot = equation%varRotation(iDir)%varTransformIndices(1:3)
    ndofs = poly_proj%body_2D%ndofs


    dofLoop: do iDof = 1, ndofs
        res(iDof, rot) = atl_acoustic_2d_physFlux( &
          &   state = state(iDof,rot),             &
          &   acoustic = equation%acoustic,        &
          &   idir = iDir                          )

    end do dofLoop

  end subroutine atl_modg_2d_acoustic_physFlux
  ! ****************************************************************************


  ! ****************************************************************************
  !> Calculate the numerical flux for acoustic equation and MODG scheme
  subroutine atl_modg_2d_acoustic_numFlux( mesh, equation, facedata, scheme )
    ! --------------------------------------------------------------------------
    !> The mesh you are working with.
    type(atl_cube_elem_type), intent(in) :: mesh
    !> The equation you solve.
    type(atl_equations_type), intent(in) :: equation
    !> The face representation of the state.
    type(atl_facedata_type), intent(inout) :: facedata
    !> Parameters of the modal dg scheme
    type(atl_modg_2d_scheme_type), intent(in) :: scheme
    ! --------------------------------------------------------------------------
    integer :: iDir, nFaceDofs
    ! --------------------------------------------------------------------------

    ! Numerical flux for faces in all 2 spatial face directions
    select case(scheme%basisType)
      case(Q_space)
        nFaceDofs = (scheme%maxPolyDegree+1)
      case(P_space)
?? copy :: getDofsPTens1D(scheme%maxPolyDegree, nFaceDofs)
    end select

    ! Calculate the numerical fluxes for the faces in all 2 spatial face
    ! directions
    do iDir = 1,2
      call atl_acoustic_2d_numflux(nTotalFaces =                         &
        &  size(facedata%faceRep(iDir)%dat,1),                        &
        &  nSides = size(mesh%faces%faces(iDir)%computeFace%leftPos), &
        &  nFaceDofs = nFaceDofs,                                     &
        &  faceRep  = facedata%faceRep(iDir)%dat,                     &
        &  faceFlux = facedata%faceFlux(iDir)%dat,                    &
        &  leftPos  = mesh%faces%faces(iDir)%computeFace%leftPos,     &
        &  rightPos = mesh%faces%faces(iDir)%computeFace%rightPos,    &
        &  var = equation%varRotation(iDir)%varTransformIndices(1:3), &
        &  acoustic = equation%acoustic,                              &
        &  iDir = iDir                                                )
    end do

  end subroutine atl_modg_2d_acoustic_numFlux
  ! ****************************************************************************

end module atl_modg_2d_acoustic_kernel_module
