! Copyright (c) 2013-2014, 2016 Verena Krupp <verena.krupp@uni-siegen.de>
! Copyright (c) 2014, 2016 Harald Klimach <harald.klimach@uni-siegen.de>
! Copyright (c) 2014-2018 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2014 Nikhil Anand <nikhil.anand@uni-siegen.de>
! Copyright (c) 2016 Tobias Girresser <tobias.girresser@student.uni-siegen.de>
! Copyright (c) 2017 Daniel Petró <daniel.petro@student.uni-siegen.de>
!
! Permission to use, copy, modify, and distribute this software for any
! purpose with or without fee is hereby granted, provided that the above
! copyright notice and this permission notice appear in all copies.
!
! THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
! WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
! MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
! ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
! WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
! ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
! OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
! **************************************************************************** !

!> Module to describe the acoustic equation system (linearized gas dynamics).
module atl_eqn_acoustic_module
  ! Treelm modules
  use env_module,             only: rk

  ! Aotus modules
  use aotus_module,           only: flu_State, aot_get_val
  use aot_out_general_module, only: aot_out_type
  use aot_out_module,         only: aot_out_val
  use aot_table_module,       only: aot_table_open, &
    &                               aot_table_close
  ! Treelm modules
  use tem_tools_module,       only: tem_horizontalSpacer
  use tem_logging_module,     only: logUnit
!!VK for harvester  use tem_aux_module,         only: tem_abort

  ! Ateles modules
  use atl_materialFun_module, only: atl_materialFun_type

  implicit none

  private

  public :: atl_acoustic_type
  public :: atl_load_acoustic
  public :: atl_save_acoustic
  public :: atl_dump_acoustic_eqn

  !> Datatype for acoustic equations.
  !!
  type atl_acoustic_type
    integer :: ndims
    real(kind=rk) :: density_0     !< background density
    real(kind=rk),allocatable :: velocity_0(:) !< background velocity (x,y,z) direction
    real(kind=rk) :: pressure_0    !< background pressure
    real(kind=rk) :: speedOfSound  !< speed of sound, only depends on p0 and rho0
    type(atl_materialFun_type) :: penalization    !< The functions for the penalizations
  end type atl_acoustic_type


contains


  ! ****************************************************************************
  !> summary: subroutine to intialize acoustic equation with background
  subroutine atl_load_acoustic( acoustic, conf, eq_table )
    ! --------------------------------------------------------------------------
    type(atl_acoustic_type), intent(inout) :: acoustic
    type(flu_State)                 :: conf
    integer, intent(in)             :: eq_table

    ! --------------------------------------------------------------------------
    integer                       :: iError, acoustic_table
    ! --------------------------------------------------------------------------
    ! allocate the dimension of background velocity array according to dimension
    allocate(acoustic%velocity_0(acoustic%ndims))

    call tem_horizontalSpacer(funit=logUnit(5))
    write(logUnit(5),*) 'Reading background parameters for Acoustic equation . '

    ! Open subtable for backgroudn properties
    call aot_table_open( L       = conf,           &
      &                  parent  = eq_table,       &
      &                  tHandle = acoustic_table, &
      &                  key     = 'background'    )

    if(acoustic_table.eq.0) then
      write(logUnit(1),*) 'ERROR in init_acoustic: no background properties &
        &                  defined, stopping ...'
!!VK for harvester      call tem_abort()
    end if

    !read the data from the equation table of the lua file
    call aot_get_val( l       = conf,               &
      &               thandle = acoustic_table,     &
      &               key     = 'density',          &
      &               val     = acoustic%density_0, &
      &               errcode = ierror              )
!! No background velocity for pure acoustic equation is allowed
!!VK    call aot_get_val(l = conf, thandle = acoustic_table, key = 'velocityX', &
!!VK      &              val = acoustic%velocity_0(1), &
!!VK      &              errcode = ierror)
!!VK    call aot_get_val(l = conf, thandle = acoustic_table, key = 'velocityY', &
!!VK      &              val = acoustic%velocity_0(2), &
!!VK      &              errcode = ierror)
!!VK    if (acoustic%ndims.eq.3) then
!!VK      call aot_get_val(l = conf, thandle = acoustic_table, key = 'velocityZ', &
!!VK        &              val = acoustic%velocity_0(3), &
!!VK        &              errcode = ierror)
!!VK    end if
    acoustic%velocity_0(1) = 0.0
    acoustic%velocity_0(2) = 0.0
    if (acoustic%ndims.eq.3) then
      acoustic%velocity_0(3) = 0.0
    end if
    call aot_get_val( l       = conf,                &
      &               thandle = acoustic_table,      &
      &               key     = 'pressure',          &
      &               val     = acoustic%pressure_0, &
      &               errcode = ierror               )

    ! c = sqrt( p/rho - 1/2 * v^2)
    acoustic%SpeedOfSound = sqrt( acoustic%pressure_0 / acoustic%density_0)


    ! Close the Lua table with the background information
    call aot_table_close( L = conf, tHandle = acoustic_table )

  end subroutine atl_load_acoustic
  ! ****************************************************************************


  ! ****************************************************************************
  !> dump the equation variables into the lua file
  subroutine atl_save_acoustic(me, eqn_name, conf)
    ! --------------------------------------------------------------------------
    type(atl_acoustic_type), intent(in) :: me
    character(len=*), intent(in) :: eqn_name
    type(aot_out_type) :: conf
    ! --------------------------------------------------------------------------

    call aot_out_val(put_conf= conf, vname = 'name', val = trim(eqn_name))

    ! Dump equation Properties
    call aot_out_val( put_conf = conf,        &
      &               vname    = 'density',   &
      &               val      = me%density_0 )
!!VK    call aot_out_val( put_conf = conf,            &
!!VK      &               vname    = 'velocityX',     &
!!VK      &               val      = me%velocity_0(1) )
!!VK    call aot_out_val( put_conf = conf,            &
!!VK      &               vname    = 'velocityY',     &
!!VK      &               val      = me%velocity_0(2) )
!!VK
!!VK    if (me%ndims.eq.3) then
!!VK      call aot_out_val( put_conf = conf,            &
!!VK        &               vname    = 'velocityZ',     &
!!VK        &               val      = me%velocity_0(3) )
!!VK    end if

    call aot_out_val( put_conf = conf,         &
      &               vname    = 'pressure',   &
      &               val      = me%pressure_0 )

  end subroutine atl_save_acoustic
  ! ****************************************************************************


  ! ****************************************************************************
  ! write the equation variables into logUnit
  subroutine atl_dump_acoustic_eqn( me, dumpUnit )
    ! --------------------------------------------------------------------------
    type(atl_acoustic_type), intent(in) :: me
    integer :: dumpUnit
    ! --------------------------------------------------------------------------

     write(dumpUnit,*) 'Background variable for acoustic equation.'
     write(dumpUnit,*) 'Background Density =', me%density_0
     write(dumpUnit,*) 'Background VelocityX =', me%velocity_0(1)
     write(dumpUnit,*) 'Background VelocityY =', me%velocity_0(2)
     if (me%ndims.eq.3) then
        write(dumpUnit,*) 'Background VelocityZ =', me%velocity_0(3)
     end if
     write(dumpUnit,*) 'Background Pressure =', me%pressure_0

  end subroutine atl_dump_acoustic_eqn
  ! ****************************************************************************

end module atl_eqn_acoustic_module
