/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.settings.notification.app;

import static android.app.NotificationManager.IMPORTANCE_DEFAULT;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import android.app.NotificationChannel;
import android.content.Context;

import androidx.fragment.app.FragmentActivity;
import androidx.preference.Preference;

import com.android.settings.SettingsPreferenceFragment;
import com.android.settings.notification.NotificationBackend;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.robolectric.RobolectricTestRunner;
import org.robolectric.RuntimeEnvironment;
import org.robolectric.shadows.ShadowApplication;

@RunWith(RobolectricTestRunner.class)
public class ConversationPromotePreferenceControllerTest {

    private Context mContext;
    @Mock
    private NotificationBackend mBackend;
    @Mock
    SettingsPreferenceFragment mFragment;

    private ConversationPromotePreferenceController mController;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        ShadowApplication shadowApplication = ShadowApplication.getInstance();
        mContext = RuntimeEnvironment.application;
        when(mFragment.getActivity()).thenReturn(mock(FragmentActivity.class));
        mController = spy(new ConversationPromotePreferenceController(
                mContext, mFragment, mBackend));

    }

    @Test
    public void testNoCrashIfNoOnResume() {
        mController.isAvailable();
        mController.updateState(mock(Preference.class));
        mController.handlePreferenceTreeClick(mock(Preference.class));
    }

    @Test
    public void testIsAvailable_notConversation() {
        NotificationBackend.AppRow appRow = new NotificationBackend.AppRow();
        NotificationChannel channel = new NotificationChannel("", "", IMPORTANCE_DEFAULT);
        mController.onResume(appRow, channel, null, null, null, null);
        assertFalse(mController.isAvailable());
    }

    @Test
    public void testIsAvailable_conversation_notDemoted() {
        NotificationBackend.AppRow appRow = new NotificationBackend.AppRow();
        NotificationChannel channel = new NotificationChannel("", "", IMPORTANCE_DEFAULT);
        channel.setConversationId("a", "a");
        mController.onResume(appRow, channel, null, null, null, null);
        assertFalse(mController.isAvailable());
    }

    @Test
    public void testIsAvailable_conversation_demoted() {
        NotificationBackend.AppRow appRow = new NotificationBackend.AppRow();
        NotificationChannel channel = new NotificationChannel("", "", IMPORTANCE_DEFAULT);
        channel.setConversationId("a", "a");
        channel.setDemoted(true);
        mController.onResume(appRow, channel, null, null, null, null);
        assertTrue(mController.isAvailable());
    }

    @Test
    public void testHandlePreferenceClick() {
        NotificationBackend.AppRow appRow = new NotificationBackend.AppRow();
        NotificationChannel channel = new NotificationChannel("", "", IMPORTANCE_DEFAULT);
        channel.setConversationId("a", "a");
        channel.setDemoted(true);
        channel.setBypassDnd(true);
        mController.onResume(appRow, channel, null, null, null, null);

        Preference pref = mock(Preference.class);
        when(pref.getKey()).thenReturn("convo_promote");
        assertTrue(mController.handlePreferenceTreeClick(pref));

        ArgumentCaptor<NotificationChannel> captor =
                ArgumentCaptor.forClass(NotificationChannel.class);

        verify(mBackend).updateChannel(eq(null), anyInt(), captor.capture());
        assertFalse(captor.getValue().isDemoted());
        assertFalse(captor.getValue().canBypassDnd());

        verify(mFragment).getActivity();
    }

    @Test
    public void testHandlePreferenceClick_wrongKey() {
        NotificationBackend.AppRow appRow = new NotificationBackend.AppRow();
        NotificationChannel channel = new NotificationChannel("", "", IMPORTANCE_DEFAULT);
        channel.setConversationId("a", "a");
        channel.setDemoted(true);
        mController.onResume(appRow, channel, null, null, null, null);

        Preference pref = mock(Preference.class);
        when(pref.getKey()).thenReturn("wrong");
        assertFalse(mController.handlePreferenceTreeClick(pref));

        verify(mBackend, never()).updateChannel(eq(null), anyInt(), any());
        verify(mFragment, never()).getActivity();
    }
}
