/**************************************************************************

Copyright 2000, 2001 ATI Technologies Inc., Ontario, Canada, and
                     VA Linux Systems Inc., Fremont, California.

All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice (including the
next paragraph) shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE COPYRIGHT OWNER(S) AND/OR ITS SUPPLIERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

**************************************************************************/

/**
 * \file radeon_screen.c
 * Screen initialization functions for the Radeon driver.
 *
 * \author Kevin E. Martin <martin@valinux.com>
 * \author  Gareth Hughes <gareth@valinux.com>
 */

#include <errno.h>
#include "main/glheader.h"
#include "main/imports.h"
#include "main/mtypes.h"
#include "main/framebuffer.h"
#include "main/renderbuffer.h"
#include "main/fbobject.h"

#define STANDALONE_MMIO
#include "radeon_chipset.h"
#include "radeon_macros.h"
#include "radeon_screen.h"
#include "radeon_common.h"
#include "radeon_common_context.h"
#if defined(RADEON_R100)
#include "radeon_context.h"
#include "radeon_tex.h"
#elif defined(RADEON_R200)
#include "r200_context.h"
#include "r200_tex.h"
#endif

#include "utils.h"

#include "GL/internal/dri_interface.h"

/* Radeon configuration
 */
#include "xmlpool.h"

#define DRI_CONF_COMMAND_BUFFER_SIZE(def,min,max) \
DRI_CONF_OPT_BEGIN_V(command_buffer_size,int,def, # min ":" # max ) \
        DRI_CONF_DESC(en,"Size of command buffer (in KB)") \
        DRI_CONF_DESC(de,"Grösse des Befehlspuffers (in KB)") \
DRI_CONF_OPT_END

#if defined(RADEON_R100)	/* R100 */
PUBLIC const char __driConfigOptions[] =
DRI_CONF_BEGIN
    DRI_CONF_SECTION_PERFORMANCE
        DRI_CONF_TCL_MODE(DRI_CONF_TCL_CODEGEN)
        DRI_CONF_FTHROTTLE_MODE(DRI_CONF_FTHROTTLE_IRQS)
        DRI_CONF_VBLANK_MODE(DRI_CONF_VBLANK_DEF_INTERVAL_0)
        DRI_CONF_MAX_TEXTURE_UNITS(3,2,3)
        DRI_CONF_HYPERZ(false)
        DRI_CONF_COMMAND_BUFFER_SIZE(8, 8, 32)
    DRI_CONF_SECTION_END
    DRI_CONF_SECTION_QUALITY
        DRI_CONF_TEXTURE_DEPTH(DRI_CONF_TEXTURE_DEPTH_FB)
        DRI_CONF_DEF_MAX_ANISOTROPY(1.0,"1.0,2.0,4.0,8.0,16.0")
        DRI_CONF_NO_NEG_LOD_BIAS(false)
        DRI_CONF_FORCE_S3TC_ENABLE(false)
        DRI_CONF_COLOR_REDUCTION(DRI_CONF_COLOR_REDUCTION_DITHER)
        DRI_CONF_ROUND_MODE(DRI_CONF_ROUND_TRUNC)
        DRI_CONF_DITHER_MODE(DRI_CONF_DITHER_XERRORDIFF)
        DRI_CONF_ALLOW_LARGE_TEXTURES(2)
    DRI_CONF_SECTION_END
    DRI_CONF_SECTION_DEBUG
        DRI_CONF_NO_RAST(false)
    DRI_CONF_SECTION_END
DRI_CONF_END;
static const GLuint __driNConfigOptions = 15;

#elif defined(RADEON_R200)

PUBLIC const char __driConfigOptions[] =
DRI_CONF_BEGIN
    DRI_CONF_SECTION_PERFORMANCE
        DRI_CONF_TCL_MODE(DRI_CONF_TCL_CODEGEN)
        DRI_CONF_FTHROTTLE_MODE(DRI_CONF_FTHROTTLE_IRQS)
        DRI_CONF_VBLANK_MODE(DRI_CONF_VBLANK_DEF_INTERVAL_0)
        DRI_CONF_MAX_TEXTURE_UNITS(6,2,6)
        DRI_CONF_HYPERZ(false)
        DRI_CONF_COMMAND_BUFFER_SIZE(8, 8, 32)
    DRI_CONF_SECTION_END
    DRI_CONF_SECTION_QUALITY
        DRI_CONF_TEXTURE_DEPTH(DRI_CONF_TEXTURE_DEPTH_FB)
        DRI_CONF_DEF_MAX_ANISOTROPY(1.0,"1.0,2.0,4.0,8.0,16.0")
        DRI_CONF_NO_NEG_LOD_BIAS(false)
        DRI_CONF_FORCE_S3TC_ENABLE(false)
        DRI_CONF_COLOR_REDUCTION(DRI_CONF_COLOR_REDUCTION_DITHER)
        DRI_CONF_ROUND_MODE(DRI_CONF_ROUND_TRUNC)
        DRI_CONF_DITHER_MODE(DRI_CONF_DITHER_XERRORDIFF)
        DRI_CONF_ALLOW_LARGE_TEXTURES(2)
        DRI_CONF_TEXTURE_BLEND_QUALITY(1.0,"0.0:1.0")
    DRI_CONF_SECTION_END
    DRI_CONF_SECTION_DEBUG
        DRI_CONF_NO_RAST(false)
    DRI_CONF_SECTION_END
    DRI_CONF_SECTION_SOFTWARE
        DRI_CONF_NV_VERTEX_PROGRAM(false)
    DRI_CONF_SECTION_END
DRI_CONF_END;
static const GLuint __driNConfigOptions = 17;

#endif

#ifndef RADEON_INFO_TILE_CONFIG
#define RADEON_INFO_TILE_CONFIG 0x6
#endif

static int
radeonGetParam(__DRIscreen *sPriv, int param, void *value)
{
  int ret;
  drm_radeon_getparam_t gp = { 0 };
  struct drm_radeon_info info = { 0 };

  if (sPriv->drm_version.major >= 2) {
      info.value = (uint64_t)(uintptr_t)value;
      switch (param) {
      case RADEON_PARAM_DEVICE_ID:
          info.request = RADEON_INFO_DEVICE_ID;
          break;
      case RADEON_PARAM_NUM_GB_PIPES:
          info.request = RADEON_INFO_NUM_GB_PIPES;
          break;
      case RADEON_PARAM_NUM_Z_PIPES:
          info.request = RADEON_INFO_NUM_Z_PIPES;
          break;
      case RADEON_INFO_TILE_CONFIG:
	  info.request = RADEON_INFO_TILE_CONFIG;
          break;
      default:
          return -EINVAL;
      }
      ret = drmCommandWriteRead(sPriv->fd, DRM_RADEON_INFO, &info, sizeof(info));
  } else {
      gp.param = param;
      gp.value = value;

      ret = drmCommandWriteRead(sPriv->fd, DRM_RADEON_GETPARAM, &gp, sizeof(gp));
  }
  return ret;
}

#if defined(RADEON_R100)
static const __DRItexBufferExtension radeonTexBufferExtension = {
    { __DRI_TEX_BUFFER, __DRI_TEX_BUFFER_VERSION },
   radeonSetTexBuffer,
   radeonSetTexBuffer2,
};
#endif

#if defined(RADEON_R200)
static const __DRItexBufferExtension r200TexBufferExtension = {
    { __DRI_TEX_BUFFER, __DRI_TEX_BUFFER_VERSION },
   r200SetTexBuffer,
   r200SetTexBuffer2,
};
#endif

static void
radeonDRI2Flush(__DRIdrawable *drawable)
{
    radeonContextPtr rmesa;

    rmesa = (radeonContextPtr) drawable->driContextPriv->driverPrivate;
    radeonFlush(rmesa->glCtx);
}

static const struct __DRI2flushExtensionRec radeonFlushExtension = {
    { __DRI2_FLUSH, __DRI2_FLUSH_VERSION },
    radeonDRI2Flush,
    dri2InvalidateDrawable,
};

static __DRIimage *
radeon_create_image_from_name(__DRIscreen *screen,
                              int width, int height, int format,
                              int name, int pitch, void *loaderPrivate)
{
   __DRIimage *image;
   radeonScreenPtr radeonScreen = screen->driverPrivate;

   if (name == 0)
      return NULL;

   image = CALLOC(sizeof *image);
   if (image == NULL)
      return NULL;

   switch (format) {
   case __DRI_IMAGE_FORMAT_RGB565:
      image->format = MESA_FORMAT_RGB565;
      image->internal_format = GL_RGB;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   case __DRI_IMAGE_FORMAT_XRGB8888:
      image->format = MESA_FORMAT_XRGB8888;
      image->internal_format = GL_RGB;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   case __DRI_IMAGE_FORMAT_ARGB8888:
      image->format = MESA_FORMAT_ARGB8888;
      image->internal_format = GL_RGBA;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   default:
      free(image);
      return NULL;
   }

   image->data = loaderPrivate;
   image->cpp = _mesa_get_format_bytes(image->format);
   image->width = width;
   image->pitch = pitch;
   image->height = height;

   image->bo = radeon_bo_open(radeonScreen->bom,
                              (uint32_t)name,
                              image->pitch * image->height * image->cpp,
                              0,
                              RADEON_GEM_DOMAIN_VRAM,
                              0);

   if (image->bo == NULL) {
      FREE(image);
      return NULL;
   }

   return image;
}

static __DRIimage *
radeon_create_image_from_renderbuffer(__DRIcontext *context,
                                      int renderbuffer, void *loaderPrivate)
{
   __DRIimage *image;
   radeonContextPtr radeon = context->driverPrivate;
   struct gl_renderbuffer *rb;
   struct radeon_renderbuffer *rrb;

   rb = _mesa_lookup_renderbuffer(radeon->glCtx, renderbuffer);
   if (!rb) {
      _mesa_error(radeon->glCtx,
                  GL_INVALID_OPERATION, "glRenderbufferExternalMESA");
      return NULL;
   }

   rrb = radeon_renderbuffer(rb);
   image = CALLOC(sizeof *image);
   if (image == NULL)
      return NULL;

   image->internal_format = rb->InternalFormat;
   image->format = rb->Format;
   image->cpp = rrb->cpp;
   image->data_type = rb->DataType;
   image->data = loaderPrivate;
   radeon_bo_ref(rrb->bo);
   image->bo = rrb->bo;

   image->width = rb->Width;
   image->height = rb->Height;
   image->pitch = rrb->pitch / image->cpp;

   return image;
}

static void
radeon_destroy_image(__DRIimage *image)
{
   radeon_bo_unref(image->bo);
   FREE(image);
}

static __DRIimage *
radeon_create_image(__DRIscreen *screen,
                    int width, int height, int format,
                    unsigned int use,
                    void *loaderPrivate)
{
   __DRIimage *image;
   radeonScreenPtr radeonScreen = screen->driverPrivate;

   image = CALLOC(sizeof *image);
   if (image == NULL)
      return NULL;

   switch (format) {
   case __DRI_IMAGE_FORMAT_RGB565:
      image->format = MESA_FORMAT_RGB565;
      image->internal_format = GL_RGB;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   case __DRI_IMAGE_FORMAT_XRGB8888:
      image->format = MESA_FORMAT_XRGB8888;
      image->internal_format = GL_RGB;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   case __DRI_IMAGE_FORMAT_ARGB8888:
      image->format = MESA_FORMAT_ARGB8888;
      image->internal_format = GL_RGBA;
      image->data_type = GL_UNSIGNED_BYTE;
      break;
   default:
      free(image);
      return NULL;
   }

   image->data = loaderPrivate;
   image->cpp = _mesa_get_format_bytes(image->format);
   image->width = width;
   image->height = height;
   image->pitch = ((image->cpp * image->width + 255) & ~255) / image->cpp;

   image->bo = radeon_bo_open(radeonScreen->bom,
                              0,
                              image->pitch * image->height * image->cpp,
                              0,
                              RADEON_GEM_DOMAIN_VRAM,
                              0);

   if (image->bo == NULL) {
      FREE(image);
      return NULL;
   }

   return image;
}

static GLboolean
radeon_query_image(__DRIimage *image, int attrib, int *value)
{
   switch (attrib) {
   case __DRI_IMAGE_ATTRIB_STRIDE:
      *value = image->pitch * image->cpp;
      return GL_TRUE;
   case __DRI_IMAGE_ATTRIB_HANDLE:
      *value = image->bo->handle;
      return GL_TRUE;
   case __DRI_IMAGE_ATTRIB_NAME:
      radeon_gem_get_kernel_name(image->bo, (uint32_t *) value);
      return GL_TRUE;
   default:
      return GL_FALSE;
   }
}

static struct __DRIimageExtensionRec radeonImageExtension = {
    { __DRI_IMAGE, __DRI_IMAGE_VERSION },
   radeon_create_image_from_name,
   radeon_create_image_from_renderbuffer,
   radeon_destroy_image,
   radeon_create_image,
   radeon_query_image
};

static int radeon_set_screen_flags(radeonScreenPtr screen, int device_id)
{
   screen->device_id = device_id;
   screen->chip_flags = 0;
   switch ( device_id ) {
   case PCI_CHIP_RN50_515E:
   case PCI_CHIP_RN50_5969:
	return -1;

   case PCI_CHIP_RADEON_LY:
   case PCI_CHIP_RADEON_LZ:
   case PCI_CHIP_RADEON_QY:
   case PCI_CHIP_RADEON_QZ:
      screen->chip_family = CHIP_FAMILY_RV100;
      break;

   case PCI_CHIP_RS100_4136:
   case PCI_CHIP_RS100_4336:
      screen->chip_family = CHIP_FAMILY_RS100;
      break;

   case PCI_CHIP_RS200_4137:
   case PCI_CHIP_RS200_4337:
   case PCI_CHIP_RS250_4237:
   case PCI_CHIP_RS250_4437:
      screen->chip_family = CHIP_FAMILY_RS200;
      break;

   case PCI_CHIP_RADEON_QD:
   case PCI_CHIP_RADEON_QE:
   case PCI_CHIP_RADEON_QF:
   case PCI_CHIP_RADEON_QG:
      /* all original radeons (7200) presumably have a stencil op bug */
      screen->chip_family = CHIP_FAMILY_R100;
      screen->chip_flags = RADEON_CHIPSET_TCL | RADEON_CHIPSET_BROKEN_STENCIL;
      break;

   case PCI_CHIP_RV200_QW:
   case PCI_CHIP_RV200_QX:
   case PCI_CHIP_RADEON_LW:
   case PCI_CHIP_RADEON_LX:
      screen->chip_family = CHIP_FAMILY_RV200;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_R200_BB:
   case PCI_CHIP_R200_QH:
   case PCI_CHIP_R200_QL:
   case PCI_CHIP_R200_QM:
      screen->chip_family = CHIP_FAMILY_R200;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV250_If:
   case PCI_CHIP_RV250_Ig:
   case PCI_CHIP_RV250_Ld:
   case PCI_CHIP_RV250_Lf:
   case PCI_CHIP_RV250_Lg:
      screen->chip_family = CHIP_FAMILY_RV250;
      screen->chip_flags = R200_CHIPSET_YCBCR_BROKEN | RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV280_5960:
   case PCI_CHIP_RV280_5961:
   case PCI_CHIP_RV280_5962:
   case PCI_CHIP_RV280_5964:
   case PCI_CHIP_RV280_5965:
   case PCI_CHIP_RV280_5C61:
   case PCI_CHIP_RV280_5C63:
      screen->chip_family = CHIP_FAMILY_RV280;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RS300_5834:
   case PCI_CHIP_RS300_5835:
   case PCI_CHIP_RS350_7834:
   case PCI_CHIP_RS350_7835:
      screen->chip_family = CHIP_FAMILY_RS300;
      break;

   case PCI_CHIP_R300_AD:
   case PCI_CHIP_R300_AE:
   case PCI_CHIP_R300_AF:
   case PCI_CHIP_R300_AG:
   case PCI_CHIP_R300_ND:
   case PCI_CHIP_R300_NE:
   case PCI_CHIP_R300_NF:
   case PCI_CHIP_R300_NG:
      screen->chip_family = CHIP_FAMILY_R300;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV350_AP:
   case PCI_CHIP_RV350_AQ:
   case PCI_CHIP_RV350_AR:
   case PCI_CHIP_RV350_AS:
   case PCI_CHIP_RV350_AT:
   case PCI_CHIP_RV350_AV:
   case PCI_CHIP_RV350_AU:
   case PCI_CHIP_RV350_NP:
   case PCI_CHIP_RV350_NQ:
   case PCI_CHIP_RV350_NR:
   case PCI_CHIP_RV350_NS:
   case PCI_CHIP_RV350_NT:
   case PCI_CHIP_RV350_NV:
      screen->chip_family = CHIP_FAMILY_RV350;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_R350_AH:
   case PCI_CHIP_R350_AI:
   case PCI_CHIP_R350_AJ:
   case PCI_CHIP_R350_AK:
   case PCI_CHIP_R350_NH:
   case PCI_CHIP_R350_NI:
   case PCI_CHIP_R360_NJ:
   case PCI_CHIP_R350_NK:
      screen->chip_family = CHIP_FAMILY_R350;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV370_5460:
   case PCI_CHIP_RV370_5462:
   case PCI_CHIP_RV370_5464:
   case PCI_CHIP_RV370_5B60:
   case PCI_CHIP_RV370_5B62:
   case PCI_CHIP_RV370_5B63:
   case PCI_CHIP_RV370_5B64:
   case PCI_CHIP_RV370_5B65:
   case PCI_CHIP_RV380_3150:
   case PCI_CHIP_RV380_3152:
   case PCI_CHIP_RV380_3154:
   case PCI_CHIP_RV380_3155:
   case PCI_CHIP_RV380_3E50:
   case PCI_CHIP_RV380_3E54:
      screen->chip_family = CHIP_FAMILY_RV380;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_R420_JN:
   case PCI_CHIP_R420_JH:
   case PCI_CHIP_R420_JI:
   case PCI_CHIP_R420_JJ:
   case PCI_CHIP_R420_JK:
   case PCI_CHIP_R420_JL:
   case PCI_CHIP_R420_JM:
   case PCI_CHIP_R420_JO:
   case PCI_CHIP_R420_JP:
   case PCI_CHIP_R420_JT:
   case PCI_CHIP_R481_4B49:
   case PCI_CHIP_R481_4B4A:
   case PCI_CHIP_R481_4B4B:
   case PCI_CHIP_R481_4B4C:
   case PCI_CHIP_R423_UH:
   case PCI_CHIP_R423_UI:
   case PCI_CHIP_R423_UJ:
   case PCI_CHIP_R423_UK:
   case PCI_CHIP_R430_554C:
   case PCI_CHIP_R430_554D:
   case PCI_CHIP_R430_554E:
   case PCI_CHIP_R430_554F:
   case PCI_CHIP_R423_5550:
   case PCI_CHIP_R423_UQ:
   case PCI_CHIP_R423_UR:
   case PCI_CHIP_R423_UT:
   case PCI_CHIP_R430_5D48:
   case PCI_CHIP_R430_5D49:
   case PCI_CHIP_R430_5D4A:
   case PCI_CHIP_R480_5D4C:
   case PCI_CHIP_R480_5D4D:
   case PCI_CHIP_R480_5D4E:
   case PCI_CHIP_R480_5D4F:
   case PCI_CHIP_R480_5D50:
   case PCI_CHIP_R480_5D52:
   case PCI_CHIP_R423_5D57:
      screen->chip_family = CHIP_FAMILY_R420;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV410_5E4C:
   case PCI_CHIP_RV410_5E4F:
   case PCI_CHIP_RV410_564A:
   case PCI_CHIP_RV410_564B:
   case PCI_CHIP_RV410_564F:
   case PCI_CHIP_RV410_5652:
   case PCI_CHIP_RV410_5653:
   case PCI_CHIP_RV410_5657:
   case PCI_CHIP_RV410_5E48:
   case PCI_CHIP_RV410_5E4A:
   case PCI_CHIP_RV410_5E4B:
   case PCI_CHIP_RV410_5E4D:
      screen->chip_family = CHIP_FAMILY_RV410;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RS480_5954:
   case PCI_CHIP_RS480_5955:
   case PCI_CHIP_RS482_5974:
   case PCI_CHIP_RS482_5975:
   case PCI_CHIP_RS400_5A41:
   case PCI_CHIP_RS400_5A42:
   case PCI_CHIP_RC410_5A61:
   case PCI_CHIP_RC410_5A62:
      screen->chip_family = CHIP_FAMILY_RS400;
      break;

   case PCI_CHIP_RS600_793F:
   case PCI_CHIP_RS600_7941:
   case PCI_CHIP_RS600_7942:
      screen->chip_family = CHIP_FAMILY_RS600;
      break;

   case PCI_CHIP_RS690_791E:
   case PCI_CHIP_RS690_791F:
      screen->chip_family = CHIP_FAMILY_RS690;
      break;
   case PCI_CHIP_RS740_796C:
   case PCI_CHIP_RS740_796D:
   case PCI_CHIP_RS740_796E:
   case PCI_CHIP_RS740_796F:
      screen->chip_family = CHIP_FAMILY_RS740;
      break;

   case PCI_CHIP_R520_7100:
   case PCI_CHIP_R520_7101:
   case PCI_CHIP_R520_7102:
   case PCI_CHIP_R520_7103:
   case PCI_CHIP_R520_7104:
   case PCI_CHIP_R520_7105:
   case PCI_CHIP_R520_7106:
   case PCI_CHIP_R520_7108:
   case PCI_CHIP_R520_7109:
   case PCI_CHIP_R520_710A:
   case PCI_CHIP_R520_710B:
   case PCI_CHIP_R520_710C:
   case PCI_CHIP_R520_710E:
   case PCI_CHIP_R520_710F:
      screen->chip_family = CHIP_FAMILY_R520;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV515_7140:
   case PCI_CHIP_RV515_7141:
   case PCI_CHIP_RV515_7142:
   case PCI_CHIP_RV515_7143:
   case PCI_CHIP_RV515_7144:
   case PCI_CHIP_RV515_7145:
   case PCI_CHIP_RV515_7146:
   case PCI_CHIP_RV515_7147:
   case PCI_CHIP_RV515_7149:
   case PCI_CHIP_RV515_714A:
   case PCI_CHIP_RV515_714B:
   case PCI_CHIP_RV515_714C:
   case PCI_CHIP_RV515_714D:
   case PCI_CHIP_RV515_714E:
   case PCI_CHIP_RV515_714F:
   case PCI_CHIP_RV515_7151:
   case PCI_CHIP_RV515_7152:
   case PCI_CHIP_RV515_7153:
   case PCI_CHIP_RV515_715E:
   case PCI_CHIP_RV515_715F:
   case PCI_CHIP_RV515_7180:
   case PCI_CHIP_RV515_7181:
   case PCI_CHIP_RV515_7183:
   case PCI_CHIP_RV515_7186:
   case PCI_CHIP_RV515_7187:
   case PCI_CHIP_RV515_7188:
   case PCI_CHIP_RV515_718A:
   case PCI_CHIP_RV515_718B:
   case PCI_CHIP_RV515_718C:
   case PCI_CHIP_RV515_718D:
   case PCI_CHIP_RV515_718F:
   case PCI_CHIP_RV515_7193:
   case PCI_CHIP_RV515_7196:
   case PCI_CHIP_RV515_719B:
   case PCI_CHIP_RV515_719F:
   case PCI_CHIP_RV515_7200:
   case PCI_CHIP_RV515_7210:
   case PCI_CHIP_RV515_7211:
      screen->chip_family = CHIP_FAMILY_RV515;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV530_71C0:
   case PCI_CHIP_RV530_71C1:
   case PCI_CHIP_RV530_71C2:
   case PCI_CHIP_RV530_71C3:
   case PCI_CHIP_RV530_71C4:
   case PCI_CHIP_RV530_71C5:
   case PCI_CHIP_RV530_71C6:
   case PCI_CHIP_RV530_71C7:
   case PCI_CHIP_RV530_71CD:
   case PCI_CHIP_RV530_71CE:
   case PCI_CHIP_RV530_71D2:
   case PCI_CHIP_RV530_71D4:
   case PCI_CHIP_RV530_71D5:
   case PCI_CHIP_RV530_71D6:
   case PCI_CHIP_RV530_71DA:
   case PCI_CHIP_RV530_71DE:
      screen->chip_family = CHIP_FAMILY_RV530;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_R580_7240:
   case PCI_CHIP_R580_7243:
   case PCI_CHIP_R580_7244:
   case PCI_CHIP_R580_7245:
   case PCI_CHIP_R580_7246:
   case PCI_CHIP_R580_7247:
   case PCI_CHIP_R580_7248:
   case PCI_CHIP_R580_7249:
   case PCI_CHIP_R580_724A:
   case PCI_CHIP_R580_724B:
   case PCI_CHIP_R580_724C:
   case PCI_CHIP_R580_724D:
   case PCI_CHIP_R580_724E:
   case PCI_CHIP_R580_724F:
   case PCI_CHIP_R580_7284:
      screen->chip_family = CHIP_FAMILY_R580;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV570_7280:
   case PCI_CHIP_RV560_7281:
   case PCI_CHIP_RV560_7283:
   case PCI_CHIP_RV560_7287:
   case PCI_CHIP_RV570_7288:
   case PCI_CHIP_RV570_7289:
   case PCI_CHIP_RV570_728B:
   case PCI_CHIP_RV570_728C:
   case PCI_CHIP_RV560_7290:
   case PCI_CHIP_RV560_7291:
   case PCI_CHIP_RV560_7293:
   case PCI_CHIP_RV560_7297:
      screen->chip_family = CHIP_FAMILY_RV560;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_R600_9400:
   case PCI_CHIP_R600_9401:
   case PCI_CHIP_R600_9402:
   case PCI_CHIP_R600_9403:
   case PCI_CHIP_R600_9405:
   case PCI_CHIP_R600_940A:
   case PCI_CHIP_R600_940B:
   case PCI_CHIP_R600_940F:
      screen->chip_family = CHIP_FAMILY_R600;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV610_94C0:
   case PCI_CHIP_RV610_94C1:
   case PCI_CHIP_RV610_94C3:
   case PCI_CHIP_RV610_94C4:
   case PCI_CHIP_RV610_94C5:
   case PCI_CHIP_RV610_94C6:
   case PCI_CHIP_RV610_94C7:
   case PCI_CHIP_RV610_94C8:
   case PCI_CHIP_RV610_94C9:
   case PCI_CHIP_RV610_94CB:
   case PCI_CHIP_RV610_94CC:
   case PCI_CHIP_RV610_94CD:
      screen->chip_family = CHIP_FAMILY_RV610;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV630_9580:
   case PCI_CHIP_RV630_9581:
   case PCI_CHIP_RV630_9583:
   case PCI_CHIP_RV630_9586:
   case PCI_CHIP_RV630_9587:
   case PCI_CHIP_RV630_9588:
   case PCI_CHIP_RV630_9589:
   case PCI_CHIP_RV630_958A:
   case PCI_CHIP_RV630_958B:
   case PCI_CHIP_RV630_958C:
   case PCI_CHIP_RV630_958D:
   case PCI_CHIP_RV630_958E:
   case PCI_CHIP_RV630_958F:
      screen->chip_family = CHIP_FAMILY_RV630;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV670_9500:
   case PCI_CHIP_RV670_9501:
   case PCI_CHIP_RV670_9504:
   case PCI_CHIP_RV670_9505:
   case PCI_CHIP_RV670_9506:
   case PCI_CHIP_RV670_9507:
   case PCI_CHIP_RV670_9508:
   case PCI_CHIP_RV670_9509:
   case PCI_CHIP_RV670_950F:
   case PCI_CHIP_RV670_9511:
   case PCI_CHIP_RV670_9515:
   case PCI_CHIP_RV670_9517:
   case PCI_CHIP_RV670_9519:
      screen->chip_family = CHIP_FAMILY_RV670;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV620_95C0:
   case PCI_CHIP_RV620_95C2:
   case PCI_CHIP_RV620_95C4:
   case PCI_CHIP_RV620_95C5:
   case PCI_CHIP_RV620_95C6:
   case PCI_CHIP_RV620_95C7:
   case PCI_CHIP_RV620_95C9:
   case PCI_CHIP_RV620_95CC:
   case PCI_CHIP_RV620_95CD:
   case PCI_CHIP_RV620_95CE:
   case PCI_CHIP_RV620_95CF:
      screen->chip_family = CHIP_FAMILY_RV620;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV635_9590:
   case PCI_CHIP_RV635_9591:
   case PCI_CHIP_RV635_9593:
   case PCI_CHIP_RV635_9595:
   case PCI_CHIP_RV635_9596:
   case PCI_CHIP_RV635_9597:
   case PCI_CHIP_RV635_9598:
   case PCI_CHIP_RV635_9599:
   case PCI_CHIP_RV635_959B:
      screen->chip_family = CHIP_FAMILY_RV635;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RS780_9610:
   case PCI_CHIP_RS780_9611:
   case PCI_CHIP_RS780_9612:
   case PCI_CHIP_RS780_9613:
   case PCI_CHIP_RS780_9614:
   case PCI_CHIP_RS780_9615:
   case PCI_CHIP_RS780_9616:
      screen->chip_family = CHIP_FAMILY_RS780;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;
   case PCI_CHIP_RS880_9710:
   case PCI_CHIP_RS880_9711:
   case PCI_CHIP_RS880_9712:
   case PCI_CHIP_RS880_9713:
   case PCI_CHIP_RS880_9714:
   case PCI_CHIP_RS880_9715:
      screen->chip_family = CHIP_FAMILY_RS880;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV770_9440:
   case PCI_CHIP_RV770_9441:
   case PCI_CHIP_RV770_9442:
   case PCI_CHIP_RV770_9443:
   case PCI_CHIP_RV770_9444:
   case PCI_CHIP_RV770_9446:
   case PCI_CHIP_RV770_944A:
   case PCI_CHIP_RV770_944B:
   case PCI_CHIP_RV770_944C:
   case PCI_CHIP_RV770_944E:
   case PCI_CHIP_RV770_9450:
   case PCI_CHIP_RV770_9452:
   case PCI_CHIP_RV770_9456:
   case PCI_CHIP_RV770_945A:
   case PCI_CHIP_RV770_945B:
   case PCI_CHIP_RV770_945E:
   case PCI_CHIP_RV790_9460:
   case PCI_CHIP_RV790_9462:
   case PCI_CHIP_RV770_946A:
   case PCI_CHIP_RV770_946B:
   case PCI_CHIP_RV770_947A:
   case PCI_CHIP_RV770_947B:
      screen->chip_family = CHIP_FAMILY_RV770;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV730_9480:
   case PCI_CHIP_RV730_9487:
   case PCI_CHIP_RV730_9488:
   case PCI_CHIP_RV730_9489:
   case PCI_CHIP_RV730_948A:
   case PCI_CHIP_RV730_948F:
   case PCI_CHIP_RV730_9490:
   case PCI_CHIP_RV730_9491:
   case PCI_CHIP_RV730_9495:
   case PCI_CHIP_RV730_9498:
   case PCI_CHIP_RV730_949C:
   case PCI_CHIP_RV730_949E:
   case PCI_CHIP_RV730_949F:
      screen->chip_family = CHIP_FAMILY_RV730;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV710_9540:
   case PCI_CHIP_RV710_9541:
   case PCI_CHIP_RV710_9542:
   case PCI_CHIP_RV710_954E:
   case PCI_CHIP_RV710_954F:
   case PCI_CHIP_RV710_9552:
   case PCI_CHIP_RV710_9553:
   case PCI_CHIP_RV710_9555:
   case PCI_CHIP_RV710_9557:
   case PCI_CHIP_RV710_955F:
      screen->chip_family = CHIP_FAMILY_RV710;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

   case PCI_CHIP_RV740_94A0:
   case PCI_CHIP_RV740_94A1:
   case PCI_CHIP_RV740_94A3:
   case PCI_CHIP_RV740_94B1:
   case PCI_CHIP_RV740_94B3:
   case PCI_CHIP_RV740_94B4:
   case PCI_CHIP_RV740_94B5:
   case PCI_CHIP_RV740_94B9:
      screen->chip_family = CHIP_FAMILY_RV740;
      screen->chip_flags = RADEON_CHIPSET_TCL;
      break;

    case PCI_CHIP_CEDAR_68E0:
    case PCI_CHIP_CEDAR_68E1:
    case PCI_CHIP_CEDAR_68E4:
    case PCI_CHIP_CEDAR_68E5:
    case PCI_CHIP_CEDAR_68E8:
    case PCI_CHIP_CEDAR_68E9:
    case PCI_CHIP_CEDAR_68F1:
    case PCI_CHIP_CEDAR_68F2:
    case PCI_CHIP_CEDAR_68F8:
    case PCI_CHIP_CEDAR_68F9:
    case PCI_CHIP_CEDAR_68FE:
       screen->chip_family = CHIP_FAMILY_CEDAR;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_REDWOOD_68C0:
    case PCI_CHIP_REDWOOD_68C1:
    case PCI_CHIP_REDWOOD_68C8:
    case PCI_CHIP_REDWOOD_68C9:
    case PCI_CHIP_REDWOOD_68D8:
    case PCI_CHIP_REDWOOD_68D9:
    case PCI_CHIP_REDWOOD_68DA:
    case PCI_CHIP_REDWOOD_68DE:
       screen->chip_family = CHIP_FAMILY_REDWOOD;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_JUNIPER_68A0:
    case PCI_CHIP_JUNIPER_68A1:
    case PCI_CHIP_JUNIPER_68A8:
    case PCI_CHIP_JUNIPER_68A9:
    case PCI_CHIP_JUNIPER_68B0:
    case PCI_CHIP_JUNIPER_68B8:
    case PCI_CHIP_JUNIPER_68B9:
    case PCI_CHIP_JUNIPER_68BA:
    case PCI_CHIP_JUNIPER_68BE:
    case PCI_CHIP_JUNIPER_68BF:
       screen->chip_family = CHIP_FAMILY_JUNIPER;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_CYPRESS_6880:
    case PCI_CHIP_CYPRESS_6888:
    case PCI_CHIP_CYPRESS_6889:
    case PCI_CHIP_CYPRESS_688A:
    case PCI_CHIP_CYPRESS_6898:
    case PCI_CHIP_CYPRESS_6899:
    case PCI_CHIP_CYPRESS_689B:
    case PCI_CHIP_CYPRESS_689E:
       screen->chip_family = CHIP_FAMILY_CYPRESS;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_HEMLOCK_689C:
    case PCI_CHIP_HEMLOCK_689D:
       screen->chip_family = CHIP_FAMILY_HEMLOCK;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_PALM_9802:
    case PCI_CHIP_PALM_9803:
    case PCI_CHIP_PALM_9804:
    case PCI_CHIP_PALM_9805:
    case PCI_CHIP_PALM_9806:
    case PCI_CHIP_PALM_9807:
       screen->chip_family = CHIP_FAMILY_PALM;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_SUMO_9640:
    case PCI_CHIP_SUMO_9641:
    case PCI_CHIP_SUMO_9647:
    case PCI_CHIP_SUMO_9648:
    case PCI_CHIP_SUMO_964A:
    case PCI_CHIP_SUMO_964E:
    case PCI_CHIP_SUMO_964F:
       screen->chip_family = CHIP_FAMILY_SUMO;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

    case PCI_CHIP_SUMO2_9642:
    case PCI_CHIP_SUMO2_9643:
    case PCI_CHIP_SUMO2_9644:
    case PCI_CHIP_SUMO2_9645:
       screen->chip_family = CHIP_FAMILY_SUMO2;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

   case PCI_CHIP_BARTS_6720:
   case PCI_CHIP_BARTS_6721:
   case PCI_CHIP_BARTS_6722:
   case PCI_CHIP_BARTS_6723:
   case PCI_CHIP_BARTS_6724:
   case PCI_CHIP_BARTS_6725:
   case PCI_CHIP_BARTS_6726:
   case PCI_CHIP_BARTS_6727:
   case PCI_CHIP_BARTS_6728:
   case PCI_CHIP_BARTS_6729:
   case PCI_CHIP_BARTS_6738:
   case PCI_CHIP_BARTS_6739:
   case PCI_CHIP_BARTS_673E:
       screen->chip_family = CHIP_FAMILY_BARTS;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

   case PCI_CHIP_TURKS_6740:
   case PCI_CHIP_TURKS_6741:
   case PCI_CHIP_TURKS_6742:
   case PCI_CHIP_TURKS_6743:
   case PCI_CHIP_TURKS_6744:
   case PCI_CHIP_TURKS_6745:
   case PCI_CHIP_TURKS_6746:
   case PCI_CHIP_TURKS_6747:
   case PCI_CHIP_TURKS_6748:
   case PCI_CHIP_TURKS_6749:
   case PCI_CHIP_TURKS_6750:
   case PCI_CHIP_TURKS_6758:
   case PCI_CHIP_TURKS_6759:
   case PCI_CHIP_TURKS_675F:
       screen->chip_family = CHIP_FAMILY_TURKS;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

   case PCI_CHIP_CAICOS_6760:
   case PCI_CHIP_CAICOS_6761:
   case PCI_CHIP_CAICOS_6762:
   case PCI_CHIP_CAICOS_6763:
   case PCI_CHIP_CAICOS_6764:
   case PCI_CHIP_CAICOS_6765:
   case PCI_CHIP_CAICOS_6766:
   case PCI_CHIP_CAICOS_6767:
   case PCI_CHIP_CAICOS_6768:
   case PCI_CHIP_CAICOS_6770:
   case PCI_CHIP_CAICOS_6778:
   case PCI_CHIP_CAICOS_6779:
       screen->chip_family = CHIP_FAMILY_CAICOS;
       screen->chip_flags = RADEON_CHIPSET_TCL;
       break;

   default:
      fprintf(stderr, "unknown chip id 0x%x, can't guess.\n",
	      device_id);
      return -1;
   }

   return 0;
}

static radeonScreenPtr
radeonCreateScreen2(__DRIscreen *sPriv)
{
   radeonScreenPtr screen;
   int i;
   int ret;
   uint32_t device_id = 0;

   /* Allocate the private area */
   screen = (radeonScreenPtr) CALLOC( sizeof(*screen) );
   if ( !screen ) {
      fprintf(stderr, "%s: Could not allocate memory for screen structure", __FUNCTION__);
      fprintf(stderr, "leaving here\n");
      return NULL;
   }

   radeon_init_debug();

   /* parse information in __driConfigOptions */
   driParseOptionInfo (&screen->optionCache,
		       __driConfigOptions, __driNConfigOptions);

   screen->chip_flags = 0;

   screen->irq = 1;

   ret = radeonGetParam(sPriv, RADEON_PARAM_DEVICE_ID, &device_id);
   if (ret) {
     FREE( screen );
     fprintf(stderr, "drm_radeon_getparam_t (RADEON_PARAM_DEVICE_ID): %d\n", ret);
     return NULL;
   }

   ret = radeon_set_screen_flags(screen, device_id);
   if (ret == -1)
     return NULL;

   if (getenv("R300_NO_TCL"))
	   screen->chip_flags &= ~RADEON_CHIPSET_TCL;

   if (screen->chip_family <= CHIP_FAMILY_RS200)
	   screen->chip_flags |= RADEON_CLASS_R100;
   else if (screen->chip_family <= CHIP_FAMILY_RV280)
	   screen->chip_flags |= RADEON_CLASS_R200;

   i = 0;
   screen->extensions[i++] = &dri2ConfigQueryExtension.base;

#if defined(RADEON_R100)
   screen->extensions[i++] = &radeonTexBufferExtension.base;
#endif

#if defined(RADEON_R200)
   screen->extensions[i++] = &r200TexBufferExtension.base;
#endif

   screen->extensions[i++] = &radeonFlushExtension.base;
   screen->extensions[i++] = &radeonImageExtension.base;

   screen->extensions[i++] = NULL;
   sPriv->extensions = screen->extensions;

   screen->driScreen = sPriv;
   screen->bom = radeon_bo_manager_gem_ctor(sPriv->fd);
   if (screen->bom == NULL) {
       free(screen);
       return NULL;
   }
   return screen;
}

/* Destroy the device specific screen private data struct.
 */
static void
radeonDestroyScreen( __DRIscreen *sPriv )
{
    radeonScreenPtr screen = (radeonScreenPtr)sPriv->driverPrivate;

    if (!screen)
        return;

#ifdef RADEON_BO_TRACK
    radeon_tracker_print(&screen->bom->tracker, stderr);
#endif
    radeon_bo_manager_gem_dtor(screen->bom);

    /* free all option information */
    driDestroyOptionInfo (&screen->optionCache);

    FREE( screen );
    sPriv->driverPrivate = NULL;
}


/* Initialize the driver specific screen private data.
 */
static GLboolean
radeonInitDriver( __DRIscreen *sPriv )
{
    sPriv->driverPrivate = (void *) radeonCreateScreen2( sPriv );
    if ( !sPriv->driverPrivate ) {
        radeonDestroyScreen( sPriv );
        return GL_FALSE;
    }

    return GL_TRUE;
}



/**
 * Create the Mesa framebuffer and renderbuffers for a given window/drawable.
 *
 * \todo This function (and its interface) will need to be updated to support
 * pbuffers.
 */
static GLboolean
radeonCreateBuffer( __DRIscreen *driScrnPriv,
                    __DRIdrawable *driDrawPriv,
                    const struct gl_config *mesaVis,
                    GLboolean isPixmap )
{
    radeonScreenPtr screen = (radeonScreenPtr) driScrnPriv->driverPrivate;

    const GLboolean swDepth = GL_FALSE;
    const GLboolean swAlpha = GL_FALSE;
    const GLboolean swAccum = mesaVis->accumRedBits > 0;
    const GLboolean swStencil = mesaVis->stencilBits > 0 &&
	mesaVis->depthBits != 24;
    gl_format rgbFormat;
    struct radeon_framebuffer *rfb;

    if (isPixmap)
      return GL_FALSE; /* not implemented */

    rfb = CALLOC_STRUCT(radeon_framebuffer);
    if (!rfb)
      return GL_FALSE;

    _mesa_initialize_window_framebuffer(&rfb->base, mesaVis);

    if (mesaVis->redBits == 5)
        rgbFormat = _mesa_little_endian() ? MESA_FORMAT_RGB565 : MESA_FORMAT_RGB565_REV;
    else if (mesaVis->alphaBits == 0)
        rgbFormat = _mesa_little_endian() ? MESA_FORMAT_XRGB8888 : MESA_FORMAT_XRGB8888_REV;
    else
        rgbFormat = _mesa_little_endian() ? MESA_FORMAT_ARGB8888 : MESA_FORMAT_ARGB8888_REV;

    /* front color renderbuffer */
    rfb->color_rb[0] = radeon_create_renderbuffer(rgbFormat, driDrawPriv);
    _mesa_add_renderbuffer(&rfb->base, BUFFER_FRONT_LEFT, &rfb->color_rb[0]->base);
    rfb->color_rb[0]->has_surface = 1;

    /* back color renderbuffer */
    if (mesaVis->doubleBufferMode) {
      rfb->color_rb[1] = radeon_create_renderbuffer(rgbFormat, driDrawPriv);
	_mesa_add_renderbuffer(&rfb->base, BUFFER_BACK_LEFT, &rfb->color_rb[1]->base);
	rfb->color_rb[1]->has_surface = 1;
    }

    if (mesaVis->depthBits == 24) {
      if (mesaVis->stencilBits == 8) {
	struct radeon_renderbuffer *depthStencilRb =
           radeon_create_renderbuffer(MESA_FORMAT_S8_Z24, driDrawPriv);
	_mesa_add_renderbuffer(&rfb->base, BUFFER_DEPTH, &depthStencilRb->base);
	_mesa_add_renderbuffer(&rfb->base, BUFFER_STENCIL, &depthStencilRb->base);
	depthStencilRb->has_surface = screen->depthHasSurface;
      } else {
	/* depth renderbuffer */
	struct radeon_renderbuffer *depth =
           radeon_create_renderbuffer(MESA_FORMAT_X8_Z24, driDrawPriv);
	_mesa_add_renderbuffer(&rfb->base, BUFFER_DEPTH, &depth->base);
	depth->has_surface = screen->depthHasSurface;
      }
    } else if (mesaVis->depthBits == 16) {
        /* just 16-bit depth buffer, no hw stencil */
	struct radeon_renderbuffer *depth =
           radeon_create_renderbuffer(MESA_FORMAT_Z16, driDrawPriv);
	_mesa_add_renderbuffer(&rfb->base, BUFFER_DEPTH, &depth->base);
	depth->has_surface = screen->depthHasSurface;
    }

    _mesa_add_soft_renderbuffers(&rfb->base,
	    GL_FALSE, /* color */
	    swDepth,
	    swStencil,
	    swAccum,
	    swAlpha,
	    GL_FALSE /* aux */);
    driDrawPriv->driverPrivate = (void *) rfb;

    return (driDrawPriv->driverPrivate != NULL);
}


static void radeon_cleanup_renderbuffers(struct radeon_framebuffer *rfb)
{
	struct radeon_renderbuffer *rb;

	rb = rfb->color_rb[0];
	if (rb && rb->bo) {
		radeon_bo_unref(rb->bo);
		rb->bo = NULL;
	}
	rb = rfb->color_rb[1];
	if (rb && rb->bo) {
		radeon_bo_unref(rb->bo);
		rb->bo = NULL;
	}
	rb = radeon_get_renderbuffer(&rfb->base, BUFFER_DEPTH);
	if (rb && rb->bo) {
		radeon_bo_unref(rb->bo);
		rb->bo = NULL;
	}
}

void
radeonDestroyBuffer(__DRIdrawable *driDrawPriv)
{
    struct radeon_framebuffer *rfb;
    if (!driDrawPriv)
	return;

    rfb = (void*)driDrawPriv->driverPrivate;
    if (!rfb)
	return;
    radeon_cleanup_renderbuffers(rfb);
    _mesa_reference_framebuffer((struct gl_framebuffer **)(&(driDrawPriv->driverPrivate)), NULL);
}

#define ARRAY_SIZE(a) (sizeof (a) / sizeof ((a)[0]))

/**
 * This is the driver specific part of the createNewScreen entry point.
 * Called when using DRI2.
 *
 * \return the struct gl_config supported by this driver
 */
static const
__DRIconfig **radeonInitScreen2(__DRIscreen *psp)
{
   GLenum fb_format[3];
   GLenum fb_type[3];
   /* GLX_SWAP_COPY_OML is only supported because the Intel driver doesn't
    * support pageflipping at all.
    */
   static const GLenum back_buffer_modes[] = {
     GLX_NONE, GLX_SWAP_UNDEFINED_OML, /*, GLX_SWAP_COPY_OML*/
   };
   uint8_t depth_bits[4], stencil_bits[4], msaa_samples_array[1];
   int color;
   __DRIconfig **configs = NULL;

   if (!radeonInitDriver(psp)) {
       return NULL;
    }
   depth_bits[0] = 0;
   stencil_bits[0] = 0;
   depth_bits[1] = 16;
   stencil_bits[1] = 0;
   depth_bits[2] = 24;
   stencil_bits[2] = 0;
   depth_bits[3] = 24;
   stencil_bits[3] = 8;

   msaa_samples_array[0] = 0;

   fb_format[0] = GL_RGB;
   fb_type[0] = GL_UNSIGNED_SHORT_5_6_5;

   fb_format[1] = GL_BGR;
   fb_type[1] = GL_UNSIGNED_INT_8_8_8_8_REV;

   fb_format[2] = GL_BGRA;
   fb_type[2] = GL_UNSIGNED_INT_8_8_8_8_REV;

   for (color = 0; color < ARRAY_SIZE(fb_format); color++) {
      __DRIconfig **new_configs;

      new_configs = driCreateConfigs(fb_format[color], fb_type[color],
				     depth_bits,
				     stencil_bits,
				     ARRAY_SIZE(depth_bits),
				     back_buffer_modes,
				     ARRAY_SIZE(back_buffer_modes),
				     msaa_samples_array,
				     ARRAY_SIZE(msaa_samples_array),
				     GL_TRUE);
      if (configs == NULL)
	 configs = new_configs;
      else
	 configs = driConcatConfigs(configs, new_configs);
   }

   if (configs == NULL) {
      fprintf(stderr, "[%s:%u] Error creating FBConfig!\n", __func__,
              __LINE__);
      return NULL;
   }

   return (const __DRIconfig **)configs;
}

const struct __DriverAPIRec driDriverAPI = {
   .InitScreen      = radeonInitScreen2,
   .DestroyScreen   = radeonDestroyScreen,
#if defined(RADEON_R200)
   .CreateContext   = r200CreateContext,
   .DestroyContext  = r200DestroyContext,
#else
   .CreateContext   = r100CreateContext,
   .DestroyContext  = radeonDestroyContext,
#endif
   .CreateBuffer    = radeonCreateBuffer,
   .DestroyBuffer   = radeonDestroyBuffer,
   .MakeCurrent     = radeonMakeCurrent,
   .UnbindContext   = radeonUnbindContext,
};

/* This is the table of extensions that the loader will dlsym() for. */
PUBLIC const __DRIextension *__driDriverExtensions[] = {
    &driCoreExtension.base,
    &driDRI2Extension.base,
    NULL
};
