# -*- coding: utf-8 -*-

# database のアクセス用スクリプト
# 2013/12/04 written by kei9 

import sqlite3
import sys

import db_supports
from amuletskillsholder import AmuletSkillsHolder
from amuletidskillidholder import AmuletIdSkillIdHolder
from seedsholder import SeedsHolder
from sufficientsholder import SufficientsHolder

class DataBaseAccessor(object):
    u""" this is access class to database """
    def __init__(self, db_name):
        u""" db_name is database name to access """
        self._db_name = db_name
        self._connect = sqlite3.connect(self._db_name)
        self._cursor = self._connect.cursor()

        self._skill_id2name_dict, self._skill_name2id_dict = self._get_skill_dict()
        (self._amulet_id2name_dict, self._amulet_name2id_dict, 
            self._amulet_id2minmax_table_dict, self._amulet_id2skill2_table_dict,
            self._amulet_id2sufficient_table_dict,
            self._amulet_id2skill1_id_dict, self._amulet_id2skill2_id_dict) = self._get_amulet_dict()

    def _print_dicts(self):
        u""" for debug """
        print self._skill_id2name_dict
        print self._skill_name2id_dict
        print self._amulet_id2name_dict
        print self._amulet_name2id_dict
        print self._amulet_id2minmax_table_dict
        print self._amulet_id2skill2_table_dict
        print self._amulet_id2sufficient_table_dict
        print self._amulet_id2skill1_id_dict
        print self._amulet_id2skill2_id_dict

    def get_dicts(self, clone_enable=False):
        u""" get all dicts 
            if clone_enable is True, return cloend dictionary,
            otherwise return reference dictionary

            returns (skill_id2name_dict, skill_name2id_dict, 
                amulet_id2name_dict, amulet_name2id_dict,
                amulet_id2skill_ids)
        """
        if clone_enable:
            return (dict(self._skill_id2name_dict), 
                    dict(self._skill_name2id_dict),
                    dict(self._amulet_id2name_dict), 
                    dict(self._amulet_name2id_dict),
                    dict(self._amulet_id2skill1_id_dict),
                    dict(self._amulet_id2skill2_id_dict))
        else:
            return (self._skill_id2name_dict, 
                    self._skill_name2id_dict,
                    self._amulet_id2name_dict, 
                    self._amulet_name2id_dict,
                    self._amulet_id2skill1_id_dict,
                    self._amulet_id2skill2_id_dict)


    def _get_skill_dict(self):
        u""" create id2name, name2id dict of skill 
            return (dict_id2name, dict_name2id)
        """
        skill_id2name_dict, skill_name2id_dict = {}, {}
        self._cursor.execute(db_supports.SKILL_TABLE_SELECT_ALL_SQL)
        for val in self._cursor.fetchall():
            skill_id, name = val[0], val[1]
            skill_id2name_dict[skill_id] = name
            skill_name2id_dict[name] = skill_id

        return skill_id2name_dict, skill_name2id_dict

    def _get_amulet_dict(self):
        u""" create id2name, name2id, id2minmax_table, id2skill2_table, id2sufficient_table,
            id2skill1_ids, id2skill2_id dict of amulet 
        return (dict_id2name, dict_name2id, 
            id2minmax_table, id2skill2_table, id2sufficient_table, id2skill1_ids, id2skill2_ids)
        """
        amulet_id2name_dict, amulet_name2id_dict = {}, {}
        id2minmax_table, id2skill2_table, id2sufficient_table = {}, {}, {}
        id2skill1_ids, id2skill2_ids = {}, {}
        self._cursor.execute(db_supports.AMULET_TABLE_SELECT_ALL_SQL)
        for val in self._cursor.fetchall():
            amulet_id, name = val[0], val[1]
            amulet_id2name_dict[amulet_id] = name
            amulet_name2id_dict[name] = amulet_id

        # id 2 minmax table name
        self._cursor.execute(db_supports.MIN_MAX_MASTER_TABLE_SELECT_ALL_SQL)
        for val in self._cursor.fetchall():
            amulet_id, table = val[0], val[1]
            id2minmax_table[amulet_id] = table
 
        # id 2 skill2 table name
        self._cursor.execute(db_supports.SEED2_SKILL2_MASTER_TABLE_SELECT_ALL_SQL)
        for val in self._cursor.fetchall():
            amulet_id, table = val[0], val[1]
            id2skill2_table[amulet_id] = table

        # id 2 sufficient table name
        self._cursor.execute(db_supports.SUFFICIENT_MASTER_TABLE_SELECT_ALL_SQL)
        for val in self._cursor.fetchall():
            amulet_id, table = val[0], val[1]
            id2sufficient_table[amulet_id] = table

        # id 2 non zero skill1 ids
        # id 2 non zero skill2 ids
        for amu_id in amulet_id2name_dict.keys():
            table = id2minmax_table[amu_id]
            sql = db_supports.MIN_MAX_TABLE_SELECT_NON_ZERO_SKILL1_ID_SQL.format(table_name=table)
            self._cursor.execute(sql)
            ls = [x[0] for x in self._cursor.fetchall()]
            id2skill1_ids[amu_id] = tuple(ls)

            sql = db_supports.MIN_MAX_TABLE_SELECT_NON_ZERO_SKILL2_ID_SQL.format(table_name=table)
            self._cursor.execute(sql)
            ls = [x[0] for x in self._cursor.fetchall()]
            id2skill2_ids[amu_id] = tuple(ls)

        return (amulet_id2name_dict, amulet_name2id_dict, 
                id2minmax_table, id2skill2_table, id2sufficient_table,
                id2skill1_ids, id2skill2_ids)

    def select_seed2s(self, amu_id2skill_id_list_dict):
        u""" select seed2 from dict of amulet_id to skill2_id_list,
        skill_id_list: list of skill_id, if skill_id is unknown, skill_id must be None
        """
        seed_set = None
        for key, val in amu_id2skill_id_list_dict.items():
            if key in self._amulet_id2skill2_table_dict:
                second_skill_table = self._amulet_id2skill2_table_dict[key]
                amulet_skill = AmuletSkillsHolder(val)

                if amulet_skill.is_empty():
                    continue

                sql = db_supports.SEED2_SKILL2_TABLE_SELECT_SEED_SQL.format(table_name=second_skill_table) + amulet_skill.get_where_and_sql()
                self._cursor.execute(sql)

                if seed_set is None:
                    seed_set = set([x[0] for x in self._cursor.fetchall()])
                else:
                    seed_set = seed_set.intersection(set([x[0] for x in self._cursor.fetchall()]))

        if seed_set is None:
            return set()
        else:
            return seed_set

    def select_seed1s_tenun(self, amu_id_skill1_id_list, key_tenun):
        u""" select seed1 from list of amulet_id and skill1_id,
        key_tenun must be KEY_TENUN555 or KEY_TENUN888
        result_num: result number of amulet
        when either amu_id or skill1_id is None, this means that no. has no result
        """
        result_num = len(amu_id_skill1_id_list)
        for amu_id, skill1_id in reversed(amu_id_skill1_id_list):
            if amu_id is not None and skill1_id is not None:
                break
            else:
                result_num -= 1
        amulet_ids = [x[0] for x in amu_id_skill1_id_list[:result_num]]
        skill1_ids = [x[1] for x in amu_id_skill1_id_list[:result_num]]
        holder = AmuletIdSkillIdHolder(amulet_ids, skill1_ids)

        if key_tenun == db_supports.KEY_TENUN555:
            sql = db_supports.SEED1_TENUN555_TABLE_SELECT_SQL + holder.get_where_and_sql()
        elif key_tenun == db_supports.KEY_TENUN888:
            sql = db_supports.SEED1_TENUN888_TABLE_SELECT_SQL + holder.get_where_and_sql()
        else:
            raise NotImplementedError(u"Tenun must be 555 or 888")
        self._cursor.execute(sql)
        seed_set = set([x[0] for x in self._cursor.fetchall()])
        return seed_set

    def select_seed2s_from_sufficient_val(self, amu_id, sufficient_val, slot_num, skill2_id):
        u""" get seeds set from slot number, skill2_id, amulet id, sufficient value
        if skill2_id is None, search all skills
        return (threshold, threshold1_seed2_set, threshold2_seed2_set, inishie_seed2_set)
        """
        th1_seeds_set = set()
        th2_seeds_set = set()
        ini_seeds_set = set()
        if amu_id in self._amulet_id2name_dict.keys():
            table = self._amulet_id2sufficient_table_dict[amu_id]
            # threshold
            sql = db_supports.SUFFICIENT_TABLE_SELECT_THRESHOLD_SQL.format(
                    table_name=table, sufficient_val=sufficient_val)
            self._cursor.execute(sql)
            try:
                threshold = self._cursor.fetchone()[slot_num-1]
            except TypeError, e:
                threshold = None

            if threshold is not None:
                table = self._amulet_id2skill2_table_dict[amu_id]
                if skill2_id is not None:
                    sql1 = db_supports.SEED2_THRESHOLD1_TABLE_SELECT_SEEDS_FROM_THRESHOLD1_AND_SKILL2_SQL.format(
                        threshold1=threshold,
                        skill_table=table,
                        skill_id=skill2_id)
                    sql2 = db_supports.SEED2_THRESHOLD2_TABLE_SELECT_SEEDS_FROM_THRESHOLD2_AND_SKILL2_SQL.format(
                        threshold2=threshold,
                        skill_table=table,
                        skill_id=skill2_id)
                    sql3 = db_supports.SEED2_INISHIE_TABLE_SELECT_SEED2_FROM_SKILL2_THRESHOLD_SQL.format(
                        threshold=threshold,
                        skill2_id=skill2_id)
                else:
                    sql1 = db_supports.SEED2_THRESHOLD1_TABLE_SELECT_SEEDS_FROM_THRESHOLD1_SQL.format(
                        threshold1=threshold)
                    sql2 = db_supports.SEED2_THRESHOLD2_TABLE_SELECT_SEEDS_FROM_THRESHOLD2_SQL.format(
                        threshold2=threshold)
                    sql3 = db_supports.SEED2_INISHIE_TABLE_SELECT_SEED2_FROM_THRESHOLD_SQL.format(
                        threshold=threshold)
                self._cursor.execute(sql1)
                th1_seeds_set = set([x[0] for x in self._cursor.fetchall()])
                self._cursor.execute(sql2)
                th2_seeds_set = set([x[0] for x in self._cursor.fetchall()])

                if amu_id == self._amulet_name2id_dict[db_supports.INISHIE_AMULET]:
                    self._cursor.execute(sql3)
                    ini_seeds_set = set([x[0] for x in self._cursor.fetchall()])

        return (threshold, th1_seeds_set, th2_seeds_set, ini_seeds_set)

    def select_minmax_from_skill_ids(self, amulet_id, skill_ids):
        u""" お守りのidとスキルidのリストから、スキルIDと取りうるスキルの最大最小値の辞書を得る
            return dict{skill_id:(min1, max1, min2, max2)}
        """
        minmax_table = self._amulet_id2minmax_table_dict[amulet_id]
        amulet_skill = AmuletSkillsHolder(skill_ids)
        result_dict = {}
        if amulet_skill.is_empty() is False:
            sql = db_supports.MIN_MAX_TABLE_SELECT_ALL_SQL.format(table_name=minmax_table) + amulet_skill.get_where_in_sql()
            self._cursor.execute(sql)

            for row in self._cursor.fetchall():
                result_dict[row[0]] = tuple(row[1:])
        return result_dict

    def select_threshold_from_sufficient(self, amulet_id, sufficient_vals):
        u""" お守りのidと充足値のリストから、充足値とスロットごとの判定値の辞書を得る
            return dict({sufficient_val:(slot1_th, slot2_th, slot3_th)})
        """
        suff_table = self._amulet_idsufficient_table_dict[amulet_id]
        amulet_skill = SufficientsHolder(sufficient_vals)
        result_dict = {}
        if amulet_skill.is_empty() is False:
            sql = db_supports.SUFFICIENT_TABLE_SELECT_ALL_SQL.format(table_name=suff_table) + amulet_skill.get_where_in_sql()
            self._cursor.execute(sql)

            for row in self._cursor.fetchall():
                result_dict[row[0]] = tuple(row[1])
        return result_dict

    def select_skill2s_from_seed2s(self, seed_set):
        u""" get amulet id to (random_seed2 to skill ids dict) dict,
        and random_seed2 to threshold1 dict
        and random_seed2 to threshold2 dict
        return: skill_dict, threshold1_dict, threshold2_dict
        """
        skill_dict, threshold1_dict, threshold2_dict = {}, {}, {}

        # skill dict
        seeds = SeedsHolder(seed_set)
        for amu_id, table in self._amulet_id2skill2_table_dict.items():
            sql = db_supports.SEED2_SKILL2_TABLE_SELECT_ALL_SQL.format(
                    table_name=table) + seeds.get_where_in_sql()
            self._cursor.execute(sql)
            seed2skills_dic = {}
            for row in self._cursor.fetchall():
                seed2skills_dic[row[0]] = [x for x in row[1:]]
            skill_dict[amu_id] = seed2skills_dic

        # threshold1 dict
        sql = db_supports.SEED2_THRESHOLD1_TABLE_SELECT_ALL_SQL + seeds.get_where_in_sql()
        self._cursor.execute(sql)
        for row in self._cursor.fetchall():
            threshold1_dict[row[0]] = [x for x in row[1:]]

        # threshold2 dict
        sql = db_supports.SEED2_THRESHOLD2_TABLE_SELECT_ALL_SQL + seeds.get_where_in_sql()
        self._cursor.execute(sql)
        for row in self._cursor.fetchall():
            threshold2_dict[row[0]] = [x for x in row[1:]]

        return skill_dict, threshold1_dict, threshold2_dict

    def select_inishie_skill2_from_seed2(self, seed2_set):
        u""" get inishie's dict of (skill2_id, threshold1, threshold2) from seed2 value """
        result_dict = {}
        for seed2 in seed2_set:
            sql = db_supports.SEED2_INISHIE_TABLE_SELECT_FROM_SEED2_SQL.format(seed2=seed2)
            self._cursor.execute(sql)
            row = self._cursor.fetchone()
            result_dict[seed2] = (row[0], row[1], row[2])
        return result_dict

    def select_table_nos_from_seed2(self, seed2_set):
        u""" get dict of (table_no, no) from seed2 value """
        result_dict = {}
        for seed2 in seed2_set:
            sql = db_supports.SEED2_TABLE_NO_TABLE_SELECT_FROM_SEED2_SQL.format(seed2=seed2)
            self._cursor.execute(sql)
            row = self._cursor.fetchone()
            result_dict[seed2] = (row[0], row[1])
        return result_dict

    def select_table_nos_from_seed1(self, seed1_set, key_tenun):
        u""" get dict of (table_no, no) from seed1 value based on key_tenun"""
        result_dict = {}
        if key_tenun == db_supports.KEY_TENUN555:
            sql_format = db_supports.SEED1_TENUN555_TABLE_SELECT_TABLE_NOS_FROM_SEED1_SQL
        elif key_tenun == db_supports.KEY_TENUN888:
            sql_format = db_supports.SEED1_TENUN888_TABLE_SELECT_TABLE_NOS_FROM_SEED1_SQL
        else:
            raise NotImplementedError(u"Tenun must be 555 or 888")

        for seed1 in seed1_set:
            sql = sql_format.format(seed1=seed1)
            self._cursor.execute(sql)
            row = self._cursor.fetchone()
            result_dict[seed1] = (row[0], row[1])
        return result_dict

    def select_near_seed1s_from_seed1(self, seed1, smaller_num, larger_num, key_tenun):
        u""" get list of (no, seed1) near arg's seed1 
        smaller_num means numbers to get smaller than arg's seed1
        larger_num means numbers to get larger than arg's seed1"""
        result_list = []
        if key_tenun == db_supports.KEY_TENUN555:
            sql_format = db_supports.SEED1_TENUN555_TABLE_SELECT_NEAR_SEED1S_FROM_NO_SQL
        elif key_tenun == db_supports.KEY_TENUN888:
            sql_format = db_supports.SEED1_TENUN888_TABLE_SELECT_NEAR_SEED1S_FROM_NO_SQL
        else:
            raise NotImplementedError(u"Tenun must be 555 or 888")
        table_no, no = self.select_table_nos_from_seed1([seed1], key_tenun)[seed1]
        sql = sql_format.format(table_no=table_no,min_no=(no-smaller_num),max_no=(no+larger_num))
        self._cursor.execute(sql)
        for row in self._cursor.fetchall():
            result_list.append((row[0], row[1]))
        return result_list

    def select_all_from_seed1s(self, seed1_set, key_tenun):
        u""" seed1に対応する(no, table_no, result_num, 
        (amulet_id1, ..., amulet_id7), (skill1_id1, ..., skill1_id7))
        の辞書を得る。存在しない場所はNoneで置き換えてある。
        """
        result_dict = {}
        if key_tenun == db_supports.KEY_TENUN555:
            sql_format = db_supports.SEED1_TENUN555_TABLE_SELECT_ALL_FROM_SEED1_SQL
        elif key_tenun == db_supports.KEY_TENUN888:
            sql_format = db_supports.SEED1_TENUN888_TABLE_SELECT_ALL_FROM_SEED1_SQL
        else:
            raise NotImplementedError(u"Tenun must be 555 or 888")

        for seed1 in seed1_set:
            sql = sql_format.format(seed1=seed1)
            self._cursor.execute(sql)
            row = self._cursor.fetchone()
            if key_tenun == db_supports.KEY_TENUN555:
                amu_ids = [None if x == db_supports.NO_DATA else x for x in row[3:9]]
                skill_ids = [None if x == db_supports.NO_DATA else x for x in row[9:15]]
            elif key_tenun == db_supports.KEY_TENUN888:
                amu_ids = [None if x == db_supports.NO_DATA else x for x in row[3:10]]
                skill_ids = [None if x == db_supports.NO_DATA else x for x in row[10:17]]
            else:
                raise NotImplementedError(u"Tenun must be 555 or 888")
            result_dict[seed1] = (row[0], row[1], row[2], amu_ids, skill_ids)
        return result_dict

    def get_sufficient_value(self, amu_id, skill1_id, skill2_id, skill1_val, skill2_val):
        u""" 充足値を計算する。
        指定したスキルが見つからない場合はNoneを返す 
        skill2_idがNoneのときはskill1のみから計算する
        return (sufficient_val, max1, max2)
        """
        if amu_id in self._amulet_id2minmax_table_dict:
            table = self._amulet_id2minmax_table_dict[amu_id]
            sql = db_supports.MIN_MAX_TABLE_SELECT_MAX1_OF_SKILL_SQL.format(
                    table_name=table,skill_id=skill1_id)
            self._cursor.execute(sql)
            max1 = self._cursor.fetchone()[0]
            if skill2_id is not None:
                sql = db_supports.MIN_MAX_TABLE_SELECT_MAX2_OF_SKILL_SQL.format(
                        table_name=table,skill_id=skill2_id)
                self._cursor.execute(sql)
                max2 = self._cursor.fetchone()[0]
            else:
                max2 = 1

            if max1 is None or max2 is None:
                return None

            #print skill1_val, "/", max1 , skill2_val, "/", max2
            try:
                skill1_val = max1 if skill1_val > max1 else skill1_val
                if skill2_id is None:
                    skill2_val = 0
                else:
                    skill2_val = max2 if skill2_val > max2 else skill2_val
                val1 = (10 * skill1_val) // max1 if skill1_val >= 0 else 0
                val2 = (10 * skill2_val) // max2 if skill2_val >= 0 else 0
            except TypeError, e:
                return None
            except ZeroDivisionError, e:
                return None
            else:
                return  (val1 + val2, max1, max2)

        else:
            return None

    def get_skill_minmax(self):
        u""" 全てのお守りのスキルにおいて、その最大値と最小値を返す 
            return (min1, max1, min2, max2)
        """
        min1_ls, max1_ls, min2_ls, max2_ls = [],[],[],[]
        # get min max of each amulet
        for amu_id, table in self._amulet_id2minmax_table_dict.items():
            sql = db_supports.MIN_MAX_TABLE_SELECT_MIN1_SQL.format(table_name=table)
            self._cursor.execute(sql)
            min1_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MAX1_SQL.format(table_name=table)
            self._cursor.execute(sql)
            max1_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MIN2_SQL.format(table_name=table)
            self._cursor.execute(sql)
            min2_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MAX2_SQL.format(table_name=table)
            self._cursor.execute(sql)
            max2_ls.append(self._cursor.fetchone()[0])

        return (min(min1_ls), max(max1_ls), min(min2_ls), max(max2_ls))

    def get_skill_ids(self, amulet_id):
        u""" 全てのお守りのスキルにおいて、その最大値と最小値を返す 
            return (min1, max1, min2, max2)
        """
        min1_ls, max1_ls, min2_ls, max2_ls = [],[],[],[]
        # get min max of each amulet
        for amu_id, table in self._amulet_id2minmax_table_dict.items():
            sql = db_supports.MIN_MAX_TABLE_SELECT_MIN1_SQL.format(table_name=table)
            self._cursor.execute(sql)
            min1_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MAX1_SQL.format(table_name=table)
            self._cursor.execute(sql)
            max1_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MIN2_SQL.format(table_name=table)
            self._cursor.execute(sql)
            min2_ls.append(self._cursor.fetchone()[0])

            sql = db_supports.MIN_MAX_TABLE_SELECT_MAX2_SQL.format(table_name=table)
            self._cursor.execute(sql)
            max2_ls.append(self._cursor.fetchone()[0])

        return (min(min1_ls), max(max1_ls), min(min2_ls), max(max2_ls))

    def close(self):
        u""" close database accessor """
        self._cursor.close()
        self._connect.close()

if __name__ == "__main__":
    db = DataBaseAccessor("test.sqlite3")
    #db._print_dicts()
    hikari = [None, 57, None, None, None, None, None]
    huru = [54, None, None, None, None, None, None]
    #yuga = [None, None, 98, 75, None, None, None]
    yuga = [None, None, 98, None, None, None, None]
    dic = {2:hikari, 3:huru, 4:yuga}
    #seeds = db.select_seeds(dic)
    seeds = set([58241, 176])
    skill_dic, slot_dic = db.select_skills_from_seeds(seeds)

    print "seeds: ", seeds
    print "amu_id to seed2skill dict: ", skill_dic
    print "seed2slot dict", slot_dic
    db.close()
