package jp.sf.amateras.functions;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

public class FunctionsTest extends TestCase {

	public void testH() {
		assertEquals("", Functions.h(null));
		assertEquals("&lt;a href=&#034;&#034;&gt;", Functions.h("<a href=\"\">"));
		assertEquals("[true, false]", Functions.h(new boolean[]{true, false}));
		assertEquals("[1, 2, 3]", Functions.h(new int[]{1, 2, 3}));
		assertEquals("[1, 2, 3]", Functions.h(new short[]{1, 2, 3}));
		assertEquals("[1, 2, 3]", Functions.h(new long[]{1, 2, 3}));
		assertEquals("[1, 2, 3]", Functions.h(new byte[]{1, 2, 3}));
		assertEquals("[1.0, 2.0, 3.0]", Functions.h(new double[]{1, 2, 3}));
		assertEquals("[1.0, 2.0, 3.0]", Functions.h(new float[]{1, 2, 3}));
		assertEquals("[a, b, c]", Functions.h(new char[]{'a', 'b', 'c'}));
		assertEquals("[a, b, c]", Functions.h(new String[]{"a", "b", "c"}));
	}

	public void testU() {
		assertEquals("", Functions.u(null));
		assertEquals("%E3%81%82%E3%81%84%E3%81%86%E3%81%88%E3%81%8A", Functions.u("あいうえお"));
	}

	public void testBr() {
		assertEquals("", Functions.br(null));
		assertEquals("あいうえお<br />かきくけこ", Functions.br("あいうえお\nかきくけこ"));
	}

	public void testNbsp() {
		assertEquals("", Functions.nbsp(null));
		assertEquals("あいうえお &nbsp;&nbsp;かきくけこ", Functions.nbsp("あいうえお   かきくけこ"));
	}

	public void testMask() {
		assertEquals("", Functions.mask(null));
		assertEquals("********", Functions.mask("password"));
	}

	public void testDate() throws Exception {
		assertEquals("", Functions.date(null));
		assertEquals("2010/02/01", Functions.date(createDate("2010/02/01 00:00:00")));
	}

	public void testDatetime() throws Exception {
		assertEquals("", Functions.datetime(null));
		assertEquals("2010/02/01 12:34:56", Functions.datetime(createDate("2010/02/01 12:34:56")));
	}

	public void testTime() throws Exception {
		assertEquals("", Functions.time(null));
		assertEquals("12:34:56", Functions.time(createDate("2010/02/01 12:34:56")));
	}

	public void testNumber() {
		assertEquals("", Functions.number(null, ""));
		assertEquals("1,000,000", Functions.number(1000000, "#,###"));
	}

	public void testCut() {
		assertEquals("", Functions.cut(null, 1));
		assertEquals("abc", Functions.cut("abcdef", 3));
		assertEquals("abcdef", Functions.cut("abcdef", 7));
	}

	public void testLink() {
		assertEquals("", Functions.link(null));
		assertEquals("<a href=\"http://www.google.co.jp\">http://www.google.co.jp</a>で検索", 
				Functions.link("http://www.google.co.jpで検索"));
	}

	public void testJson() {
		assertEquals("null", Functions.json(null));
		
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("name", "Naoki Takezoe");
		map.put("gender", "male");
		assertEquals("{\"name\":\"Naoki Takezoe\",\"gender\":\"male\"}", Functions.json(map));
	}
	
	private Date createDate(String date) throws ParseException {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		return formatter.parse(date);
	}
	
	public static void testMatches(){
		assertTrue(Functions.matches("abcdef", ".+def$"));
		assertFalse(Functions.matches("abcdef", "bcd"));
		assertFalse(Functions.matches(null, "bcd"));
	}
	
	public static void testReplace(){
		assertEquals("Ruby Programmer", Functions.replace("Java Programmer", "Java", "Ruby"));
		assertEquals("NetBeans NetBeans", Functions.replace("Eclipse Eclipse", "Eclipse", "NetBeans"));
		assertEquals("JSP x.x", Functions.replace("JSP 2.0", "[0-9]\\.[0-9]", "x.x"));
		assertEquals("", Functions.replace(null, "[0-9]\\.[0-9]", "x.x"));
	}
	
	public static void testQuery(){
		assertEquals("", Functions.query("aaaa"));
		
		Map<String, Object> map = new LinkedHashMap<String, Object>();
		map.put("name", "Naoki Takezoe");
		map.put("programming", new Object[]{"Java", "Perl"});
		
		String query = Functions.query(map);
		assertEquals("name=Naoki+Takezoe&programming=Java&programming=Perl", query);
	}
	
	public void testContainsString(){
		assertTrue(Functions.contains("abcdefg", "cd"));
		assertFalse(Functions.contains("abcdefg", "hi"));
		assertFalse(Functions.contains(null, "hi"));
		assertFalse(Functions.contains("abcdefg", null));
	}
	
	public void testContainsCollection(){
		List<String> list = new ArrayList<String>();
		list.add("aaa");
		list.add("bbb");
		
		assertTrue(Functions.contains(list, "aaa"));
		assertTrue(Functions.contains(list, "bbb"));
		assertFalse(Functions.contains(list, "ccc"));
	}

	public void testContainsMap(){
		Map<String, String> map = new HashMap<String, String>();
		map.put("key1", "value1");
		map.put("key2", "value2");
		
		assertTrue(Functions.contains(map, "value1"));
		assertTrue(Functions.contains(map, "value2"));
		assertFalse(Functions.contains(map, "value3"));
	}
	
	public void testJs(){
		assertEquals("", Functions.js(null));
		assertEquals("aa\\'bb", Functions.js("aa'bb"));
		assertEquals("aa\\r\\nbb", Functions.js("aa\r\nbb"));
		assertEquals("aa\\t\\bbb", Functions.js("aa\t\bbb"));
		assertEquals("aa\\\\bbb", Functions.js("aa\\bbb"));
	}
	
	public void testConcat(){
		assertEquals("aaabbb", Functions.concat("aaa", "bbb"));
		assertEquals("aaa", Functions.concat("aaa", null));
		assertEquals("bbb", Functions.concat(null, "bbb"));
	}

}
