package net.y3n20u.aeszip;

import static net.y3n20u.aeszip.CommonValues.DEFAULT_ENCRYPT_STRENGTH_MODE;
import static net.y3n20u.aeszip.CommonValues.ITERATION_COUNT;
import static net.y3n20u.aeszip.CommonValues.LENGTH_AUTHENTICATION_CODE;
import static net.y3n20u.aeszip.CommonValues.LENGTH_PASSWORD_VERIFICATION_VALUE;
import static net.y3n20u.aeszip.CommonValues.MAX_FOUR_BYTE_FIELD_LONG;
import static net.y3n20u.aeszip.CommonValues.MESSAGE_INVALID_METHOD;
import static net.y3n20u.aeszip.CommonValues.MESSAGE_OFFSET_INVALID;
import static net.y3n20u.aeszip.CommonValues.METHOD_AES;
import static net.y3n20u.aeszip.CommonValues.METHOD_DEFLATED;
import static net.y3n20u.aeszip.CommonValues.METHOD_STORED;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.SecureRandom;
import java.text.MessageFormat;
import java.util.Calendar;
import java.util.zip.ZipEntry;

import net.y3n20u.rfc2898.Pbkdf2;
import net.y3n20u.util.ByteHelper;

/**
 * 
 * @author y3n20u@gmail.com
 */
public class AesZipEntry extends ZipEntry {
	
	// extra field header ID: 0x9901 - 2 bytes
	// data size: 0x0007 - 2 bytes
	// integer version number: 0x0002 (AE-2) - 2 bytes
	// vender ID: "AE" - 2 bytes
	// 01 99 07 00 02 00 41 45
	private static final byte[] AES_EXTRA_BYTES = ByteHelper.getBytes("01 99 07 00 02 00 41 45");
	
	private final EncryptionStrengthMode _encryptionStrengthMode;
	private final byte[] _saltValue;
	private final byte[] _encryptionKey;
	private final byte[] _authenticationKey;
	private final byte[] _passwordVerificationValue;
	private int _method = METHOD_AES;
	private long _relativeOffsetOfLocalFileHeader;
	
	
	public AesZipEntry(String name, byte[] password) {
		this(name, DEFAULT_ENCRYPT_STRENGTH_MODE, password);
	}
	
	public AesZipEntry(String name, EncryptionStrengthMode mode, byte[] password) {
		super(name);
		_encryptionStrengthMode = mode;
		_saltValue = AesZipEntry.generateSaltValue(_encryptionStrengthMode.getSaltLength());
		
		// derive keys
		int derivedKeyLen = _encryptionStrengthMode.getKeyLength() + _encryptionStrengthMode.getKeyLength() + LENGTH_PASSWORD_VERIFICATION_VALUE;
		byte[] derivedKeyByPbkdf2 = new Pbkdf2().deriveKey(password, _saltValue, ITERATION_COUNT, derivedKeyLen);
		_encryptionKey = new byte[_encryptionStrengthMode.getKeyLength()];
		System.arraycopy(derivedKeyByPbkdf2, 0, _encryptionKey, 0, _encryptionKey.length);
		_authenticationKey = new byte[_encryptionStrengthMode.getKeyLength()];
		System.arraycopy(derivedKeyByPbkdf2, _encryptionKey.length, _authenticationKey, 0, _authenticationKey.length);
		_passwordVerificationValue = new byte[LENGTH_PASSWORD_VERIFICATION_VALUE];
		System.arraycopy(derivedKeyByPbkdf2, _encryptionKey.length + _authenticationKey.length, _passwordVerificationValue, 0, _passwordVerificationValue.length);
	}
	
	@Override
	public long getCompressedSize() {
		long originalCompressedSize = super.getCompressedSize();
		if (originalCompressedSize < 0) {
			// FIXME
			throw new IllegalStateException();
		}
		return originalCompressedSize + _encryptionStrengthMode.getSaltLength()
		+ LENGTH_PASSWORD_VERIFICATION_VALUE + LENGTH_AUTHENTICATION_CODE;
	}
	
	@Override
	public void setMethod(int method) {
		if (method != METHOD_AES && method != METHOD_STORED) {
			// FIXME:
			throw new IllegalArgumentException();
		}
		_method = method;
	}
	
	@Override
	public int getMethod() {
		return _method;
	}

	@Override
	public byte[] getExtra() {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		byte[] originalExtra = super.getExtra();		
		try {
			// extra field of original entry.
			if (originalExtra != null) {
				baos.write(originalExtra);
			}
		} catch (IOException ioe) {
			// FIXME Auto-generated catch block
			ioe.printStackTrace();
		}
		if (this.getMethod() != METHOD_AES) {
			return baos.toByteArray();
		}
		
		try {
			// extra field header ID: 0x9901 - 2 bytes
			// data size: 0x0007 - 2 bytes
			// integer version number: 0x0002 (AE-2) - 2 bytes
			// vender ID: "AE" - 2 bytes
			// 01 99 07 00 02 00 41 45
			baos.write(AES_EXTRA_BYTES);
		} catch (IOException ioe) {
			// FIXME Auto-generated catch block
			ioe.printStackTrace();
		}

		// integer mode value indicating AES encryption strength - 1 byte
		baos.write(this.getStrengthModeValue());
		
		// actual compression method used to compress the file - 2 bytes
		short value = this.getActualCompressionMethod();
		baos.write((byte) (value & 0xff));
		baos.write((byte) ((value >>> 8) & 0xff));
		
		return baos.toByteArray();
	}
	
	public void setActualCompressionMethod(int method) {
		super.setMethod(method);
	}
	
	public short getActualCompressionMethod() {
		// FIXME: verify the method value.
		short actualCompressionMethod = (short) super.getMethod();
		if (actualCompressionMethod != METHOD_STORED && actualCompressionMethod != METHOD_DEFLATED) {
			throw new IllegalArgumentException(MessageFormat.format(MESSAGE_INVALID_METHOD, actualCompressionMethod));
		}
		return actualCompressionMethod;
	}

	/**
	 * set the offset of this entry in the zip file.
	 * 
	 * @param offset
	 *            offset (4-byte value)
	 * @throws IllegalArgumentException
	 *             the parameter is too big or too small.
	 */
	public void setRelativeOffsetOfLocalFileHeader(long offset) {
		if (offset < 0 || offset > MAX_FOUR_BYTE_FIELD_LONG) {
			throw new IllegalArgumentException(MessageFormat.format(MESSAGE_OFFSET_INVALID, offset,
					MAX_FOUR_BYTE_FIELD_LONG));
		}
		_relativeOffsetOfLocalFileHeader = offset;
	}

	public long getRelativeOffsetOfLocalFileHeader() {
		return _relativeOffsetOfLocalFileHeader;
	}

	public short getLastModTime() {
		return AesZipEntry.generateTime(super.getTime());
	}

	public short getLastModDate() {
		return AesZipEntry.generateDate(super.getTime());
	}

	public byte getStrengthModeValue() {
		return _encryptionStrengthMode.getModeValue();
	}
	
	public byte[] getSaltValue() {
		return _saltValue;
	}
	
	public byte[] getEncryptionKey() {
		return _encryptionKey;
	}
	
	public byte[] getAuthenticationKey() {
		return _authenticationKey;
	}
	
	public byte[] getPasswordVerificationValue() {
		return _passwordVerificationValue;
	}
	
	/*
	 * TODO: Is the randomness sufficiet ??
	 */
	private static byte[] generateSaltValue(int length) {
		byte[] r = new byte[length];
		new SecureRandom().nextBytes(r);
		return r;
	}
	
	private static short generateTime(long time) {
		Calendar c = Calendar.getInstance();
		c.setTimeInMillis(time);
		if (c.get(Calendar.YEAR) < 1980) {
			return 0;
		}
		int hour = c.get(Calendar.HOUR_OF_DAY);
		int minute = c.get(Calendar.MINUTE);
		int second = c.get(Calendar.SECOND);
		return (short) (hour << 11 | minute << 5 | second >> 1);
	}

	private static short generateDate(long time) {
		Calendar c = Calendar.getInstance();
		c.setTimeInMillis(time);
		int year = c.get(Calendar.YEAR);
		int month = c.get(Calendar.MONTH);
		int date = c.get(Calendar.DATE);
		if (year < 1980) {
			year = 1980;
			month = 1;
			date = 1;
		}
		return (short) ((year - 1980) << 9 | (month + 1) << 5 | date);
	}
}
