/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.io.IOException;
import java.io.StringReader;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2012/07/24
 */
public final class SchemeOptions {

	//
	private static final int INIT = 1000;
	private static final int EVAL = 1010;
	private static final int EXEC = 1020;
	private static final String REPORT_LOAD_TIME =
			"--report-load-time";

	//
	private static long loadTime;

	//
	private SchemeOptions() {}

	//
	static Locale parseLocale(String s) {
		String[] ls;

		if(s == null || s.equals(""))  return null;
		ls = s.split("(-|\\.|[_,])");
		switch(ls.length) {
		case 1:
			return new Locale(ls[0].toLowerCase());
		case 2:
			return new Locale(
					ls[0].toLowerCase(),
					ls[1].toUpperCase());
		case 3:
			return new Locale(
					ls[0].toLowerCase(),
					ls[1].toUpperCase(),
					ls[2].toUpperCase());
		default:
			return null;
		}
	}

	/**
	 * @param opt2
	 * @return
	 */
	public static Scheme preparseOption(String[] args) {
		int opt = 0;
		Set<String> libs = new LinkedHashSet<String>();
		Locale l = null;
		String s;

		// initialize
		libs.add("core");

		// parse options (pass 1)
		outer: while(opt < args.length) {
			s = args[opt++];
			if(s.equals("-version") || s.equals("--version")) {
				System.out.print("Schluessel, Version ");
				System.out.println(Scheme.SCHLUSH_VERSION);
				System.exit(0);
			} else if(s.equals(REPORT_LOAD_TIME)) {
				loadTime = System.currentTimeMillis();
			} else if(s.equals("--")) {
				break outer;
			}
		}
		return Scheme.newInstance(l);
	}

	/**
	 * 
	 * @param args
	 * @return
	 */
	public static int parseOption(String[] args, Scheme eval) {
		int opt = 0, state = INIT;
		boolean end = false;
		long endtime;
		String s, u;
		Datum d;

		// parse properties (pass 1)
		try {
			u = System.getProperty("schluessel.foldcase");
			if(Boolean.parseBoolean(u)) {
				Symbol.foldCase = Symbol.LOWER_FOLD_CASE;
			} else {
				Symbol.foldCase = Symbol.NO_FOLD_CASE;
			}
		} catch(SecurityException e) {
			// ignore
		}

		// parse options (pass 2)
		try {
			while(opt < args.length) {
				s = args[opt++];
				switch(state) {
				case INIT:
					if(s.equals("-e") || s.equals("--eval")) {
						state = EVAL;
					} else if(s.equals("-E") || s.equals("--exec")) {
						state = EXEC;
					} else if(s.equals("--fold-case")) {
						Symbol.foldCase = Symbol.LOWER_FOLD_CASE;
					} else if(s.equals("--no-fold-case")) {
						Symbol.foldCase = Symbol.NO_FOLD_CASE;
					} else if(s.equals("--")) {
						if(end)  System.exit(0);
						return opt;
					} else if(s.equals("-")) {
						if(end)  System.exit(0);
						return opt - 1;
					} else if(s.equals(REPORT_LOAD_TIME)) {
						endtime = System.currentTimeMillis();
						System.err.print("Load time: ");
						System.err.print(
								(endtime - loadTime) / 1000.0);
						System.err.println(" seconds");
					} else {
						System.err.print("Invalid option: ");
						System.err.println(s);
						System.exit(2);
					}
					break;
				case EVAL:
					eval.read(new StringReader(s));
					end = true;
					state = INIT;
					break;
				case EXEC:
					d = NanoParser.exec(s);
					if(d != null) {
						System.out.println(LispUtils.toDisplay(d));
					}
					end = true;
					state = INIT;
					break;
				}
			}
		} catch(ReadFileException e) {
			System.err.println(e.getMessage());
			System.exit(2);
		} catch (IOException e) {
			System.err.println(e.getMessage());
			System.exit(2);
		}

		if(end)  System.exit(0);
		return opt;
	}

}
