/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.painter;

import java.awt.Graphics;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.JPanel;

import net.morilib.lisp.Datum;
import net.morilib.lisp.painter.drawer.CoordinateMap;
import net.morilib.lisp.painter.drawer.Drawer;
import net.morilib.lisp.painter.geom.Point2D;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2010/12/12
 */
public class SchlushWindowFrame
extends Datum implements SchlushFrame {
	
	//
	private class SFrame extends JPanel {

		//
		private static final long serialVersionUID = 8388728689806126886L;
		
		//
		private void paintImage(Graphics g, ImageInfo info) {
			Point2D p0, ps;
			
			if(coordinate != null) {
				p0 = coordinate.transform(
						frame.getWidth(),
						frame.getHeight(),
						info.x, info.y);
				ps = coordinate.transformSize(
						frame.getWidth(),
						frame.getHeight(),
						info.sx, info.sy);
				g.drawImage(info.img,
						p0.getXInt(),
						p0.getYInt() - ps.getYInt(),
						this);
			} else {
				g.drawImage(info.img, (int)info.x, (int)info.y, this);
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
		 */
		@Override
		protected void paintComponent(Graphics g) {
			super.paintComponent(g);
			
			synchronized(images) {
				for(ImageInfo info : images) {
					paintImage(g, info);
				}
			}
			synchronized(drawers) {
				for(Drawer drawer : drawers) {
					drawer.draw(
							g,
							frame.getWidth(),
							frame.getHeight(),
							coordinate);
				}
			}
		}
		
	}
	
	//
	/*package*/ JFrame frame;
	private List<ImageInfo> images  = new ArrayList<ImageInfo>();
	private List<Drawer>    drawers = new ArrayList<Drawer>();
	private CoordinateMap coordinate;
	private boolean disposed = false;
	
	/**
	 * 
	 * @param x
	 * @param y
	 * @param coordinate
	 */
	public SchlushWindowFrame(
			int x, int y, CoordinateMap coordinate) {
		frame = new JFrame();
		frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
		frame.setSize(x, y);
		frame.add(new SFrame());
		frame.setVisible(true);
		frame.addWindowListener(new WindowListener() {

			@Override
			public void windowOpened(WindowEvent e) {
			}

			@Override
			public void windowClosing(WindowEvent e) {
			}

			@Override
			public void windowClosed(WindowEvent e) {
				disposed = true;
			}

			@Override
			public void windowIconified(WindowEvent e) {
			}

			@Override
			public void windowDeiconified(WindowEvent e) {
			}

			@Override
			public void windowActivated(WindowEvent e) {
			}

			@Override
			public void windowDeactivated(WindowEvent e) {
			}
			
		});
		this.coordinate = coordinate;
	}
	
	
	public SchlushWindowFrame(int x, int y) {
		this(x, y, null);
	}

	//
	private void loadImage(Image img) throws InterruptedException {
		MediaTracker mt = new MediaTracker(frame);
		
		mt.addImage(img, 1);
		mt.waitForAll();
	}
	
	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#addImage(net.morilib.lisp.painter.SchlushFrame.ImageInfo)
	 */
	public void addImage(ImageInfo img) throws InterruptedException {
		Image img2 = null;
		Point2D ps;
		
		if(disposed) {
			return;
		}
		
		if(coordinate != null) {
			ps = coordinate.transformSize(
					frame.getWidth(),
					frame.getHeight(),
					img.sx, img.sy);
			
			if(ps.getXInt() > 0 && ps.getYInt() > 0) {
				img2 = img.img.getScaledInstance(
						ps.getXInt(), ps.getYInt(),
						Image.SCALE_SMOOTH);
			}
		} else {
			if((int)img.sx > 0 && (int)img.sy > 0) {
				img2 = img.img.getScaledInstance(
						(int)img.sx, (int)img.sy,
						Image.SCALE_SMOOTH);
			}
		}
		
		if(img2 != null) {
			loadImage(img.img);
			synchronized(images) {
				images.add(new ImageInfo(
						img2, img.x, img.y, img.sx, img.sy));
			}
			frame.repaint();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#addDrawer(net.morilib.lisp.painter.drawer.Drawer)
	 */
	public void addDrawer(Drawer drawer) {
		if(disposed) {
			return;
		}
		
		synchronized(drawers) {
			drawers.add(drawer);
		}
		frame.repaint();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#getImageWidth(java.awt.Image)
	 */
	@Override
	public int getImageWidth(Image img) {
		return img.getWidth(frame);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#getImageHeight(java.awt.Image)
	 */
	@Override
	public int getImageHeight(Image img) {
		return img.getHeight(frame);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#clearScreen()
	 */
	@Override
	public void clearScreen() {
		if(disposed) {
			return;
		}
		
		synchronized(images) {
			images.clear();
		}
		synchronized(drawers) {
			drawers.clear();
		}
		frame.repaint();
	}

	/**
	 * @return the coordinate
	 */
	public CoordinateMap getCoordinate() {
		return coordinate;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#getHeight()
	 */
	@Override
	public int getHeight() {
		return frame.getHeight();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.painter.SchlushFrame#getWidth()
	 */
	@Override
	public int getWidth() {
		return frame.getWidth();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#toDisplayString(java.lang.StringBuilder)
	 */
	@Override
	public void toDisplayString(StringBuilder buf) {
		buf.append("#<window-frame>");
	}
	
}
