/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

public class OutputPort extends Datum {
	
	//
	private Writer writer;
	private PrintWriter lineWriter;
	private boolean standard;
	private boolean closed = false;
	private LispMessage msg;
	
	//
	private OutputPort(Writer wr, boolean std, LispMessage msg) {
		this.writer     = wr;
		this.lineWriter = new PrintWriter(this.writer);
		this.standard   = std;
		this.msg        = msg;
	}
	
	
	/*package*/ OutputPort(Writer wr, LispMessage msg) {
		this(new BufferedWriter(wr), false, msg);
	}
	
	
	/*package*/ OutputPort(String fname, LispMessage msg) {
		try {
			Writer wr;
			
			wr = new OutputStreamWriter(new FileOutputStream(fname));
			this.writer     = new BufferedWriter(wr);
			this.lineWriter = new PrintWriter(this.writer);
			this.standard   = false;
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}
		
	}
	
	
	/*package*/ static OutputPort getStringPort(LispMessage msg) {
		return new OutputPort(new StringWriter(), false, msg);
	}
	
	
	public static OutputPort getStandard(LispMessage msg) {
		return new OutputPort(
				new OutputStreamWriter(System.out), true, msg);
	}
	
	
	public void writeChar(LispCharacter ch) {
		if(closed) {
			throw msg.getError("err.port.closed");
		}
		
		try {
			if(standard) {
				System.out.print(ch.getCharacter());
			} else {
				writer.write(ch.getCharacter());
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	public void write(Datum d) {
		if(closed) {
			throw msg.getError("err.port.closed");
		}
		
		if(standard) {
			System.out.print(LispUtils.getResult(d));
		} else {
			lineWriter.print(LispUtils.getResult(d));
		}
	}
	
	
	public void display(Datum d) {
		if(closed) {
			throw msg.getError("err.port.closed");
		}
		
		if(standard) {
			System.out.print(LispUtils.print(d));
		} else {
			lineWriter.print(LispUtils.print(d));
		}
	}
	
	
	public void newline() {
		if(closed) {
			throw msg.getError("err.port.closed");
		}
		
		if(standard) {
			System.out.println();
		} else {
			lineWriter.println();
		}
	}
	
	
	public void close() {
		try {
			if(!closed && !standard) {
				writer.close();
				closed = true;
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	public boolean isStandard() {
		return standard;
	}
	
	
	public boolean isTypePort() {
		return true;
	}
	
	
	public boolean isStringPort() {
		return writer instanceof StringWriter;
	}
	
	
	public String getOutputString() {
		if(isStringPort()) {
			return new String(writer.toString());
		} else {
			throw new IllegalStateException();
		}
	}
	
}
