/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.util;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class StateGraph<T, S> {
	
	private Map<T, Map<S, T>> graph = new HashMap<T, Map<S, T>>();
	private Map<T, StateIn> stmap = new HashMap<T, StateIn>();
	
	
	public static interface State<T> {
		
		public State<T> go(T alphabet);
		
	}
	
	private class StateIn implements State<T> {
		
		private T st;
		
		private StateIn(T st) {
			this.st = st;
		}
		
		public State<T> go(T alphabet) {
			return stmap.get(graph.get(st).get(alphabet));
		}

	}
	
	private static class Immutable<T, S> extends StateGraph<T, S> {
		
		private StateGraph<T, S> wrapee;
		
		private Immutable(StateGraph<T, S> w) {
			wrapee = w;
		}
		
		//
		@Override
		public void addNode(T nst) {
			throw new UnsupportedOperationException();
		}

		//
		@Override
		public void addTrans(T st, S i, T nst) {
			throw new UnsupportedOperationException();
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#get(java.lang.Object, java.lang.Object)
		 */
		@Override
		public T get(T st, S inp) {
			return wrapee.get(st, inp);
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#getAllNodes()
		 */
		@Override
		public Set<T> getAllNodes() {
			return wrapee.getAllNodes();
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#getEdgeMap(java.lang.Object)
		 */
		@Override
		public Map<S, T> getEdgeMap(T st) {
			return wrapee.getEdgeMap(st);
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#getState(java.lang.Object)
		 */
		@Override
		public State<T> getState(T st) {
			return wrapee.getState(st);
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#isState(java.lang.Object)
		 */
		@Override
		public boolean isState(T st) {
			return wrapee.isState(st);
		}
		
		//
		@Override
		public int stateSize() {
			return wrapee.stateSize();
		}

		/* (non-Javadoc)
		 * @see net.morilib.util.StateGraph#putEdgeMap(java.lang.Object, java.util.Map)
		 */
		@Override
		public void addEdgeMap(T st, Map<S, T> edges) {
			throw new UnsupportedOperationException();
		}
		
	}
	
	
	public static<T, S> StateGraph<T, S> unmodifiableGraph(
			StateGraph<T, S> w) {
		if(w instanceof Immutable<?, ?>) {
			return w;
		} else {
			return new Immutable<T, S>(w);
		}
	}
	
	
	private Map<S, T> newNode(T nst) {
		Map<S, T> ne = graph.get(nst);
		
		if(ne == null) {
			ne = new HashMap<S, T>();
			graph.put(nst, ne);
			stmap.put(nst, new StateIn(nst));
		}
		return ne;
	}
	
	
	public void addNode(T nst) {
		newNode(nst);
	}
	
	
	public void addTrans(T st, S i, T nst) {
		Map<S, T> edges;
		
		if(st == null || i == null || nst == null) {
			throw new NullPointerException();
		}
		
		edges = graph.get(st);
		if(edges == null) {
			edges = new HashMap<S, T>();
			edges.put(i, nst);
			graph.put(st, edges);
			stmap.put(st, new StateIn(st));
		} else {
			edges.put(i, nst);
		}
		
		addNode(nst);
	}
	
	
	public State<T> getState(T st) {
		State<T> res = stmap.get(st);
		
		if(res == null) {
			throw new NullPointerException();
		}
		return res;
	}
	
	
	public Set<T> getAllNodes() {
		return Collections.unmodifiableSet(graph.keySet());
	}
	
	
	public Map<S, T> getEdgeMap(T st) {
		return Collections.unmodifiableMap(graph.get(st));
	}
	
	
	public void addEdgeMap(T st, Map<S, T> edges) {
		Map<S, T> nd = newNode(st);
		
		for(Map.Entry<S, T> e : edges.entrySet()) {
			nd.put(e.getKey(), e.getValue());
		}
	}
	
	
	public T get(T st, S inp) {
		Map<S, T> edg = graph.get(st);
		T res;
		if(st == null || inp == null) {
			throw new NullPointerException();
		} else if(edg == null) {
			throw new IllegalArgumentException();
		}
		
		if((res = edg.get(inp)) == null) {
			throw new IllegalArgumentException();
		} else {
			return res;
		}
	}
	
	
	public boolean isState(T st) {
		return graph.containsKey(st);
	}
	
	
	public int stateSize() {
		return graph.size();
	}
	
}
