/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public final class Environment {
	
	private Environment rootenv;
	private Map<Symbol, Datum> binds = Collections.synchronizedMap(
			new HashMap<Symbol, Datum>());
	
	
	/*package*/ Environment() {
		rootenv = null;
	}
	
	
	/*package*/ Environment(Environment rootenv) {
		this.rootenv = rootenv;
	}
	
	
	public void bindDatum(Datum sym, Datum d) {
		if(sym instanceof Symbol) {
			binds.put((Symbol)sym, d);
		} else {
			//System.out.println(sym);
			throw new LispException("Parameter is not a symbol");
		}
	}
	
	
	/*package*/ void bindDatumWithoutScope(Datum sym, Datum d) {
		if(sym instanceof SymbolName) {
			binds.put(((SymbolName)sym).getSymbol(), d);
		} else {
			//System.out.println(sym);
			throw new LispException("Parameter is not a symbol");
		}
	}
	
	
	public Datum getDatum(Datum sym) {
		if(sym instanceof Symbol) {
			return binds.get(sym);
		} else {
			throw new LispException("Parameter is not a symbol");
		}
	}
	
	
	public Datum findDatum(Datum sym) {
		if(sym instanceof Symbol) {
			Environment env = this;
			
			while(env != null) {
				Datum f = env.binds.get(sym);
				
				if(f != null) {
					return f;
				}
				env = env.rootenv;
			}
			return null;
		} else {
			throw new LispException("Parameter is not a symbol");
		}
	}
	
	
	/*package*/ boolean setDatum(Datum sym, Datum d) {
		if(sym instanceof Symbol) {
			Environment env = this;
			
			while(env != null) {
				Datum f = env.binds.get(sym);
				
				if(f != null) {
					env.binds.put((Symbol)sym, d);
					return true;
				}
				env = env.rootenv;
			}
			//throw new NotBoundException();
			return false;
		} else {
			throw new LispException("Parameter is not a symbol");
		}
	}
	
	
	/*package*/ Environment getGlobal() {
		Environment env = this;
		
		while(env.rootenv != null) {
			env = env.rootenv;
		}
		return env;
	}
	
	
	/*package*/ Environment getRootenv() {
		return rootenv;
	}
	
	
	/*package*/ Environment copy() {
		Environment res = new Environment();
		
		res.rootenv = rootenv;
		res.binds   = new HashMap<Symbol, Datum>(binds);
		return res;
	}
	
	
	/*package*/ Environment copyNotRoot() {
		return (rootenv != null) ? copy() : this;
	}
	
	
	/*package*/ /*Environment copyChangeRoot(Environment rt) {
		Environment res = new Environment();
		
		res.rootenv = rt;
		res.binds   = new HashMap<Symbol, Datum>(binds);
		return res;
	}*/
	
	
	public String toString() {
		Environment env = this;
		StringBuilder buf = new StringBuilder();
		
		while(env.rootenv != null) {
			buf.append(env.binds).append("->");
			env = env.rootenv;
		}
		buf.append("{Global}");
		return buf.toString();
	}
	
}
