package com.sanpudo.formula;

import java.math.BigDecimal;
import java.util.Vector;

/**
 * 関数。
 * 
 * @author Sanpudo.
 */

public class Function extends Container {
	String name;
	int numberOfArgs;
	Vector<Token> args;
	Environments environments;

	/** トークンの正規表現。 */
	static final String regex = "[a-zA-Z]\\w*\\(";

	/** コンストラクタ。 */
	Function(String formula, int location, String name, int numberOfArgs,
			Environments environments) {
		super(formula, location);
		this.name = name;
		this.numberOfArgs = numberOfArgs;
		this.environments = environments;
		this.args = new Vector<Token>();
	}

	/** doubleの評価値を返す。 */
	double value() throws FormulaEvaluatorException {
		// 引数を配列にする
		double[] dArgs = new double[args.size()];
		for (int i = 0; i < dArgs.length; i++) {
			dArgs[i] = args.get(i).value();
		}
		try {
			// システム関数のコール
			if (Environments.systemFunctions.supports(name)) {
				return Environments.systemFunctions.value(name, dArgs);
			}
			// ユーザ関数のコール
			for (FunctionImplementation fun : environments.userFunctions) {
				if (fun.supports(name)) {
					return fun.value(name, dArgs);
				}
			}
			// 関数のExceptionを検知
		} catch (FunctionEvalException e) {
			throw new FormulaEvaluatorException(e.getMessage(), e.getCause(),
					formula, location);
		}
		// 未実装関数を検知
		catch (FunctionException e) {
			throw new FormulaEvaluatorException(e.getMessage(), e
					.getFunctionName(), location);
		}
		throw new FormulaEvaluatorException(Messages.INTERNAL_ERROR, name, -1);
	}

	/** BigDecimalの評価値を返す。 */
	BigDecimal value(Rounding rounding) throws FormulaEvaluatorException {
		// 引数を配列にする
		BigDecimal[] bdArgs = new BigDecimal[args.size()];
		for (int i = 0; i < bdArgs.length; i++) {
			bdArgs[i] = args.get(i).value(rounding);
		}
		try {
			// システム関数のコール
			if (Environments.systemFunctions.supports(name)) {
				return Environments.systemFunctions.value(name, bdArgs);
			}
			// ユーザ関数のコール
			for (FunctionImplementation fun : environments.userFunctions) {
				if (fun.supports(name)) {
					return rounding.everyRound(fun.value(name, bdArgs));
				}
			}
			// 関数のExceptionを検知
		} catch (FunctionEvalException e) {
			throw new FormulaEvaluatorException(e.getMessage(), e.getCause(),
					formula, location);
		}
		// 未実装関数を検知
		catch (FunctionException e) {
			throw new FormulaEvaluatorException(e.getMessage(), e
					.getFunctionName(), location);
		}
		throw new FormulaEvaluatorException(Messages.INTERNAL_ERROR, name, -1);
	}

	/**
	 * ファクターの構造が完全で無いときに。その理由メッセージを返す。 そうでないときはnullを返す。
	 */
	String imcompleteMessage() {
		if (isOpen()) {
			return Messages.PAREN_INCOMPLETE;
		} else if (args.size() != numberOfArgs) {
			return Messages.INVALID_ARG_NO;
		}
		return null;
	}

	/**
	 * ファクターの構造が完全であるかチェックする。
	 */
	boolean isComplete() {
		if ((!isOpen()) && (args.size() == numberOfArgs)) {
			return true;
		} else {
			return false;
		}
	}

	/** 正規化した式を返す。 */
	String normalized() {
		String s = "";
		for (Token arg : args) {
			if (!s.equals("")) {
				s += ",";
			}
			s += arg.normalized();
		}
		return name + "(" + s + ")";
	}

	/** token文字列を返す。 */
	String strToken() {
		return name + "(";
	}

	/** 引数が満たされているか。 */
	boolean filled() {
		return args.size() >= numberOfArgs;
	}

	/** 子の登録。 */
	void bind(Token token) {
		this.args.add(token);
		token.parent = this;
	}

	/** 子の再登録。 */
	void reBind(Token token) {
		this.args.remove(this.args.size() - 1);
		bind(token);
	}

}
