
package jp.riken.brain.ni.samuraigraph.figure.java2d;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import jp.riken.brain.ni.samuraigraph.base.SGAxis;
import jp.riken.brain.ni.samuraigraph.base.SGDefaultValues;
import jp.riken.brain.ni.samuraigraph.base.SGIAxisBreakElement;
import jp.riken.brain.ni.samuraigraph.base.SGIAxisElement;
import jp.riken.brain.ni.samuraigraph.base.SGIConstants;
import jp.riken.brain.ni.samuraigraph.base.SGICopiable;
import jp.riken.brain.ni.samuraigraph.base.SGIDisposable;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElement;
import jp.riken.brain.ni.samuraigraph.base.SGIGraphElement;
import jp.riken.brain.ni.samuraigraph.base.SGIGridElement;
import jp.riken.brain.ni.samuraigraph.base.SGILegendElement;
import jp.riken.brain.ni.samuraigraph.base.SGIMovable;
import jp.riken.brain.ni.samuraigraph.base.SGINode;
import jp.riken.brain.ni.samuraigraph.base.SGISelectable;
import jp.riken.brain.ni.samuraigraph.base.SGIShapeElement;
import jp.riken.brain.ni.samuraigraph.base.SGISignificantDifferenceElement;
import jp.riken.brain.ni.samuraigraph.base.SGIStringElement;
import jp.riken.brain.ni.samuraigraph.base.SGITimingLineElement;
import jp.riken.brain.ni.samuraigraph.base.SGIUndoable;
import jp.riken.brain.ni.samuraigraph.base.SGProperties;
import jp.riken.brain.ni.samuraigraph.base.SGPropertyDialog;
import jp.riken.brain.ni.samuraigraph.base.SGTuple2f;
import jp.riken.brain.ni.samuraigraph.base.SGUndoManager;
import jp.riken.brain.ni.samuraigraph.base.SGUtility;
import jp.riken.brain.ni.samuraigraph.base.SGUtilityNumber;
import jp.riken.brain.ni.samuraigraph.base.SGUtilityText;
import jp.riken.brain.ni.samuraigraph.figure.SGAxisBreakSymbol;
import jp.riken.brain.ni.samuraigraph.figure.SGFigureElement;
import jp.riken.brain.ni.samuraigraph.figure.SGIAxisBreakSymbolConstants;
import jp.riken.brain.ni.samuraigraph.figure.SGAxisBreakSymbol.AxisBreakSymbolProperties;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 *
 *
 */
public class SGAxisBreakElement extends SGFigureElement
	implements SGIAxisBreakElement, SGIAxisBreakSymbolConstants
{

	/**
	 * 
	 */
	private SGIAxisElement mAxisElement = null;

	
	/**
	 * 
	 */
	private SGAxisBreakSymbolDialog mDialog = null;


	/**
	 * 
	 *
	 */
	public SGAxisBreakElement()
	{
		super();
	}


	/**
	 * 
	 */
	public void dispose()
	{
		super.dispose();
		this.mDialog.dispose();
		this.mDialog = null;
		this.mAxisElement = null;
	}


	/**
	 * 
	 */
	public boolean setDialogOwner( final Frame frame )
	{
		super.setDialogOwner(frame);
		this.createDialog();
		return true;
	}



	/**
	 * create a dialog of axis break symbols
	 */
	private boolean createDialog()
	{
		final SGAxisBreakSymbolDialog absdg
			= new SGAxisBreakSymbolDialog( this.mDialogOwner, true );
		this.mDialog = absdg;
		return true;
	}


	/**
	 * 
	 */
	public String toString()
	{
		return "SGAxisBreakElement";
	}

	
	/**
	 * 
	 * @return
	 */
	public String getClassDescription()
	{
		return "Axis Break Symbols";
	}


	
	/**
	 * 
	 */
	public boolean synchronize( final SGIFigureElement element )
	{

		boolean flag = true;
		if( element instanceof SGIGraphElement )
		{
			
		}
		else if( element instanceof SGIStringElement )
		{
			
		}
		else if( element instanceof SGILegendElement )
		{
			
		}
		else if( element instanceof SGIAxisElement )
		{
			flag = this.synchronizeToAxisElement( (SGIAxisElement)element );
		}
		else if( element instanceof SGIAxisBreakElement )
		{
			
		}
		else if( element instanceof SGISignificantDifferenceElement )
		{
			
		}
		else if( element instanceof SGITimingLineElement )
		{
			
		}
		else if( element instanceof SGIGridElement )
		{

		}
		else if( element instanceof SGIShapeElement )
		{

		}
		else
		{
			flag = element.synchronizeArgument( this );
		}


		return flag;
	}


	
	private boolean synchronizeToAxisElement( SGIAxisElement element )
	{
		ArrayList list = this.getVisibleChildList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			el.setDrawingElementsLocation();
		}
		
		return true;
	}
	
	

	/**
	 * Synchronize the element given by the argument.
	 * @param element An object to be synchronized.
	 */
	public boolean synchronizeArgument( final SGIFigureElement element )
	{
	    // this shouldn't happen
	    throw new Error();
	}


	
	/**
	 * 
	 * @param element
	 */
	public void setAxisElement( final SGIAxisElement element )
	{
		this.mAxisElement = element;
	}


	/**
	 * 
	 */
	public String getTagName()
	{
		return TAG_NAME_AXIS_BREAK;
	}

	

	/**
	 * 
	 */
	public boolean writeProperty( Element el )
	{
		return true;
	}


	
	/**
	 * 
	 */
	public boolean readProperty( final Element element )
	{
		NodeList nList = element.getElementsByTagName( SGAxisBreakSymbol.TAG_NAME_AXIS_BREAK_SYMBOL );
		for( int ii=0; ii<nList.getLength(); ii++ )
		{
			Node node = nList.item(ii);
			if( node instanceof Element )
			{
				Element el = (Element)node;
				AxisBreakSymbol abs = new AxisBreakSymbol();
				if( abs.readProperty(el) == false )
				{
					return false;
				}
				abs.initPropertiesHistory();
				this.addToList(abs);
			}
		}
		
		return true;
	}



	/**
	 * 
	 */
	public boolean setGraphRect(
		final float x, final float y, final float width, final float height )
	{
		super.setGraphRect(x,y,width,height);
		this.setLocationOfAllDrawingElements();

		return true;
	}

	
	/**
	 * 
	 */
	private boolean setLocationOfAllDrawingElements()
	{
		ArrayList list = this.mChildList;
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol abs = (AxisBreakSymbol)list.get(ii);
			abs.setDrawingElementsLocation();
		}
		return true;
	}

	
	/**
	 * Y[
	 */
	public boolean zoom( final float ratio )
	{
		super.zoom(ratio);

		ArrayList list = this.mChildList;
		for( int ii=0; ii<list.size(); ii++ )
		{
			final AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			el.zoom(ratio);
		}

		return true;
	}




	/**
	 * 
	 */
	public void paintGraphics( Graphics g, boolean clip )
	{

		final Graphics2D g2d = (Graphics2D)g;

		// draw all visible symbols
		ArrayList list = this.getVisibleChildList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			el.paintElement(g2d);
//			if( el.mFrameFlag )
//			{
//				el.drawBoundingBox(g2d);
//			}
		}
		
		
		// draw symbols around all objects
		if( this.mSymbolsVisibleFlagAroundAllObjects )
		{
			for( int ii=0; ii<list.size(); ii++ )
			{
				AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
				if( el.isValid() == false )
				{
					continue;
				}
				ArrayList pList = el.getAnchorPointList();
				SGUtilityForFigureElement.drawAnchorAsChildObject( pList, g2d );
			}
		}

		// draw symbols around focused objects
		if( this.mSymbolsVisibleFlagAroundFocusedObjects )
		{
			ArrayList fList = new ArrayList();
			this.getFocusedObjectsList( fList );
			for( int ii=0; ii<fList.size(); ii++ )
			{
				AxisBreakSymbol el = (AxisBreakSymbol)fList.get(ii);
				if( el.isValid() == false )
				{
					continue;
				}
				ArrayList pList = el.getAnchorPointList();
				SGUtilityForFigureElement.drawAnchorAsFocusedObject( pList, g2d );
			}
		}
		
	}

	
	
	/**
	 * 
	 * @param document
	 * @return
	 */
	public Element createElement( final Document document )
	{
		// create an Element object
		Element el = this.createThisElement( document );
		if( el==null )
		{
			return null;
		}

		// axis symbol
		ArrayList abList = this.getVisibleChildList();
		for( int ii=0; ii<abList.size(); ii++ )
		{
			AxisBreakSymbol abs = (AxisBreakSymbol)abList.get(ii);
			if( abs.isValid() == false )
			{
				continue;
			}

			Element elAbs = abs.createElement( document );
			if( elAbs==null )
			{
				return null;
			}
			el.appendChild( elAbs );
		}
		
		return el;
	}

	
	
	/**
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	public boolean addAxisBreakSymbol( final int x, final int y )
	{
		if( this.getGraphRect().contains( x, y ) == false )
		{
			return false;
		}

		SGAxis xAxis = mAxisElement.getAxis( SGDefaultValues.AXIS_BREAK_HORIZONTAL_AXIS );
		SGAxis yAxis = mAxisElement.getAxis( SGDefaultValues.AXIS_BREAK_PERPENDICULAR_AXIS );
		return this.addAxisBreakSymbol( xAxis, yAxis, x, y );
	}

	
	

	/**
	 * 
	 */
	public boolean addAxisBreakSymbol(
		final SGAxis xAxis, final SGAxis yAxis, final int x, final int y )
	{
		
		AxisBreakSymbol el = new AxisBreakSymbol(
			SGDefaultValues.AXIS_BREAK_LENGTH,
			SGDefaultValues.AXIS_BREAK_INTERVAL,
			SGDefaultValues.AXIS_BREAK_DISTORTION,
			SGDefaultValues.AXIS_BREAK_ANGLE,
			SGDefaultValues.AXIS_BREAK_FOR_HORIZONTAL
		);
		el.addColor( SGDefaultValues.AXIS_BREAK_INNER_COLOR );
		el.setLineColor( SGDefaultValues.AXIS_BREAK_LINE_COLOR );
		el.setLineWidth( SGDefaultValues.AXIS_BREAK_LINE_WIDTH );
		el.setMagnification( this.mMagnification );
		el.mXAxis = xAxis;
		el.mYAxis = yAxis;

		el.setLocation(x,y);
		el.setAxisValue();


		//
		this.addToList( el );

		//
		this.setChanged(true);

		// initialize history
		el.initPropertiesHistory();
		
		this.notifyToRoot();

		return true;
	}



	private int mIDCounter = 0;


	/**
	 * 
	 * @param el
	 */
	private void addToList( AxisBreakSymbol el )
	{
		this.mChildList.add(el);

		// set ID
		el.mID = this.mIDCounter;
		this.mIDCounter++;
	}


	
	/**
	 * S`vf̈ʒu
	 */
	protected boolean setAllDrawingElementsLocation()
	{
		ArrayList list = this.getVisibleChildList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			final AxisBreakSymbol el
				= (AxisBreakSymbol)list.get(ii);
			if( el.setDrawingElementsLocation() == false )
			{
				return false;
			}
		}

		return true;
	}


	/**
	 * 
	 */
	public boolean setMementoBackward()
	{
		boolean flag = super.setMementoBackward();
		if( !flag )
		{
			return false;
		}

		this.clearFocusedObjects();
		this.notifyChange();

		return true;
	}



	/**
	 * 
	 */
	public boolean setMementoForward()
	{
		boolean flag = super.setMementoForward();
		if( !flag )
		{
			return false;
		}

		this.clearFocusedObjects();
		this.notifyChange();

		return true;
	}



	
	/**
	 * 
	 */
	public boolean onMouseClicked( final MouseEvent e )
	{
		// Axis Break Symbols
		ArrayList list = this.getVisibleChildList();
		for( int ii=list.size()-1; ii>=0; ii-- )
		{
			final AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			if( this.clickDrawingElements(el,e) )
			{
				return true;
			}
		}

		return false;
	}

	
	
	/**
	 * 
	 * @return
	 */
	public boolean setTemporaryPropertiesOfFocusedObjects()
	{
		ArrayList list = this.getFocusedObjectsList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			el.mTemporaryProperties = el.getProperties();
		}
		return true;
	}



	/**
	 * 
	 * @return
	 */
	public ArrayList getPropertyDialogObserverList()
	{
		return this.getFocusedObjectsList();
	}


	
	
	/**
	 * 
	 */
	private boolean clickDrawingElements(
		final AxisBreakSymbol el, final MouseEvent e )
	{
		final int x = e.getX();
		final int y = e.getY();
		final int cnt = e.getClickCount();

		// clicked on the line elements
		if( el.contains(x,y) )
		{
			this.updateFocusedObjectsList( el, e );

			if( SwingUtilities.isLeftMouseButton(e) & cnt==1 )
			{

			}
			else if( SwingUtilities.isLeftMouseButton(e) & cnt==2 )
			{
				this.setPropertiesOfSelectedObjects();
			}
			else if( SwingUtilities.isRightMouseButton(e) & cnt==1 )
			{
				el.getPopupMenu().show( this.getComponent(), x, y );
			}

			return true;
		}


		return false;
	}



	/**
	 * 
	 */
	public boolean onMousePressed( final MouseEvent e )
	{
		final int x = e.getX();
		final int y = e.getY();

		// Axis Break Symbols
		ArrayList list = this.getVisibleChildList();
		for( int ii=list.size()-1; ii>=0; ii-- )
		{
			final AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			if( el.contains(x,y) )
			{
				this.mPressedPoint = e.getPoint();
				this.setMouseCursor( Cursor.MOVE_CURSOR );
				if( el.isSelected() )
				{
					this.mDraggableFlag = true;
				}
				return true;
			}
		}

		return false;
	}



	/**
	 * 
	 * @param e
	 */
	public boolean onMouseDragged( final MouseEvent e )
	{
		if( this.mPressedPoint==null )
		{
			return false;
		}
		if( this.mDraggableFlag == false )
		{
			return false;
		}

		final int dx = e.getX() - this.mPressedPoint.x;
		final int dy = e.getY() - this.mPressedPoint.y;
		
		ArrayList list = this.getFocusedObjectsList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			el.translate(dx,dy);
		}
		
		this.mPressedPoint = e.getPoint();

		return true;
	}


	

	/**
	 * 
	 * @return
	 */
	public boolean setChangedFocusedObjects()
	{
		ArrayList list = this.getFocusedObjectsList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			SGProperties temp = el.mTemporaryProperties;
			if( temp!=null )
			{
				SGProperties p = el.getProperties();
				if( p.equals(temp)==false )
				{
					el.setChanged(true);
				}
			}
		}
		return true;
	}
	
	
	
	/**
	 * 
	 * @param e
	 */
	public boolean onMouseReleased( final MouseEvent e )
	{

		// Axis Break Symbol
		ArrayList list = this.getFocusedObjectsList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			Rectangle2D rect = el.getElementBounds();
			if( rect.contains(e.getPoint()) )
			{
				this.setMouseCursor( Cursor.HAND_CURSOR );
			}
			else
			{
				this.setMouseCursor( Cursor.DEFAULT_CURSOR );
			}
		}

		this.mDraggableFlag = false;
		this.notifyToRoot();
		
		return true;
	}
	
	
	
	/**
	 * 
	 */
	public boolean onDrawingElement( final int x, final int y )
	{
		ArrayList list = this.getVisibleChildList();
		for( int ii=list.size()-1; ii>=0; ii-- )
		{
			final AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			if( el.isValid() == false )
			{
				continue;
			}
			final boolean flag = el.contains(x,y);
//			el.mFrameFlag = flag;
			if( flag )
			{
				this.setMouseCursor( Cursor.HAND_CURSOR );
				return true;
			}
		}

		return false;
	}



	
	/**
	 * 
	 */
	public boolean getMarginAroundGraphRect(
		final SGTuple2f topAndBottom,
		final SGTuple2f leftAndRight )
	{

		if( super.getMarginAroundGraphRect(topAndBottom,leftAndRight) == false )
		{
			return false;
		}

		ArrayList list = this.getVisibleChildList();

		ArrayList axisBreakRectList = new ArrayList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			AxisBreakSymbol el = (AxisBreakSymbol)list.get(ii);
			Rectangle2D rect = el.getElementBounds();
			axisBreakRectList.add( rect );
		}
		
		Rectangle2D rectAxisBreakSymbols = null;
		if( axisBreakRectList.size()!=0 )
		{
			rectAxisBreakSymbols = SGUtility.createUnion( axisBreakRectList );
		}
		
		
		Rectangle2D gRect = this.getGraphRect();
		
		ArrayList rectList = new ArrayList();
		rectList.add( gRect );
		
		if( rectAxisBreakSymbols!=null )
		{
			rectList.add( rectAxisBreakSymbols );
		}
		
		Rectangle2D rectAll = SGUtility.createUnion( rectList );
		
		final float top = (float)( gRect.getY() - rectAll.getY() );
		final float bottom = (float)(
			( rectAll.getY() + rectAll.getHeight() )
			- ( gRect.getY() + gRect.getHeight() ) );
		final float left = (float)( gRect.getX() - rectAll.getX() );
		final float right = (float)(
			( rectAll.getX() + rectAll.getWidth() )
			- ( gRect.getX() + gRect.getWidth() ) );
		
		topAndBottom.x += top;
		topAndBottom.y += bottom;
		leftAndRight.x += left;
		leftAndRight.y += right;
		
		return true;
	}



	/**
	 * 
	 */
	protected Set getAvailableChildSet()
	{
		Set set = new HashSet();
		List mList = this.mUndoManager.getMementoList();
		for( int ii=0; ii<mList.size(); ii++ )
		{
			AxisProperties ap = (AxisProperties)mList.get(ii);
			set.addAll( ap.visibleAxisBreakSymbolList );
		}

		return set;		
	}



	/**
	 * 
	 */
	public static class AxisProperties extends SGProperties
	{
		ArrayList visibleAxisBreakSymbolList = new ArrayList();

		AxisProperties(){}
		
		public void dispose()
		{
			this.visibleAxisBreakSymbolList.clear();
			this.visibleAxisBreakSymbolList = null;
		}
		
		/**
		 * 
		 */
		public boolean equals( final Object obj )
		{
			if( ( obj instanceof AxisProperties) == false )
			{
				return false;
			}
			
			AxisProperties p = (AxisProperties)obj;
			if( p.visibleAxisBreakSymbolList.equals(this.visibleAxisBreakSymbolList) == false )
			{
				return false;
			}

			return true;
		}
	}


	/**
	 * 
	 */
	public SGProperties getProperties()
	{
		AxisProperties wp = new AxisProperties();
		wp.visibleAxisBreakSymbolList = this.getVisibleChildList();
		return wp;
	}



	/**
	 * 
	 */
	public boolean setProperties( final SGProperties p )
	{
		if( ( p instanceof AxisProperties ) == false ) return false;

		AxisProperties wp = (AxisProperties)p;
		final boolean flag = this.setVisibleChildList( wp.visibleAxisBreakSymbolList );
		if( !flag )
		{
			return false;
		}

		return true;
	}



	/**
	 * Create copies of the focused objects.
	 * @return
	 */
	public boolean duplicateFocusedObjects()
	{
		final int ox = (int)( this.mMagnification*OFFSET_DUPLICATED_OBJECT_X );
		final int oy = (int)( this.mMagnification*OFFSET_DUPLICATED_OBJECT_Y );

		ArrayList cList = this.duplicateObjects();
		for( int ii=0; ii<cList.size(); ii++ )
		{
			// duplicate
			AxisBreakSymbol el = (AxisBreakSymbol)cList.get(ii);

			// translate the duplicate
			el.translate( ox, oy );

			// set selected
			el.setSelected(true);

			// add to the list
			this.addToList(el);

			// initialize history
			el.initPropertiesHistory();
		}

		if( cList.size()!=0 )
		{
			this.setChanged(true);
		}
		
//		this.repaint();

		return true;
	}



	/**
	 * Paste the objects.
	 * @param list of the objects to be pasted
	 * @return true:succeeded, false:failed
	 */
	public boolean paste( ArrayList list )
	{
		final float mag = this.getMagnification();
		final int ox = (int)( mag*OFFSET_DUPLICATED_OBJECT_X );
		final int oy = (int)( mag*OFFSET_DUPLICATED_OBJECT_Y );

		int cnt = 0;
		for( int ii=0; ii<list.size(); ii++ )
		{
			Object obj = list.get(ii);
			if( obj instanceof AxisBreakSymbol )
			{
				AxisBreakSymbol abs = (AxisBreakSymbol)obj;

				// translate the instance to be pasted
				abs.translate( ox, oy );

				SGProperties p = abs.getProperties();
				
				AxisBreakSymbol el = new AxisBreakSymbol();
				el.setMagnification(mag);
				el.setProperties(p);

				el.mXAxis = this.mAxisElement.getAxisInCube( abs.mTempXAxis );
				el.mYAxis = this.mAxisElement.getAxisInCube( abs.mTempYAxis );

				el.setDrawingElementsLocation();
				el.setAxisValue();

				// add to the list
				this.addToList(el);

				// initialize history
				el.initPropertiesHistory();

				cnt++;
			}
		}
		
		if( cnt!=0 )
		{
			this.setChanged(true);
		}

//		this.repaint();

		return true;
	}


	
	/**
	 * 
	 */
	class AxisBreakSymbol extends SGAxisBreakSymbol2D
		implements ActionListener, SGIAxisBreakSymbolDialogObserver,
			SGIUndoable, SGISelectable, SGIMovable, SGICopiable, SGINode, SGIDisposable
	{

		/**
		 * ID number.
		 */
		private int mID;


		/**
		 * 
		 */
		private SGAxis mXAxis = null;
		
		
		/**
		 * 
		 */
		private SGAxis mYAxis = null;

		
		/**
		 * 
		 */
		private double mXValue;

		
		/**
		 * 
		 */
		private double mYValue;

		
		/**
		 * 
		 */
		private boolean mFrameFlag = false;


		/**
		 * 
		 */
		private SGProperties mTemporaryProperties = null;


		/**
		 * Pop-up menu.
		 */
		private JPopupMenu mPopupMenu = new JPopupMenu();


		/**
		 * 
		 */
		AxisBreakSymbol()
		{
			super();
			this.init();
		}


		/**
		 * 
		 */
		AxisBreakSymbol(
			final float length,
			final float interval,
			final float dist,
			final float angle,
			final boolean horizontal )
		{
			super(length,interval,dist,angle,horizontal);
			this.init();
		}


		/**
		 * 
		 * @return
		 */
		private boolean init()
		{
			this.createPopupMenu();
			return true;
		}


		/**
		 * 
		 */
		public void dispose()
		{
			super.dispose();
			this.mPopupMenu = null;
			this.mTemporaryProperties = null;
			this.mUndoManager.dispose();
			this.mUndoManager = null;
			this.mXAxis = null;
			this.mYAxis = null;
		}


		public String toString()
		{
			return "AxisBreak:"+this.mID;
		}

		public void finalize()
		{
//			System.out.println( "finalize:"+this );
		}


		/**
		 * 
		 */
		public int getXAxisConfiguration()
		{
			return SGAxisBreakElement.this.mAxisElement.getConfigurationInPlane( this.mXAxis );
		}


		/**
		 * 
		 */
		public int getYAxisConfiguration()
		{
			return SGAxisBreakElement.this.mAxisElement.getConfigurationInPlane( this.mYAxis );
		}


		/**
		 * 
		 */
		public void setXAxisConfiguration( final int config )
		{
			this.mXAxis = this.getAxis(config);
		}

		/**
		 * 
		 */
		public void setYAxisConfiguration( final int config )
		{
			this.mYAxis = this.getAxis(config);
		}

		private SGAxis getAxis( final int config )
		{
			return mAxisElement.getAxisInPlane( config );
		}


		/**
		 * 
		 */
		public double getXValue()
		{
			return this.mXValue;
		}


		/**
		 * 
		 */
		public double getYValue()
		{
			return this.mYValue;
		}


		/**
		 * 
		 */
		public Color getInnerColor()
		{
			return this.getColor(0);
		}


		/**
		 * 
		 */
		public float getLineWidth( final String unit )
		{
			return (float)SGUtilityText.convertFromPoint( this.getLineWidth(), unit );
		}

		/**
		 * 
		 */
		public float getLength( final String unit )
		{
			return (float)SGUtilityText.convertFromPoint( this.getLength(), unit );
		}


		/**
		 * 
		 */
		public float getInterval( final String unit )
		{
			return (float)SGUtilityText.convertFromPoint( this.getInterval(), unit );
		}


		/**
		 * 
		 * @param value
		 */
		public void setXValue( final double value )
		{
			this.mXValue = value;
		}


		/**
		 * 
		 * @param value
		 */
		public void setYValue( final double value )
		{
			this.mYValue = value;
		}


		/**
		 * 
		 */
		public void setLineWidth( final float width, final String unit )
		{
			this.setLineWidth( (float)SGUtilityText.convertToPoint( width, unit ) );
		}


		/**
		 * 
		 */
		public void setLength( final float len, final String unit )
		{
			this.setLength( (float)SGUtilityText.convertToPoint( len, unit ) );
		}


		/**
		 * 
		 */
		public void setInterval( final float interval, final String unit )
		{
			this.setInterval( (float)SGUtilityText.convertToPoint( interval, unit ) );
		}


		/**
		 * 
		 * @param config
		 * @param value
		 * @return
		 */
		public boolean hasValidXValue( final int config, final Number value )
		{
			final SGAxis axis = (config==-1) ? this.mXAxis : mAxisElement.getAxisInPlane( config );
			final double v = (value!=null) ? value.doubleValue() : this.getXValue();
			return axis.isValidValue(v);
		}

		/**
		 * 
		 * @param config
		 * @param value
		 * @return
		 */
		public boolean hasValidYValue( final int config, final Number value )
		{
			final SGAxis axis = (config==-1) ? this.mYAxis : mAxisElement.getAxisInPlane( config );
			final double v = (value!=null) ? value.doubleValue() : this.getYValue();
			return axis.isValidValue(v);
		}


		/**
		 * 
		 */
		public void setInnerColor( final Color cl )
		{
			this.setColor( cl );
		}


		/**
		 * 
		 */
		public Object copy()
		{
			AxisBreakSymbol el = new AxisBreakSymbol();
			el.setMagnification( this.mMagnification );
			el.setProperties( this.getProperties() );
			el.setLocation( this.getX(), this.getY() );
			el.mTempXAxis = mAxisElement.getConfigurationInCube( this.mXAxis );
			el.mTempYAxis = mAxisElement.getConfigurationInCube( this.mYAxis );
			return el;
		}


		private int mTempXAxis = -1;
		private int mTempYAxis = -1;


		/**
		 * Returns a property dialog.
		 * @return property dialog
		 */
		public SGPropertyDialog getPropertyDialog()
		{
			return SGAxisBreakElement.this.mDialog;
		}



		/**
		 * Returns a list of child nodes.
		 * @return a list of chid nodes
		 */
		public ArrayList getChildNodes()
		{
			return new ArrayList();
		}


		/**
		 * 
		 */
		public String getClassDescription()
		{
			return "";
		}


		/**
		 * 
		 * @return
		 */
		public String getInstanceDescription()
		{
			SGIAxisElement aElement = SGAxisBreakElement.this.mAxisElement;
			String xAxis = aElement.getAxisLocation( this.mXAxis );
			String yAxis = aElement.getAxisLocation( this.mYAxis );

			String str = "";
			str += this.mID + ": " + xAxis + ", " + yAxis + ", ";
			str += "( X=" + this.mXValue + ", Y=" + this.mYValue + " )";
			return str;
		}


		/**
		 * 
		 * @return
		 */
		private JPopupMenu getPopupMenu()
		{
			return this.mPopupMenu;
		}
		
		
		/**
		 * 
		 */
		private boolean createPopupMenu()
		{
			JPopupMenu p = this.mPopupMenu;
			
			p.setBounds( 0, 0, 100, 100 );

			p.add( new JLabel( "  -- AxisBreak --" ) );
			p.addSeparator();

			SGUtility.addItem( p, this, MENUCMD_MOVE_TO_FRONT );
			SGUtility.addItem( p, this, MENUCMD_MOVE_TO_BACK );

			p.addSeparator();

			SGUtility.addItem( p, this, MENUCMD_CUT );
			SGUtility.addItem( p, this, MENUCMD_COPY );
			SGUtility.addItem( p, this, MENUCMD_PASTE );

			p.addSeparator();

			SGUtility.addItem( p, this, MENUCMD_DELETE );
			SGUtility.addItem( p, this, MENUCMD_DUPLICATE );

			p.addSeparator();

			SGUtility.addItem( p, this, MENUCMD_PROPERTY );

			return true;
		}



		/**
		 * 
		 */
		public SGTuple2f getLocation()
		{
			final float x = this.mMagnification*this.mLocation.x + mGraphRectX;
			final float y = this.mMagnification*this.mLocation.y + mGraphRectY;
			return new SGTuple2f(x,y);
		}


		/**
		 * 
		 */
		public void setLocation( final float x, final float y)
		{
			final float dx = x - this.getX();
			final float dy = y - this.getY();

			this.mLocation.x = ( x - mGraphRectX )/this.mMagnification;
			this.mLocation.y = ( y - mGraphRectY )/this.mMagnification;
			this.create();
		}


		/**
		 * 
		 */
		public void translate( final float dx, final float dy )
		{
			this.setLocation( this.getX() + dx, this.getY() + dy );
			this.setAxisValue();
			this.setDrawingElementsLocation();
		}
		
		
		/**
		 * 
		 * @param g2d
		 */
		private void drawBoundingBox( final Graphics2D g2d )
		{
			g2d.setStroke( new BasicStroke(1) );
			g2d.setPaint( Color.BLACK );
			g2d.draw( this.getElementBounds() );
		}



		/**
		 * 
		 * @return
		 */
		private ArrayList getAnchorPointList()
		{
			ArrayList pList1 = this.getAnchorPointList( this.mCurve1 );
			ArrayList pList2 = this.getAnchorPointList( this.mCurve2 );
			ArrayList list = new ArrayList();
			list.addAll( pList1 );
			list.addAll( pList2 );
			return list;
		}


		/**
		 * 
		 * @return
		 */
		private ArrayList getAnchorPointList( final Shape sh )
		{
			ArrayList pList = new ArrayList();

			PathIterator itr = sh.getPathIterator(null);
			ArrayList list = SGUtilityJava2D.getSegmentList( itr );
			Point2D start = (Point2D)list.get(0);
			Point2D end = (Point2D)list.get(list.size()-1);
			Point2D middle = new Point2D.Float();
			middle.setLocation(
				( start.getX() + end.getX() )/2.0,
				( start.getY() + end.getY() )/2.0
			);
			pList.add( start );
			pList.add( end );
			pList.add( middle );

			return pList;
		}
	

		/**
		 * 
		 * @param e
		 */
		public void actionPerformed( final ActionEvent e )
		{
			final String command = e.getActionCommand();
			final Object source = e.getSource();

			if( command.equals( MENUCMD_PROPERTY ) )
			{
				SGAxisBreakElement.this.setPropertiesOfSelectedObjects();
			}
			else if(
				command.equals( MENUCMD_COPY )
				| command.equals( MENUCMD_CUT )
				| command.equals( MENUCMD_PASTE )
				| command.equals( MENUCMD_DELETE )
				| command.equals( MENUCMD_DUPLICATE )
				| command.equals( MENUCMD_MOVE_TO_FRONT )
				| command.equals( MENUCMD_MOVE_TO_BACK )
			)
			{
				notifyToListener( command );
			}

		}


		private boolean mValidFlag = true;

		public boolean isValid()
		{
			return this.mValidFlag;
		}

		public void setValid( final boolean b )
		{
			this.mValidFlag = b;
		}


		/**
		 * 
		 */
		public boolean setDrawingElementsLocation()
		{
//System.out.println(mGraphRectX+"  "+mGraphRectY);
			final float x = calcLocation( this.mXValue, this.mXAxis, true );
			final float y = calcLocation( this.mYValue, this.mYAxis, false );

			if( Float.isNaN(x) | Float.isNaN(y) )
			{
				this.setValid( false );
				return false;
			}
			else
			{
				this.setValid( true );
			}

			this.setLocation(x,y);
			return true;
		}


		
		/**
		 * 
		 * @return
		 */
		public boolean setAxisValue()
		{
			final double xValue = calcValue( this.getX(), this.mXAxis, true );
			final double yValue = calcValue( this.getY(), this.mYAxis, false );

			this.mXValue = getNumberInRangeOrder( xValue, this.mXAxis );
			this.mYValue = getNumberInRangeOrder( yValue, this.mYAxis );
			
			return true;
		}
		
		
		
		/**
		 * @return
		 */
		public String getTagName()
		{
			return TAG_NAME_AXIS_BREAK_SYMBOL;
		}
		

		/**
		 * 
		 * @param document
		 * @return
		 */
		public Element createElement( final Document document )
		{
			Element el = document.createElement( this.getTagName() );
			if( this.writeProperty(el) == false )
			{
				return null;
			}
			return el;
		}
		
		
		/**
		 * 
		 * @param el
		 * @return
		 */
		public boolean writeProperty( final Element el )
		{
			String cm = SGUtilityNumber.cm;
			String pt = SGUtilityNumber.pt;
			String degree = SGUtilityNumber.degree;

			el.setAttribute( KEY_X_VALUE, Double.toString( this.mXValue ) );
			el.setAttribute( KEY_Y_VALUE, Double.toString( this.mYValue ) );
			el.setAttribute( KEY_X_AXIS_POSITION, mAxisElement.getAxisLocation( this.mXAxis ) );
			el.setAttribute( KEY_Y_AXIS_POSITION, mAxisElement.getAxisLocation( this.mYAxis ) );

			el.setAttribute( KEY_LENGTH, Float.toString( this.mLength*SGIConstants.CM_POINT_RATIO ) + cm );
			el.setAttribute( KEY_INTERVAL, Float.toString( this.mInterval*SGIConstants.CM_POINT_RATIO ) + cm );
			el.setAttribute( KEY_DISTORTION, Float.toString( this.mDistortion ) );
			el.setAttribute( KEY_ANGLE, Float.toString( this.mAngle/RADIAN_DEGREE_RATIO ) + degree );
			el.setAttribute( KEY_LINE_WIDTH, Float.toString( this.mLineWidth ) + pt );
			el.setAttribute( KEY_HORIZONTAL, Boolean.toString( this.mForHorizontalAxisFlag ) );
			el.setAttribute( KEY_LINE_COLOR, SGUtilityText.getColorString( this.mLineColor ) );
			el.setAttribute( KEY_INNER_COLOR, SGUtilityText.getColorListString( this.getColorList() ) );
			return true;
		}
		
		

		/**
		 * 
		 */
		public boolean readProperty( final Element element )
		{
			SGAxisBreakSymbol.AxisBreakSymbolProperties p = this.readProperty_(element);
			if( p==null )
			{
				return false;
			}
			if( this.setProperties(p) == false )
			{
				return false;
			}
			return true;
		}

		

		/**
		 * 
		 * @param element
		 * @return
		 */
		public SGAxisBreakSymbol.AxisBreakSymbolProperties readProperty_( final Element el )
		{
			
			String str = null;
			Number num = null;
			Color cl = null;
			Boolean b = null;
			ArrayList list = null;


			// x axis
			str = el.getAttribute( KEY_X_AXIS_POSITION );
			if( str.length()==0 )
			{
				return null;
			}
			final int configX = mAxisElement.getConfigurationInCube( str );
			SGAxis xAxis = mAxisElement.getAxisInCube( configX );
			if( xAxis==null )
			{
				return null;
			}
			
			
			// y axis
			str = el.getAttribute( KEY_Y_AXIS_POSITION );
			if( str.length()==0 )
			{
				return null;
			}
			final int configY = mAxisElement.getConfigurationInCube( str );
			SGAxis yAxis = mAxisElement.getAxisInCube( configY );
			if( yAxis==null )
			{
				return null;
			}


			// x value
			str = el.getAttribute( SGAxisBreakSymbol.KEY_X_VALUE );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getDouble(str);
			if( num==null )
			{
				return null;
			}
			final double xValue = num.doubleValue();
			if( xAxis.isValidValue( xValue ) == false )
			{
				return null;
			}


			// y value
			str = el.getAttribute( SGAxisBreakSymbol.KEY_Y_VALUE );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getDouble(str);
			if( num==null )
			{
				return null;
			}
			final double yValue = num.doubleValue();
			if( yAxis.isValidValue( yValue ) == false )
			{
				return null;
			}
			

			// width
			str = el.getAttribute( SGAxisBreakSymbol.KEY_LENGTH );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getLengthInPoint( str );
			if( num==null )
			{
				return null;
			}
			final float length = num.floatValue();
			
			
			// interval
			str = el.getAttribute( SGAxisBreakSymbol.KEY_INTERVAL );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getLengthInPoint( str );
			if( num==null )
			{
				return null;
			}
			final float interval = num.floatValue();


			// distortion
			str = el.getAttribute( SGAxisBreakSymbol.KEY_DISTORTION );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getFloat(str);
			if( num==null )
			{
				return null;
			}
			final float distortion = num.floatValue();


			// angle
			str = el.getAttribute( SGAxisBreakSymbol.KEY_ANGLE );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getFloat(str,degree);
			if( num==null )
			{
				return null;
			}
			final float angle = num.floatValue()*RADIAN_DEGREE_RATIO;
			
			
			// line width
			str = el.getAttribute( SGAxisBreakSymbol.KEY_LINE_WIDTH );
			if( str.length()==0 )
			{
				return null;
			}
			num = SGUtilityText.getLengthInPoint( str );
			if( num==null )
			{
				return null;
			}
			final float lineWidth = num.floatValue();


			// horizontal
			str = el.getAttribute( SGAxisBreakSymbol.KEY_HORIZONTAL );
			if( str.length()==0 )
			{
				return null;
			}
			b = SGUtilityText.getBoolean(str);
			if( b==null )
			{
				return null;
			}
			final boolean horizontal = b.booleanValue();
		

			// line color
			str = el.getAttribute( SGAxisBreakSymbol.KEY_LINE_COLOR );
			if( str.length()==0 )
			{
				return null;
			}
			cl = SGUtilityText.getColorFromString(str);
			if( cl==null )
			{
				return null;
			}
			final Color lineColor = cl;
		

			// inner color
			str = el.getAttribute( SGAxisBreakSymbol.KEY_INNER_COLOR );
			if( str.length()==0 )
			{
				return null;
			}
			list = SGUtilityText.getColorList(str);
			if( list==null )
			{
				return null;
			}
			final ArrayList innerColorList = list;


			AxisBreakSymbolWithAxesProperties p = new AxisBreakSymbolWithAxesProperties();

			p.setXValue( xValue );
			p.setYValue( yValue );
			p.mXAxis = xAxis;
			p.mYAxis = yAxis;
			p.setLength(length);
			p.setInterval(interval);
			p.setDistortion(distortion);
			p.setAngle(angle);
			p.setLineWidth(lineWidth);
			p.setHorizontal(horizontal);
			p.setLineColor(lineColor);
			p.setColor(innerColorList);

			
			//
			p.setVisible(true);
		
			return p;
		}

		
		// e|IuWFNg쐬
		public boolean prepare()
		{
			this.mTemporaryProperties = this.getProperties();
			return true;
		}


		/**
		 * 
		 */
		public boolean commit()
		{
			SGProperties p = this.getProperties();
			if( p==null )
			{
				return false;
			}

			if( p.equals(this.mTemporaryProperties) == false )
			{
				this.setChanged(true);
			}
			
			if( this.create() == false )
			{
				return false;
			}
			
			if( this.setDrawingElementsLocation() == false )
			{
				return false;
			}

//
			repaint();

			return true;
		}


		/**
		 * 
		 */
		public boolean cancel()
		{
			if( this.setProperties( this.mTemporaryProperties ) == false )
			{
				return false;
			}
			if( this.create() == false )
			{
				return false;
			}
			if( this.setDrawingElementsLocation() == false )
			{
				return false;
			}
			repaint();
			return true;
		}


		/**
		 * 
		 */
		public boolean preview()
		{
			if( this.create() == false )
			{
				return false;
			}
			
			if( this.setDrawingElementsLocation() == false )
			{
				return false;
			}

			repaint();

			return true;
		}



		private SGUndoManager mUndoManager = new SGUndoManager(this);

		/**
		 * 
		 * @return
		 */
		public SGProperties getMemento()
		{
			return this.getProperties();
		}


		/**
		 * 
		 * @param p
		 * @return
		 */
		public boolean setMemento( SGProperties p )
		{
			return this.setProperties(p);
		}


		/**
		 * 
		 * @return
		 */
		public boolean isUndoable()
		{
			return this.mUndoManager.isUndoable();
		}

	
		/**
		 * 
		 * @return
		 */
		public boolean isRedoable()
		{
			return this.mUndoManager.isRedoable();
		}


		/**
		 * 
		 */
		public boolean initPropertiesHistory()
		{
			return this.mUndoManager.initPropertiesHistory();
		}


		/**
		 * AhDs
		 */
		public boolean setMementoBackward()
		{
			if( this.mUndoManager.setMementoBackward() == false )
			{
				return false;
			}

			this.create();
			this.setDrawingElementsLocation();

			return true;
		}


		/**
		 * hDs
		 */
		public boolean setMementoForward()
		{
			if( this.mUndoManager.setMementoForward() == false )
			{
				return false;
			}

			this.create();
			this.setDrawingElementsLocation();

			return true;
		}



		/**
		 * AhD˗
		 */
		public boolean undo()
		{
			return this.setMementoBackward();
		}

	
	
		/**
		 * hD˗
		 */
		public boolean redo()
		{
			return this.setMementoForward();
		}


		/**
		 * ̍XV̎sBAhDΏۂ̑삪sꂽƂɁA^ɌĂ΂B
		 */
		public boolean updateHistory()
		{
			return this.mUndoManager.updateHistory();
		}


		/**
		 * 
		 */
		public void initUndoBuffer()
		{
			this.mUndoManager.initUndoBuffer();
		}


		/**
		 * 
		 */
		public boolean isChanged()
		{
			return this.mChangedFlag;
		}


		private boolean mChangedFlag = false;

		
		public void setChanged( final boolean b )
		{
			this.mChangedFlag = b;
		}


		/**
		 * 
		 *
		 */
		public void notifyToRoot()
		{
			SGAxisBreakElement.this.notifyToRoot();
		}

		
		
		
		/**
		 * 
		 */
		public boolean setProperties( SGProperties p )
		{

			if( ( p instanceof AxisBreakSymbolWithAxesProperties ) == false ) return false;

			if( super.setProperties(p) == false ) return false;

			AxisBreakSymbolWithAxesProperties ap = (AxisBreakSymbolWithAxesProperties)p;


			final Double x = ap.getXValue();
			if( x==null )
			{
				return false;
			}
			this.mXValue = x.doubleValue();
			
			
			final Double y = ap.getYValue();
			if( y==null )
			{
				return false;
			}
			this.mYValue = y.doubleValue();
			
			this.mXAxis = ap.mXAxis;
			this.mYAxis = ap.mYAxis;
			
			return true;
		}



		/**
		 * 
		 * @return
		 */
		public SGProperties getProperties()
		{
			final AxisBreakSymbolWithAxesProperties p = new AxisBreakSymbolWithAxesProperties();
			if( this.getProperties(p) == false )
			{
				return null;
			}
			return p;
		}

		
		/**
		 * 
		 * @param p
		 * @return
		 */
		public boolean getProperties( final SGProperties p )
		{
			if( ( p instanceof AxisBreakSymbolWithAxesProperties ) == false )
			{
				return false;
			}

			if( super.getProperties(p) == false )
			{
				return false;
			}

			final AxisBreakSymbolWithAxesProperties dp = (AxisBreakSymbolWithAxesProperties)p;
			dp.setXValue( this.mXValue );
			dp.setYValue( this.mYValue );
			dp.mXAxis = this.mXAxis;
			dp.mYAxis = this.mYAxis;

			return true;
		}



		/**
		 * Flag whether this object is focused.
		 */
		private boolean mSelectedFlag = false;


		/**
		 * Get the flag as a focused object.
		 * @return whether this object is focused.
		 */
		public boolean isSelected()
		{
			return this.mSelectedFlag;
		}


		/**
		 * Set the flag as a focused object.
		 * @param b focused
		 */
		public void setSelected( final boolean b )
		{
			this.mSelectedFlag = b;
		}

	}


	
	/**
	 * Property of Axis Break Symbol.
	 */
	public static class AxisBreakSymbolWithAxesProperties
		extends AxisBreakSymbolProperties
	{

		private double mXValue = 0.0;
		private double mYValue = 0.0;
		private SGAxis mXAxis = null;
		private SGAxis mYAxis = null;

		/**
		 * 
		 */
		public AxisBreakSymbolWithAxesProperties()
		{
			super();
		}


		/**
		 * 
		 */
		public boolean equals( final Object obj )
		{
			if( ( obj instanceof AxisBreakSymbolWithAxesProperties ) == false )
			{
				return false;
			}

			if( super.equals(obj) == false ) return false;

			AxisBreakSymbolWithAxesProperties p = (AxisBreakSymbolWithAxesProperties)obj;

			if( p.mXValue!=this.mXValue ) return false;
			if( p.mYValue!=this.mYValue ) return false;
			if( this.mXAxis.equals( p.mXAxis ) == false ) return false;
			if( this.mYAxis.equals( p.mYAxis ) == false ) return false;

			return true;
		}


		public Double getXValue()
		{
			return new Double( this.mXValue );
		}
		
		public Double getYValue()
		{
			return new Double( this.mYValue );
		}

		public SGAxis getXAxis()
		{
			return this.mXAxis;
		}

		public SGAxis getYAxis()
		{
			return this.mYAxis;
		}

		public boolean setXValue( final double value )
		{
			this.mXValue = value;
			return true;
		}

		public boolean setYValue( final double value )
		{
			this.mYValue = value;
			return true;
		}

		private boolean setXAxis( final SGAxis axis )
		{
			if( axis==null )
			{
				throw new IllegalArgumentException("axis==null");
			}
			this.mXAxis = axis;
			return true;
		}

		private boolean setYAxis( final SGAxis axis )
		{
			if( axis==null )
			{
				throw new IllegalArgumentException("axis==null");
			}
			this.mYAxis = axis;
			return true;
		}

	}


}

