# encoding: utf-8

require 'rubyplus'
require 'ropt'
require 'ini'

require 'rumix/const'
require 'rumix/function'

module Rumix
	class Config
		RUBY_18 = :ruby_18
		RUBY_191 = :ruby_191
	
		attr_accessor :src_dir, :ruby_type, :dest_dir, :ext_dest_type, :ext_dest_dir_manual
		attr_accessor :man_type, :shell_type, :installing_tool_ids
		bool_attr_accessor :real_operating, :force_overwriting, :add_start_menu, :add_path_env
		
		def initialize(dest_dir = nil)
			clear
			@dest_dir = dest_dir
		end
		
		public
		
		def ext_dest_dir
			case @ext_dest_type
			when :ruby
				File.join(@dest_dir, 'ruby/bin')
			when :system
				Rumix.system_folder_path
			when :manual
				@ext_dest_dir_manual
			end
		end
		
		def tool_list
			case @ruby_type
			when RUBY_18
				TOOL_LIST_RUBY18
			when RUBY_191
				TOOL_LIST_RUBY191
			end
		end
		
		def clear
			@src_dir = 'package/'
			@dest_dir = nil
			@real_operating = true
			@force_overwriting = false
			@ruby_type = nil
			@ext_dest_type = nil
			@ext_dest_dir_manual = nil
			@man_type = nil
			@shell_type = nil
			@add_start_menu = false
			@add_path_env = false
			@installing_tool_ids = []
			
			return self
		end
		
		def load_ini_file(path)
			ini = INI.parse_file(path)
			case ini['General']['Ruby']
			when '1.8'
				@ruby_type = RUBY_18
			when '1.9.1'
				@ruby_type = RUBY_191
			else
				raise "#{path} : unknown installing-ruby type : #{ini['General']['Ruby']}"
			end
			
			return self
		end
		
		def ruby_src_zip_name
			case ruby_type
			when RUBY_18
				'ruby18.zip'
			when RUBY_191
				'ruby191.zip'
			when nil
				nil
			else
				raise "unknown ruby type : #{@installing_ruby_type}"
			end
		end
		
		def operation_list
			re = []
			case @ruby_type
			when RUBY_18
				re << 'ruby 1.8.7 のインストール'
			when RUBY_191
				re << 'ruby 1.9.1 のインストール'
			end
			
			if ext_dest_dir then
				re << "外部ライブラリのインストール -> #{ext_dest_dir}"
			end


			case @man_type
			when :chm
				re << "HTML Help版リファレンスのインストール"
			when :html
				re << "分割HTML版リファレンスのインストール"
			end
			
			case @shell_type
			when :nyacus
				re << "NYACUS 2.2a のインストール"
			when :nyados
				re << "NYADOS 2.2a のインストール"
			end
			
			@installing_tool_ids.each do |tool_id|
				found = tool_list.find{|x| x.id == tool_id}
				re << "#{found[1]} のインストール"
			end


			if add_path_env? then
				re << "環境変数PATHに ruby.exe のパスを追加"
			end
			
			if add_start_menu? then
				re << "スタートメニューにRumixを追加"
			end
			
			re
		end
		
		def parse(argv)
			clear
			
			re = ROpt.parse(argv, 'd:ne:fi:h', 'dest:', 'no-op', 'ext:', 'force', 'ini:', 'help')
			if re then
				# help
				if re['help'] || re['h'] then
					puts(<<ECHO)
Rumix #{Rumix::VERSION} - Ruby Starter Package with Installer

Usage: rumix_cui.exe [options] [install-targets]

Options:
  -d  --dest    specifies path for installing. (REQUIRED)
  -n  --no-op   output message like operating, but does not operate.
  -e  --ext     specifies a directory path of external libraries. 
                 ('system' / 'ruby' / any path)
  -f  --force   without skipping, even if same or newer file already exists.
  -i  --ini     specifies using ini-file. (default: rumix.ini)
  -h  --help    show this help.

Install Targets:
  ruby, rubyman_chm, rubyman_html, nyacus, nyados,
  rubygems, rake, exerb, path_env, start_menu

Addition:
  - You can omit under-bar of install targets. (pathenv == path_env)
  - On multiple specifing targets, install only one-side.
    For example, 'rumix_install_cui.exe nyados nyacus' => maybe install only nyados.
ECHO
					exit
				end

				targets = re.args
				if targets.empty? then
					$stderr.puts "No install-target."
					$stderr.puts
					$stderr.puts CUI_HELP_SENTENCE
					exit
				end
				
				targets.each do |target|
				
				
					case target.downcase
					when 'ruby'
						load_ini_file(re['ini'] || re['i'] || 'rumix.ini')
	
					when 'rubyman_chm', 'rubymanchm'
						@man_type = :chm
					when 'rubyman_html', 'rubymanhtml'
						@man_type = :html
					when 'nyacus'
						@shell_type = :nyacus
					when 'nyados'
						@shell_type = :nyados

					when 'path_env', 'pathenv'
						@add_path_env = true
					when 'start_menu', 'startmenu'
						@add_start_menu = true

					when 'rubygems', 'rake', 'exerb'
						@installing_tool_ids << target.downcase.to_sym
					else
						$stderr.puts "Target '#{target}' is unknown."
						exit
					end
				end
				
				if re['no-op'] || re['n'] then
					@real_operating = false
				end
				
				if re['force'] || re['f'] then
					@force_overwriting = true
				end
				
				dest = re['dest'] || re['d']
				unless dest then
					$stderr.puts("Required specifing a path for installing.")
					$stderr.puts("Please add option '-dest PATH' or '-d PATH'")
					$stderr.puts
					$stderr.puts(CUI_HELP_SENTENCE)
					exit
				end
				@dest_dir = File.expand_path(dest)


				
				ext_path = re['ext'] || re['e']
				case ext_path
				when /^ruby$/i
					@ext_dest_type = :ruby
				when /^system$/i
					@ext_dest_type = :system
				when nil
					# no act
				else
					@ext_dest_type = :manual
					@ext_dest_dir_manual = ext_path
				end
				
				return self
			else
				return nil
			end
			
			
			
		end
	end
	
	
end
