/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
//
// Client.cc - This is the source code of StructTest-client.
//
// Usage: Client -ORBInitRef NameService=<corbaloc>
//
//  e.g.: Client -ORBInitRef NameService=corbaloc::localhost/NameService
//
//  On startup, Client looks up the object reference from the COS naming service.
//
//  The name which the object is bound to is as follows:
//        id:root   kind:context
//         |
//        id:StructTester kind:object
//

#include <sys/time.h>

#include <iostream>
using namespace std;

#ifdef ORB_IS_RTORB
#include "FloatTest.h"
#else
#include "FloatTest.hh"
#endif

ostream& operator<<(ostream& out, const FloatTest::FloatData& p)
{
	out << "(" << p.x << ")";
	return out;
}

ostream& operator<<(ostream& out, FloatTest::FloatData& p)
{
	return (out << (const FloatTest::FloatData&)p);
}

//////////////////////////////////////////////////////////////////////

class Client
{
public:
	Client(CORBA::ORB_var& orb);
	CORBA::Object_ptr getObject(const CosNaming::Name& name);
protected:
	CosNaming::NamingContext_ptr getRootContext();
private:
	CORBA::ORB_var& orb;
	CosNaming::NamingContext_var root_context;
};

Client::Client(CORBA::ORB_var& _orb): orb(_orb)
{
	this->root_context = getRootContext();
}

CosNaming::NamingContext_ptr
Client::getRootContext()
{
	CORBA::Object_var obj = orb->resolve_initial_references("NameService");
	return CosNaming::NamingContext::_narrow(obj);
}

CORBA::Object_ptr
Client::getObject(const CosNaming::Name& name)
{
	return root_context->resolve(name);
}

//////////////////////////////////////////////////////////////////////

class TesterClient: public Client
{
public:
	TesterClient(CORBA::ORB_var& orb);
	void run();
	void test_set_get();
#if 0
// ̤б
	void test_set_ret();
#endif
	void test_update();
protected:
	FloatTest::Tester_ptr getTester();
private:
	FloatTest::Tester_var tester;
};

TesterClient::TesterClient(CORBA::ORB_var& orb): Client(orb)
{
	this->tester = getTester();
}

FloatTest::Tester_ptr
TesterClient::getTester()
{
	CosNaming::Name name;
	name.length(1);
	name[0].id   = (const char*)"FloatTester";
	name[0].kind = (const char*)"object";

	CORBA::Object_var obj = getObject(name);
	return FloatTest::Tester::_narrow(obj);
}

void 
TesterClient::test_set_get()
{
	cout << "test_set_get:" << endl;

	FloatTest::FloatData p_out;
	tester->get(p_out);
	cout << "  tester->get() ==> " << p_out << endl;

	FloatTest::FloatData p_in;
	p_in.x = 10.12;
	tester->set(p_in);
	cout << "  tester->set(" << p_in << ")" << endl;

	tester->get(p_out);
	cout << "  tester->get() ==> " << p_out << endl << endl;
}

#if 0
// ̤б
void 
TesterClient::test_set_ret()
{
	cout << "test_set_ret:" << endl;

	FloatTest::FloatData p_out;
	p_out = tester->ret();
	cout << "  tester->ret() ==> " << p_out << endl;

	FloatTest::FloatData p_in;
	p_in.x = 20;
	tester->set(p_in);
	cout << "  tester->set(" << p_in << ")" << endl;

	p_out = tester->ret();
	cout << "  tester->ret() ==> " << p_out << endl << endl;
}
#endif

void
TesterClient::test_update()
{
	cout << "test_update:" << endl;

#if 0
// ̤б
	FloatTest::FloatData p_ret = tester->ret();
	cout << "  tester->ret() ==> " << p_ret << endl;
#endif

	FloatTest::FloatData p_out;
	tester->get(p_out);
	cout << "  tester->get() ==> " << p_out << endl;

	FloatTest::FloatData_var p_inout = new FloatTest::FloatData;
	p_inout->x = 0;
	cout << "  FloatData = " << p_inout << endl;

	tester->update(p_inout.inout());
	cout << "  tester->update(FloatData)" << endl;
	cout << "  FloatData = " << p_inout << endl << endl;
}

void 
TesterClient::run()
{
	test_set_get();
#if 0
// ̤б
	test_set_ret();
#endif
	test_update();
}

//////////////////////////////////////////////////////////////////////

int
main (int argc, char **argv) 
{
	try {
		CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);
		TesterClient tester(orb);
		tester.run();
		orb->destroy();
	}
	catch(CosNaming::NamingContext::NotFound& ex) {
		// This exception is thrown if any of the components of the
		// path [contexts or the object] aren't found:
		cerr << "Context not found." << endl;
	}
	catch(CORBA::COMM_FAILURE& ex) {
		cerr << "Caught system exception COMM_FAILURE -- unable to contact the "
				<< "object." << endl;
	}
	catch(CORBA::SystemException&) {
		cerr << "Caught CORBA::SystemException." << endl;
	}
	catch(CORBA::Exception&) {
		cerr << "Caught CORBA::Exception." << endl;
	}
#ifdef ORB_IS_OMNIORB
	catch(omniORB::fatalException& fe) {
		cerr << "Caught omniORB::fatalException:" << endl;
		cerr << "  file: " << fe.file() << endl;
		cerr << "  line: " << fe.line() << endl;
		cerr << "  mesg: " << fe.errmsg() << endl;
	}
#endif
	catch(...) {
		cerr << "Caught unknown exception." << endl;
	}
	return 0;
}
