/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
//
// Server.cc - This is the source code of the StructTest-server.
//
// Usage: Server -ORBInitRef NameService=<corbaloc>
//
//  e.g.: Server -ORBInitRef NameService=corbaloc::localhost/NameService
//
//  On startup, Server registers a servant into the COS naming service.
//
//  The name which the servant is bound to is as follows:
//        id:root   kind:context
//         |
//        id:StructTester kind:object
//

#include <iostream>
using namespace std;

#ifdef ORB_IS_RTORB
#include "BorderTest.h"
#else
#include "BorderTest.hh"
#endif

const string ServerName = "BorderTest-server";

//
// class Tester_impl: implementation class for BorderTest::Tester
//
class Tester_impl : public POA_BorderTest::Tester,
				public PortableServer::RefCountServantBase
{
public:
	inline Tester_impl(){ c.c = 0; s.s = 0; us.us=0; l.l = 0; ul.ul = 0; f.f = 0.0; d.d = 0.0; }

	void get_char(BorderTest::CharData& p);
	void set_char(const BorderTest::CharData& p);

	void get_short(BorderTest::ShortData& p);
	void set_short(const BorderTest::ShortData& p);

	void get_ushort(BorderTest::UShortData& p);
	void set_ushort(const BorderTest::UShortData& p);

	void get_long(BorderTest::LongData& p);
	void set_long(const BorderTest::LongData& p);

	void get_ulong(BorderTest::ULongData& p);
	void set_ulong(const BorderTest::ULongData& p);

	void get_float(BorderTest::FloatData& p);
	void set_float(const BorderTest::FloatData& p);

	void get_double(BorderTest::DoubleData& p);
	void set_double(const BorderTest::DoubleData& p);

private:
	BorderTest::CharData c;
	BorderTest::ShortData s;
	BorderTest::UShortData us;
	BorderTest::LongData l;
	BorderTest::ULongData ul;
	BorderTest::FloatData f;
	BorderTest::DoubleData d;
};

void
Tester_impl::get_char(BorderTest::CharData& p)
{
	p = this->c;
	printf("Tester_impl::get_char() ==> %d\n", c.c);
}

void 
Tester_impl::set_char(const BorderTest::CharData& p)
{
	printf("Tester_impl::set_char(%d)\n", p.c);
	this->c = p;
}

void
Tester_impl::get_short(BorderTest::ShortData& p)
{
	p = this->s;
	printf("Tester_impl::get_short() ==> %d\n", s.s);
}

void 
Tester_impl::set_short(const BorderTest::ShortData& p)
{
	printf("Tester_impl::set_short(%d)\n", p.s);
	this->s = p;
}

void
Tester_impl::get_ushort(BorderTest::UShortData& p)
{
	p = this->us;
	printf("Tester_impl::get_ushort() ==> %u\n", us.us);
}

void 
Tester_impl::set_ushort(const BorderTest::UShortData& p)
{
	printf("Tester_impl::set_ushort(%u)\n", p.us);
	this->us = p;
}

void
Tester_impl::get_long(BorderTest::LongData& p)
{
	p = this->l;
	printf("Tester_impl::get_long() ==> %d\n", l.l);
}

void 
Tester_impl::set_long(const BorderTest::LongData& p)
{
	printf("Tester_impl::set_long(%d)\n", p.l);
	this->l = p;
}

void
Tester_impl::get_ulong(BorderTest::ULongData& p)
{
	p = this->ul;
	printf("Tester_impl::get_ulong() ==> %u\n", ul.ul);
}

void 
Tester_impl::set_ulong(const BorderTest::ULongData& p)
{
	printf("Tester_impl::set_ulong(%u)\n", p.ul);
	this->ul = p;
}

void
Tester_impl::get_float(BorderTest::FloatData& p)
{
	p = this->f;
	printf("Tester_impl::get_float() ==> %e\n", f.f);
}

void 
Tester_impl::set_float(const BorderTest::FloatData& p)
{
	printf("Tester_impl::set_float(%e)\n", p.f);
	this->f = p;
}

void
Tester_impl::get_double(BorderTest::DoubleData& p)
{
	p = this->d;
	printf("Tester_impl::get_double() ==> %e\n", d.d);
}

void 
Tester_impl::set_double(const BorderTest::DoubleData& p)
{
	printf("Tester_impl::set_double(%e)\n", p.d);
	this->d = p;
}

//////////////////////////////////////////////////////////////////////

class Server
{
public:
	Server(CORBA::ORB_var& orb);
	virtual ~Server();
	void registerServant(const CosNaming::Name& name, Tester_impl* servant);
	void run();
private:
	CORBA::ORB_var& orb;
	CosNaming::NamingContext_var root_context;
#ifdef ORB_IS_RTORB
	PortableServer::POA_ptr root_poa;
#else
	PortableServer::POA_var root_poa;
#endif
protected:
	CosNaming::NamingContext_ptr getRootContext();
	PortableServer::POA_ptr getRootPOA();
};

Server::Server(CORBA::ORB_var& _orb): orb(_orb)
{
	root_context = getRootContext();
	root_poa = getRootPOA();
}

Server::~Server()
{
	//poa->destroy(TRUE, TRUE);
}

CosNaming::NamingContext_ptr
Server::getRootContext()
{
#ifdef ORB_IS_RTORB
	CORBA::Object_ptr obj = orb->resolve_initial_references("NameService");
#else
	CORBA::Object_ptr obj = orb->resolve_initial_references("NameService");
#endif
	return CosNaming::NamingContext::_narrow(obj);
}

PortableServer::POA_ptr
Server::getRootPOA()
{
#ifdef ORB_IS_RTORB
	CORBA::Object_ptr obj = orb->resolve_initial_references("RootPOA");
#else
	CORBA::Object_var obj = orb->resolve_initial_references("RootPOA");
#endif
	return PortableServer::POA::_narrow(obj);
}

void
Server::registerServant(const CosNaming::Name& name, Tester_impl* servant)
{
	//PortableServer::ObjectId_var sid = root_poa->activate_object(servant);
	root_poa->activate_object(servant);

	CORBA::Object_ptr ref = servant->_this();

	try {
		root_context->bind(name, ref);
	}
	catch(CosNaming::NamingContext::AlreadyBound& ex) {
		root_context->rebind(name, ref);
	}

	servant->_remove_ref();
}

void
Server::run()
{
	PortableServer::POAManager_var mgr = root_poa->the_POAManager();
	mgr->activate();
	orb->run();
}

//////////////////////////////////////////////////////////////////////

class TesterServer: public Server
{
public:
	TesterServer(CORBA::ORB_var& orb): Server(orb) { registerTester(); }
private:
	void registerTester();
};

void 
TesterServer::registerTester()
{
	CosNaming::Name name;
	name.length(1);
	name[0].id   = (const char*)"BorderTester";
	name[0].kind = (const char*)"object";

	Tester_impl* servant = new Tester_impl();
	registerServant(name, servant);
	cout << "Servant registered as: id=" << name[0].id << ", kind=" << name[0].kind << endl;
}

//////////////////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
	try {
		CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);
		TesterServer server(orb);
		cout << ServerName << " running.." << endl;
		server.run();
		orb->destroy();
	}
	catch(CORBA::ORB::InvalidName& ex) {
		// This should not happen!
		cerr << "Service required is invalid [does not exist]." << endl;
	}
	catch(CORBA::COMM_FAILURE& ex) {
		cerr << "Caught system exception COMM_FAILURE -- unable to contact the "
				<< "naming service." << endl;
	}
	catch(CORBA::SystemException&) {
		cerr << "Caught CORBA::SystemException." << endl;
	}
	catch(CORBA::Exception& ex) {
		cerr << "Caught CORBA::Exception." << endl;
#ifdef ORB_IS_RTORB
		cerr << ex.msg() << endl;
#endif
	}
#ifdef ORB_IS_OMNIORB
	catch(omniORB::fatalException& fe) {
		cerr << "Caught omniORB::fatalException:" << endl;
		cerr << "  file: " << fe.file() << endl;
		cerr << "  line: " << fe.line() << endl;
		cerr << "  mesg: " << fe.errmsg() << endl;
	}
#endif
	catch(...) {
		cerr << "Caught unknown exception." << endl;
	}

	return 0;
}
