/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */

#include "config.h"

#include "orbit-idl-c-backend.h"

#include <string.h>

static void
ci_build_interfaces (OIDL_C_Info *ci,
		     IDL_tree     tree)
{
	if (!tree)
		return;

	switch (IDL_NODE_TYPE (tree)) {
	case IDLN_MODULE:
		ci_build_interfaces (
			ci, IDL_MODULE (tree).definition_list);
		break;
	case IDLN_LIST: {
		IDL_tree sub;
		for (sub = tree; sub; sub = IDL_LIST (sub).next)
			ci_build_interfaces (
				ci, IDL_LIST (sub).data);
		break;
	}
	case IDLN_INTERFACE: {
		char *id;

		id = IDL_ns_ident_to_qstring (IDL_IDENT_TO_NS (
			IDL_INTERFACE (tree).ident), "_", 0);

		fprintf (ci->fh, "\t&%s__iinterface,\n", id);

		g_free (id);

		ci_build_interfaces (
			ci, IDL_INTERFACE(tree).body);
		break;
	}
	default:
		break;
	}
}

static void
ci_build_types (OIDL_C_Info *ci,
		IDL_tree     tree,
		guint       *count)
{
	if (!tree){
		fprintf(stderr, "Empty tree\n");
		return;
	}

	switch (IDL_NODE_TYPE (tree)) {
	case IDLN_MODULE:
		fprintf(stderr, "IDLN_MODULE\n");
		ci_build_types (
			ci, IDL_MODULE (tree).definition_list, count);
		break;
	case IDLN_LIST: {
		fprintf(stderr, "IDLN_LIST\n");
		IDL_tree sub;
		for (sub = tree; sub; sub = IDL_LIST (sub).next)
			ci_build_types (
				ci, IDL_LIST (sub).data, count);
		break;
	}
	case IDLN_INTERFACE:
		fprintf(stderr, "IDLN_INTERFACE\n");
		ci_build_types (
			ci, IDL_INTERFACE(tree).body, count);
		break;
	case IDLN_TYPE_DCL: {
		fprintf(stderr, "IDLN_TYPE_DCL\n");
	    IDL_tree sub;
	    for (sub = IDL_TYPE_DCL (tree).dcls; sub; sub = IDL_LIST (sub).next) {
		IDL_tree ent = IDL_LIST (sub).data;
		gchar *id;

		id = orbit_cbe_get_typespec_str (ent);

		fprintf (ci->fh, "\tTC_%s,\n", id);
		(*count)++;

		g_free (id);
	    }

	    break;
	}
	case IDLN_TYPE_STRUCT: {
		fprintf(stderr, "IDLN_TYPE_STRUCT\n");
		gchar *id;
		IDL_tree l;

		id = orbit_cbe_get_typespec_str (tree);

		fprintf (ci->fh, "\tTC_%s,\n", id);
		(*count)++;

		g_free (id);

		/* check for nested structs/enums */
		for (l = IDL_TYPE_STRUCT (tree).member_list; l; l = IDL_LIST (l).next) {
			IDL_tree dcl;

			g_assert (IDL_NODE_TYPE (IDL_LIST (l).data) == IDLN_MEMBER);
			dcl = IDL_MEMBER (IDL_LIST (l).data).type_spec;

			/* skip straight declarations */
			if (IDL_NODE_TYPE(dcl) == IDLN_TYPE_STRUCT ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_UNION ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_ENUM)
				ci_build_types (ci, dcl, count);
		}
		break;
	};
	case IDLN_TYPE_UNION: {
		fprintf(stderr, "IDLN_TYPE_UNION\n");
		gchar *id;
		IDL_tree l;

		id = orbit_cbe_get_typespec_str (tree);

		fprintf (ci->fh, "\tTC_%s,\n", id);
		(*count)++;

		g_free (id);

		/* if discriminator is an enum, register it */
		if (IDL_NODE_TYPE (IDL_TYPE_UNION (tree).switch_type_spec) == IDLN_TYPE_ENUM)
			ci_build_types (
				ci, IDL_TYPE_UNION (tree).switch_type_spec, count);

		/* check for nested structs/enums */
		for (l = IDL_TYPE_UNION (tree).switch_body; l; l = IDL_LIST (l).next) {
			IDL_tree dcl;

			g_assert (IDL_NODE_TYPE (IDL_LIST (l).data) == IDLN_CASE_STMT);
			dcl = IDL_MEMBER (
                                IDL_CASE_STMT (IDL_LIST (l).data).element_spec).type_spec;

			if (IDL_NODE_TYPE(dcl) == IDLN_TYPE_STRUCT ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_UNION ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_ENUM)
				ci_build_types (ci, dcl, count);
		}
		break;
	}
	case IDLN_EXCEPT_DCL: {
		fprintf(stderr, "IDLN_EXCEPT_DCL\n");
		gchar *id;
		IDL_tree l;

		id = orbit_cbe_get_typespec_str (tree);

		fprintf (ci->fh, "\tTC_%s,\n", id);
		(*count)++;

		g_free (id);

		/* check for nested structs/enums */
		for (l = IDL_EXCEPT_DCL (tree).members; l; l = IDL_LIST (l).next) {
			IDL_tree dcl;

			g_assert (IDL_NODE_TYPE (IDL_LIST (l).data) == IDLN_MEMBER);
			dcl = IDL_MEMBER (IDL_LIST (l).data).type_spec;

			/* skip straight declarations */
			if (IDL_NODE_TYPE(dcl) == IDLN_TYPE_STRUCT ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_UNION ||
			    IDL_NODE_TYPE(dcl) == IDLN_TYPE_ENUM)
				ci_build_types (ci, dcl, count);
		}
		break;
	}
	case IDLN_TYPE_INTEGER:
	case IDLN_TYPE_ANY:
	case IDLN_TYPE_STRING:
	case IDLN_TYPE_WIDE_STRING:
	case IDLN_TYPE_CHAR:
	case IDLN_TYPE_WIDE_CHAR:
	case IDLN_TYPE_FLOAT:
	case IDLN_TYPE_BOOLEAN:
	case IDLN_TYPE_OCTET:
	case IDLN_TYPE_SEQUENCE:
	case IDLN_TYPE_ENUM:
	case IDLN_IDENT:
	case IDLN_FORWARD_DCL:
	case IDLN_TYPE_OBJECT: {
		fprintf(stderr, "IDLN_?? %d\n", IDL_NODE_TYPE (tree)) ;
		gchar *id;

		id = orbit_cbe_get_typespec_str (tree);

		fprintf (ci->fh, "\tTC_%s,\n", id);
		(*count)++;

		g_free (id);

		break;
	}
	default:
		break;
	}
}

void
rtorb_idl_output_c_cppif (IDL_tree       tree,
			    OIDL_Run_Info *rinfo,
			    OIDL_C_Info   *ci)
{
	guint count;

	fprintf (ci->fh, OIDL_C_WARNING);
	fprintf (ci->fh, "#include <%s.h>\n", ci->c_base_name);

	count = 0;
	ci_build_types (ci, tree, &count);

	ci_build_interfaces (ci, tree);
}
