// -*- C++ -*-
/*!
 * @file RingBuffer.h
 * @brief Defautl Buffer class
 * @date $Date: 2007-12-31 03:08:06 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: RingBuffer.h 1800 2010-01-27 02:21:34Z fsi-katami $
 *
 */

#ifndef RTC_RINGBUFFER_H
#define RTC_RINGBUFFER_H

#include <vector>
#include <algorithm>
#include <iostream>

#include <coil/TimeValue.h>
#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <coil/Condition.h>
#include <coil/stringutil.h>

#include <rtm/BufferBase.h>
#include <rtm/BufferStatus.h>

#define RINGBUFFER_DEFAULT_LENGTH 8
/*!
 * @if jp
 * @namespace RTC
 *
 * @brief RTݡͥ
 *
 * @else
 *
 * @namespace RTC
 *
 * @brief RT-Component
 *
 * @endif
 */
namespace RTC
{
  /*!
   * @if jp
   * @class RingBuffer
   * @brief 󥰥Хåե饹
   * 
   * ꤷĹΥ󥰾ХåեĥХåե饹
   * ХåեΤ˥ǡǼ줿硢ʹߤΥǡϸŤǡ
   * 缡񤭤롣
   * äơХåեˤľΥХåեĹʬΥǡΤݻ롣
   *
   * )ߤμǤϡֺǸ˳ǼǡΤߥХåեɤ߽Фǽ
   *
   * @param DataType Хåե˳Ǽǡ
   *
   * @since 0.4.0
   *
   * @else
   * @class RingBuffer
   * @brief Ring buffer implementation class
   * 
   * This is the buffer implementation class with ring shaped buffer of
   * specified length.
   * If data is stored in the entire buffer, data from now on will be
   * overwritten from old data one by one.
   * Therefore, only the length of latest data is stored in the buffer.
   *
   * Note: In the current implementation, only last stored data can be read
   *       from the buffer.
   *
   * @param DataType Data type to store in the buffer
   *
   * @since 0.4.0
   *
   * @endif
   */
  template <class DataType>
  class RingBuffer
    : public BufferBase<DataType>
  {
  public:
    BUFFERSTATUS_ENUM
    typedef coil::Guard<coil::Mutex> Guard;
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     * ꤵ줿ХåեĹǥХåե롣
     *
     * @param length ХåեĹ
     * 
     * @else
     *
     * @brief Constructor
     * 
     * Constructor.
     * Initialize the buffer by specified buffer length.
     * However, if the specified length is less than two, the buffer should
     * be initialized by two in length.
     *
     * @param length Buffer length
     * 
     * @endif
     */
    RingBuffer(long int length = RINGBUFFER_DEFAULT_LENGTH)
      : m_overwrite(true), m_readback(true),
        m_timedwrite(false), m_timedread(false),
        m_wtimeout(1, 0), m_rtimeout(1, 0),
        m_length(length),
        m_wpos(0), m_rpos(0), m_fillcount(0), m_wcount(0),
        m_buffer(m_length)
    {
      this->reset();
    }
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     * 
     * @else
     *
     * @brief Virtual destractor
     * 
     * Virtual destractor
     * 
     * @endif
     */
    virtual ~RingBuffer(void)
    {
    }
    
    /*!
     * @if jp
     * @brief Хåե
     *
     * coil::Properties Ϳץѥƥˤꡢ
     * Хåե롣
     * ѤǤ륪ץỌ̇̄ϰʲ̤
     *
     * - buffer.length:
     *     ХåեĹʳοͤꤵƤ̵뤵롣
     *     Ǥ˥ХåեѾ֤Ǥ⡢Ĺꤵ줿Τ٤Ƥ
     *     ݥ󥿤롣
     *
     * - buffer.write.full_policy:
     *     񤭤뤫ɤΥݥꥷ
     *     overwrite (), do_nothing (⤷ʤ), block (֥å)
     *     block ꤷ硢 timeout ͤꤹСָ
     *     񤭹ԲǽǤХॢȤ롣
     *     ǥեȤ  overwrite ()
     *
     * - buffer.write.timeout:
     *     ॢȻ֤ [sec] ǻꤹ롣ǥեȤ 1.0 [sec]
     *     1 sec -> 1.0, 1 ms -> 0.001, ॢȤʤ -> 0.0
     *
     * - buffer.read.empty_policy:
     *     ХåեΤȤɤ߽Фݥꥷ
     *     readback (Ǹ), do_nothing (⤷ʤ), block (֥å)
     *     block ꤷ硢 timeout ͤꤹСָ
     *     ɤ߽ФԲǽǤХॢȤ롣
     *     ǥեȤ readback (Ǹ)
     *
     * - buffer.read.timeout:
     *     ॢȻ [sec] ǻꤹ롣ǥեȤ 1.0 [sec]
     *     1sec -> 1.0, 1ms -> 0.001, ॢȤʤ -> 0.0
     *
     * @else
     *
     * @endif
     */
    virtual void init(const coil::Properties& prop)
    {
      initLength(prop);
      initWritePolicy(prop);
      initReadPolicy(prop);
    }
    
    /*!
     * @if jp
     *
     * @brief ХåեĹ
     * 
     * ХåեĹ롣
     * 
     * @return ХåեĹ
     * 
     * @else
     *
     * @brief Get the buffer length
     * 
     * Get the buffer length.
     * 
     * @return Buffer length
     * 
     *
     * @endif
     */
    virtual size_t length(void) const
    {
      Guard guard(m_posmutex);
      return m_length;
    }
    
    /*!
     * @if jp
     *
     * @brief ХåեĹ򥻥åȤ
     * 
     * ХåեĹꤹ롣ԲĤʾNOT_SUPPORTED֤롣
     * μǤ BUFFER_OK ֤ʤ
     * 
     * @return BUFFER_OK: ｪλ
     *         NOT_SUPPORTED: ХåեĹѹԲ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */
    virtual ReturnCode length(size_t n)
    {
      m_buffer.resize(n);
      m_length = n;
      this->reset();
      return ::RTC::BufferStatus::BUFFER_OK; //BUFFER_OK;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեξ֤ꥻåȤ
     * 
     * Хåեɤ߽Фݥ󥿤Ƚ񤭹ߥݥ󥿤ΰ֤ꥻåȤ롣
     * μǤ BUFFER_OK ֤ʤ
     * 
     * @return BUFFER_OK: ｪλ
     *         NOT_SUPPORTED: ꥻåԲǽ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */ 
    virtual ReturnCode reset()
    {
      Guard guard(m_posmutex);
      m_fillcount = 0;
      m_wcount = 0;
      m_wpos = 0;
      m_rpos = 0;
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    
    
    //----------------------------------------------------------------------
    /*!
     * @if jp
     *
     * @brief ХåեθߤνǤΥݥ
     * 
     * ХåեθߤνǤΥݥ󥿤ޤϡnΥݥ󥿤֤
     * 
     * @param  n ߥݥ + n ΰ֤Υݥ 
     * @return ֤߰Υݥ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */ 
    virtual DataType* wptr(long int n = 0) 
    {
      Guard guard(m_posmutex);
      return &m_buffer[(m_wpos + n + m_length) % m_length];
    }
    
    /*!
     * @if jp
     *
     * @brief ߥݥ󥿤ʤ
     * 
     * ߤν񤭹֤߰Υݥ󥿤 n Ŀʤ롣
     * 񤭹߲ǽǿʾοͤꤷ硢PRECONDITION_NOT_MET
     * ֤
     * 
     * @param  n ߥݥ + n ΰ֤Υݥ 
     * @return BUFFER_OK:            ｪλ
     *         PRECONDITION_NOT_MET: n > writable()
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */ 
    virtual ReturnCode advanceWptr(long int n = 1)
    {
      // n > 0 :
      //     n satisfies n <= writable elements
      //                 n <= m_length - m_fillcout
      // n < 0 : -n = n'
      //     n satisfies n'<= readable elements
      //                 n'<= m_fillcount
      //                 n >= - m_fillcount
      if (n > 0 && n > static_cast<long int>(m_length - m_fillcount) ||
          n < 0 && n < static_cast<long int>(-m_fillcount))
        {
          return ::RTC::BufferStatus::PRECONDITION_NOT_MET;
        }

      Guard guard(m_posmutex);
      m_wpos = (m_wpos + n + m_length) % m_length;
      m_fillcount += n;
      m_wcount += n;
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    /*!
     * @if jp
     *
     * @brief Хåե˥ǡ񤭹
     * 
     * Хåե˥ǡ񤭹ࡣ񤭹ߥݥ󥿤ΰ֤ѹʤ
     * μǤϾ BUFFER_OK ֤
     * 
     * @param value 񤭹оݥǡ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     * Always BUFFER_OK will be returned in this implementation.
     *
     * @param value Target data to write.
     *
     * @return BUFFER_OK:    Successful
     *         BUFFER_ERROR: Failed
     *
     * @endif
     */
    virtual ReturnCode put(const DataType& value)
    {
      Guard guard(m_posmutex);
      m_buffer[m_wpos] = value;
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåե˽񤭹
     * 
     * Ϳ줿ǡХåե˽񤭹ࡣ
     *
     * 2(sec)3(nsec)ꤵƤʤ硢Хåեե
     * νߥ⡼ (overwrite, do_nothing, block)  init() 
     * 줿⡼ɤ˽
     *
     * 2(sec) ˰ꤵ줿ϡinit() ꤵ줿⡼
     * ˴ؤ餺block ⡼ɤȤʤꡢХåեե֤Ǥл
     * ֤ޤॢȤ롣3(nsec)ϻꤵʤ0Ȥ
     * 롣ॢԤˡɤ߽Фå¦ǥХåե
     * ɤ߽ФС֥å󥰤ϲǡ񤭹ޤ롣
     *
     * 񤭹߻˥Хåե(empty)֤ǡ̤Υåɤblock⡼
     * ɤ߽ФԤ򤷤Ƥ硢signalȯԤɤ߽Ф¦Υ֥
     * 󥰤롣
     * 
     * @param value 񤭹оݥǡ
     * @param sec   ॢȻ sec  (default -1: ̵)
     * @param nsec  ॢȻ nsec (default 0)
     * @return BUFFER_OK            ｪλ
     *         BUFFER_FULL          Хåեե
     *         TIMEOUT              ߤॢȤ
     *         PRECONDITION_NOT_MET ۾
     * 
     * @else
     *
     * @brief Write data into the buffer
     * 
     * Write data which is given argument into the buffer.
     *
     * @param value Target data for writing
     *
     * @return Writing result (Always true: writing success is returned)
     * 
     * @endif
     */
    virtual ReturnCode write(const DataType& value,
                             long int sec = -1, long int nsec = 0)
    {
      Guard guard(m_full.mutex);
      
      if (full())
        {
          
          bool timedwrite(m_timedwrite);
          bool overwrite(m_overwrite);

          if (!(sec < 0)) // if second arg is set -> block mode
            {
              timedwrite = true;
              overwrite  = false;
            }

          if (overwrite && !timedwrite)       // "overwrite" mode
            {
              advanceRptr();
            }
          else if (!overwrite && !timedwrite) // "do_nothing" mode
            {
              return ::RTC::BufferStatus::BUFFER_FULL;
            }
          else if (!overwrite && timedwrite)  // "block" mode
            {
              if (sec < 0)
                {
                  sec = m_wtimeout.sec();
                  nsec = m_wtimeout.usec() * 1000;
                }
              //  true: signaled, false: timeout
              if (!m_full.cond.wait(sec, nsec))
                {
                  return ::RTC::BufferStatus::TIMEOUT;
                }
            }
          else                                    // unknown condition
            {
              return ::RTC::BufferStatus::PRECONDITION_NOT_MET;
            }
        }
      
      bool empty_(empty());
      
      put(value);
      
      if (empty_)
        {
          Guard eguard(m_empty.mutex);
          m_empty.cond.signal();
        }
      advanceWptr(1);
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåե˽߲ǽǿ
     * 
     * Хåե˽߲ǽǿ֤
     * 
     * @return 񤭹߲ǽǿ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     *
     * @param value Target data to write.
     *
     * @return Result of having written in data (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual size_t writable() const
    {
      Guard guard(m_posmutex);
      return m_length - m_fillcount;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեfullå
     * 
     * Хåեfullåѽ貾۴ؿ
     *
     * @return fullå(true:Хåեfullfalse:Хåե)
     * 
     * @else
     *
     * @brief Check on whether the buffer is full.
     *
     * Pure virtual function to check on whether the buffer is full.
     *
     * @return True if the buffer is full, else false.
     *
     * @endif
     */
    virtual bool full(void) const
    {
      Guard guard(m_posmutex);
      return m_length == m_fillcount;
    }
    
    //----------------------------------------------------------------------
    /*!
     * @if jp
     *
     * @brief Хåեθߤɤ߽ФǤΥݥ
     * 
     * Хåեθߤɤ߽ФǤΥݥ󥿤ޤϡnΥݥ󥿤֤
     * 
     * @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
     * @return ɤ߽Ф֤Υݥ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */ 
    virtual DataType* rptr(long int n = 0)
    {
      Guard guard(m_posmutex);
      return &(m_buffer[(m_rpos + n + m_length) % m_length]);
    }
    
    /*!
     * @if jp
     *
     * @brief ɤ߽Фݥ󥿤ʤ
     * 
     * ߤɤ߽Ф֤Υݥ󥿤 n Ŀʤ롣
     * 
     * @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */ 
    virtual ReturnCode advanceRptr(long int n = 1)
    {
      // n > 0 :
      //     n satisfies n <= readable elements
      //                 n <= m_fillcout 
      // n < 0 : -n = n'
      //     n satisfies n'<= m_length - m_fillcount
      //                 n >= m_fillcount - m_length
      if ((n > 0 && n > static_cast<long int>(m_fillcount)) ||
          (n < 0 && n < static_cast<long int>(m_fillcount - m_length)))
        {
          return ::RTC::BufferStatus::PRECONDITION_NOT_MET;
        }

      Guard guard(m_posmutex);
      m_rpos = (m_rpos + n + m_length) % m_length;
      m_fillcount -= n;
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåեǡɤߤɤ߽Фݥ󥿤ΰ֤ѹʤ
     * 
     * @param value ɤ߽Фǡ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     *
     * @param value Target data to write.
     *
     * @return Result of having written in data (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual ReturnCode get(DataType& value)
    {
      Guard gaurd(m_posmutex);
      value = m_buffer[m_rpos];
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    
    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåեǡɤߤɤ߽Фݥ󥿤ΰ֤ѹʤ
     *
     * @return ɤ߽Фǡ
     * 
     * @else
     *
     * @brief Reading data from the buffer
     *
     * @return Read data
     *
     * @endif
     */
    virtual DataType& get()
    {
      Guard gaurd(m_posmutex);
      return m_buffer[m_rpos];
    }
    
    
    /*!
     * @if jp
     *
     * @brief Хåեɤ߽Ф
     * 
     * Хåե˳Ǽ줿ǡɤ߽Ф
     *
     * 2(sec)3(nsec)ꤵƤʤ硢Хåե
     * ֤Ǥɤ߽Ф⡼ (readback, do_nothing, block)  init() 
     * ꤵ줿⡼ɤ˽
     *
     * 2(sec) ˰ꤵ줿ϡinit() ꤵ줿⡼
     * ˴ؤ餺block ⡼ɤȤʤꡢХåե֤Ǥл
     * ԤॢȤ롣3(nsec)ϻꤵʤ0Ȥư
     * 롣ॢԤˡߥå¦ǥХåեؽ
     * С֥å󥰤ϲǡɤߤ롣
     *
     * ɤ߽Ф˥Хåե(empty)֤ǡ̤Υåɤblock⡼
     * ǽԤ򤷤Ƥ硢signalȯԤƽ¦Υ֥å
     * 롣
     * 
     * @param value ɤ߽Фоݥǡ
     * @param sec   ॢȻ sec  (default -1: ̵)
     * @param nsec  ॢȻ nsec (default 0)
     * @return BUFFER_OK            ｪλ
     *         BUFFER_EMPTY         Хåե
     *         TIMEOUT              ߤॢȤ
     *         PRECONDITION_NOT_MET ۾
     * 
     * @else
     *
     * @brief Readout data from the buffer
     * 
     * Readout data stored into the buffer.
     * 
     * @param value Readout data
     *
     * @return Readout result (Always true: readout success is returned)
     * 
     * @endif
     */
    virtual ReturnCode read(DataType& value,
                            long int sec = -1, long int nsec = 0)
    {
      Guard gaurd(m_empty.mutex);
      
      if (empty())
        {
          bool timedread(m_timedread);
          bool readback(m_readback);

          if (!(sec < 0)) // if second arg is set -> block mode
            {
              timedread = true;
              readback  = false;
              sec = m_rtimeout.sec();
              nsec = m_rtimeout.usec() * 1000;
            }

          if (readback && !timedread)       // "readback" mode
            {
              if (!(m_wcount > 0))
                {
                  return ::RTC::BufferStatus::BUFFER_EMPTY;
                }
              advanceRptr(-1);
            }
          else if (!readback && !timedread) // "do_nothing" mode
            {
              return ::RTC::BufferStatus::BUFFER_EMPTY;
            }
          else if (!readback && timedread)  // "block" mode
            {
              if (sec < 0)
                {
                  sec = m_rtimeout.sec();
                  nsec = m_rtimeout.usec() * 1000;
                }
              //  true: signaled, false: timeout
              if (!m_empty.cond.wait(sec, nsec))
                {
                  return ::RTC::BufferStatus::TIMEOUT;
                }
            }
          else                                    // unknown condition
            {
              return ::RTC::BufferStatus::PRECONDITION_NOT_MET;
            }
        }
      
      bool full_(full());
      
      get(value);
      advanceRptr();

      if (full_)
        {
          Guard fguard(m_full.mutex);
          m_full.cond.signal();
        }
      
      return ::RTC::BufferStatus::BUFFER_OK;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեɤ߽Фǽǿ
     * 
     * Хåեɤ߽Фǽǿ֤
     * 
     * @return ɤ߽Фǽǿ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     *
     * @param value Target data to write.
     *
     * @return Result of having written in data (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual size_t readable() const
    {
      Guard guard(m_posmutex);
      return m_fillcount;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեemptyå
     * 
     * Хåեemptyåѽ貾۴ؿ
     *
     * @return emptyå(true:Хåեemptyfalse:Хåեǡ)
     * 
     * @else
     *
     * @brief Check on whether the buffer is empty.
     *
     * Pure virtual function to check on whether the buffer is empty.
     *
     * @return True if the buffer is empty, else false.
     *
     * @endif
     */
    virtual bool empty(void) const
    {
      Guard guard(m_posmutex);
      return m_fillcount == 0;
    }
    
  private:
    inline void initLength(const coil::Properties& prop)
    {
      if (!prop["length"].empty())
        {
          size_t n;
          if (coil::stringTo(n, prop["length"].c_str()))
            {
              if (n > 0)
                {
                  this->length(n);
                }
            }
        }
    }
    
    inline void initWritePolicy(const coil::Properties& prop)
    {
      std::string policy(prop["write.full_policy"]);
      coil::normalize(policy);
      if (policy == "overwrite")
        {
          m_overwrite = true;
          m_timedwrite = false;
        }
      else if (policy == "do_nothing")
        {
          m_overwrite = false;
          m_timedwrite = false;
        }
      else if (policy == "block")
        {
          m_overwrite = false;
          m_timedwrite = true;
          
          double tm;
          if (coil::stringTo(tm, prop["write.timeout"].c_str()))
            {
              if (!(tm < 0))
                {
                  m_wtimeout = tm;
                }
            }
        }
    }
    
    inline void initReadPolicy(const coil::Properties& prop)
    {
      std::string policy(prop["read.empty_policy"]);
      if (policy == "readback")
        {
          m_readback = true;
          m_timedread = false;
        }
      else if (policy == "do_nothing")
        {
          m_readback = false;
          m_timedread = false;
        }
      else if (policy == "block")
        {
          m_readback = false;
          m_timedread = true;
          double tm;
          if (coil::stringTo(tm, prop["read.timeout"].c_str()))
            {
              m_rtimeout = tm;
            }
        }
    }
    
  private:
    bool m_overwrite;
    bool m_readback;
    bool m_timedwrite;
    bool m_timedread;
    coil::TimeValue m_wtimeout;
    coil::TimeValue m_rtimeout;
    
    size_t m_length;
    size_t m_wpos;
    size_t m_rpos;
    size_t m_fillcount;
    size_t m_wcount;
    std::vector<DataType> m_buffer;
    
    struct condition
    {
      condition() : cond(mutex) {}
      coil::Condition<coil::Mutex> cond;
      coil::Mutex mutex;
    };
    
    mutable coil::Mutex m_posmutex;
    condition m_empty;
    condition m_full;
  };
}; // namespace RTC

#endif // RTC_RINGBUFFER_H
