// -*- C++ -*-
/*!
 * @file InPort.h
 * @brief InPort template class
 * @date $Date: 2007-12-31 03:08:03 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2003-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: InPort.h 1788 2010-01-26 02:15:59Z kurihara $
 *
 */

#ifndef RTC_INPORT_H
#define RTC_INPORT_H

#include <string>
#include <vector>
#include <iostream>

#include <coil/TimeValue.h>
#include <coil/Time.h>
#include <coil/OS.h>

#include <rtm/RTC.h>
#include <rtm/Typename.h>
#include <rtm/InPortBase.h>
#include <rtm/CdrBufferBase.h>
#include <rtm/PortCallback.h>
#include <rtm/InPortConnector.h>

namespace RTC
{
  /*!
   * @if jp
   *
   * @class InPort
   *
   * @brief InPort ƥץ졼ȥ饹
   * 
   * InPort μǤ InPort<T> Υƥץ졼ȥ饹
   * <T> BasicDataType.idl ˤƤ뷿ǡФȤ
   * Time  tm ,  T data Ĺ¤ΤǤʤƤϤʤʤ
   * InPort ˥󥰥Хåե줿ǡ缡
   * Υ󥰥Хåե˳Ǽ롣󥰥ХåեΥϥǥեȤ64
   * ʤäƤ뤬󥹥ȥ饯ˤꥵꤹ뤳ȤǤ롣
   * ǡϥե饰ˤä̤ɡɾ֤졢isNew(), write(), read(),
   * isFull(), isEmpty() Υ᥽åɤˤϥɥ󥰤뤳ȤǤ롣
   *
   *
   *   
   * OnReadϥХå (ɤ߽Ф˵륤٥Ȥˤꥳ뤵)
   *
   * - void OnRead::operator(): 
   *     InPort::read() ƤӽФɤ߽ФԤݤ˥뤵롣
   *
   * - DataType OnReadConvert::operator(DataType): 
   *     InPort::read() ƤӽФǡХåեɤߤݤ˸ƤФ
   *     ǡѴԤˤϥХåեɤ߽Ф줿ͤͿ졢
   *     ѴΥǡͤȤ֤ͤread()֤ͤȤʤ롣
   *
   *
   * OnBufferϥХå (Хåե˵륤٥Ȥˤꥳ뤵)
   *
   * - void OnBufferOverwrite::operator(ConnectorId):
   *     ConnectorBase::write() ˡХåե񤭥⡼ɤ
   *     ƤꡢХåեäѤξ˸ƤӽФ륳Хå
   *     ͥIDͿ롣
   *
   * - void OnBufferEmpty::operator(ConnectorId): 
   *     ͥХåեɤߤݤˡХåեξ˸ƤӽФ롣
   *     ͥIDͿ롣
   *
   * - void OnBufferReadTimeout::operator(ConnectorId):
   *     ͥХåեɤߤݤˡХåեǤġɤ߽Ф
   *     ॢȤ˸ƤӽФ륳ХåͥID
   *     Ϳ롣
   *
   * - void OnBufferFull::operator(ConnectorId, DataType):
   *     ConnectorBase::write() ˡХåեäѤξ
   *     ƤӽФ륳ХåͥIDӽ񤭹ʤäǡ
   *     Ϳ롣
   *
   * - void OnBufferWriteTimeout::operator(ConnectorId, DataType):
   *     ConnectorBase::write() ˡХåեäѤǽߤ
   *     ॢȤ˸ƤӽФ륳ХåͥID
   *     ӽ񤭹ʤäǡͿ롣
   *
   *
   * OnConnectϥХå (³˵륤٥Ȥˤꥳ뤵)
   * 
   * - void OnConnect::operator(ConnectorProfile):
   *      ݡȤ³˸ƤӽФ륳ХåConnectorProfile
   *      Ϳ롣
   *
   * - void OnDisconnect::operator(ConnectorId):
   *      ݡȤ³ǻ˸ƤӽФ륳Хå˥ͥID
   *      Ϳ롣  
   *
   * - void OnConnectionLost::operator(ConnectorId):
   *      ݡȤ³Ȥ˸ƤӽФ륳Хå
   *      ˥ͥIDͿ롣InPortϡ¦OutPortȤ
   *      ³Ȥ硢³ŪǤΤǡ
   *      ³ OnDisconnect ХåƤӽФ롣
   *
   *
   * OnReceiveϥХå (¦˵륤٥Ȥˤꥳ뤵)
   *
   * - void OnSend::operator(ConnectorId):
   *      ǡInPortФݤ˸ƤӽФ륳Хå
   *      ˥ͥIDͿ롣
   *
   * - void OnSenderTimeout::operator(ConnectorId):
   *      ǡInPortФ줿ॢȤݤ
   *      ƤӽФ륳Хå˥ͥIDͿ롣
   *      InPortΥץХӥ󥷥塼ޤбƤ˸¤ͭ
   *
   * - void OnSenderError::operator(ConnectorId):
   *      ǡInPortФ줿餫Υ顼InPort¦
   *      ֤˸ƤӽФ륳Хå
   *      ˤϡͥIDͿ롣
   *
   *
   * OnReceiveϥХå (¦˵륤٥Ȥˤꥳ뤵)
   * 
   * - void OnReceiverFull::operator(ConnectorId):
   *      ǡInPortΤΡInPort¦ΥХåեäѤξ
   *      Τ뤿˸ƤФ륳Хå
   *       ˤϡͥIDͿ롣
   *
   * - void OnReceiverTimeout::operator(ConnectorId):
   *      ǡInPortΤΡInPort¦ΥХåեäѤ
   *      ॢȤˤΤ뤿˸ƤФ륳Хå
   *       ˤϡͥIDͿ롣
   *
   * - void OnReceiverError::operator(ConnectorId):
   *       ǡInPortΤΡInPort¦ǲ餫Υ顼֤
   *       ˸ƤӽФ륳Хå
   *       ˤϡͥIDͿ롣
   *
   * - void OnReceived::operator(ConnectorId):
   *       ǡӼλݤ˸ƤӽФ륳Хå
   *       ˤϡͥIDͿ롣
   *
   *
   * @since 0.2.0
   *
   * @else
   *
   * @class InPort
   *
   * @brief InPort template class
   *
   * This is a template class that implements InPort.
   * <T> is the type defined in BasicDataType.idl and must be the structure which
   * has both Time type tm and type-T data as a member. InPort has a ring
   * buffer internally, and stores the received data externally in this buffer
   * one by one. The size of ring buffer can be specified according to 
   * the argument of constructor, though the default size is 64. Unread
   * data and data which is already read are managed with the flag, and the
   * data can be handled by the isNew(), write(), read(), isFull() and isEmpty()
   * method etc. 
   *
   * @since 0.2.0
   *
   * @endif
   */
  template <class DataType>
  class InPort
    : public InPortBase
  {
  public:
    DATAPORTSTATUS_ENUM
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     * ѥ᡼ȤͿ T ѿ˥Хɤ롣
     *
     * @param name InPort ̾InPortBase:name() ˤ껲Ȥ롣
     * @param value  InPort ˥Хɤ T ѿ
     * @param bufsize InPort Υ󥰥ХåեΥХåեĹ(ǥե:64)
     * @param read_block ɹ֥åե饰
     *        ǡɹ̤ɥǡʤ硢Υǡޤǥ֥å
     *        ɤ(ǥե:false)
     * @param write_block ֥åե饰
     *        ǡ˥ХåեեǤä硢Хåե˶Ǥ
     *        ޤǥ֥å뤫ɤ(ǥե:false)
     * @param read_timeout ɹ֥åꤷƤʤΡǡɼ西
     *        Ȼ(ߥ)(ǥե:0)
     * @param write_timeout ֥åꤷƤʤΡǡ
     *        Ȼ(ߥ)(ǥե:0)
     *
     * @else
     *
     * @brief A constructor.
     *
     * constructor.
     * This is bound to type-T variable given as a parameter.
     *
     * @param name A name of the InPort. This name is referred by
     *             InPortBase::name().
     * @param value type-T variable that is bound to this InPort.
     * @param bufsize Buffer length of internal ring buffer of InPort
     *                (The default value:64)
     * @param read_block Flag of reading block.
     *                   When there are not unread data at reading data,
     *                   set whether to block data until receiving the next 
     *                   data. (The default value:false)
     * @param write_block Flag of writing block.
     *                    If the buffer was full at writing data, set whether 
     *                    to block data until the buffer has space. 
     *                    (The default value:false)
     * @param read_timeout Data reading timeout time (millisecond) 
     *                     when not specifying read blocking.
     *                     (The default value:0)
     * @param write_timeout Data writing timeout time (millisecond)
     *                      when not specifying writing block.
     *                      (The default value:0)
     *
     * @endif
     */
    InPort(const char* name, DataType& value,
	   int bufsize=64, 
	   bool read_block = false, bool write_block = false,
	   int read_timeout = 0, int write_timeout = 0)
      :	InPortBase(name, toTypename<DataType>()),
        m_name(name), m_value(value),
	m_OnRead(NULL),  m_OnReadConvert(NULL)
    {
    }
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    virtual ~InPort(void){};

    /*!
     * @if jp
     *
     * @brief ݡ̾Τ롣
     *
     * ݡ̾Τ롣
     *
     * @return ݡ̾
     *
     * @else
     *
     * @brief Get port name
     *
     * Get port name.
     *
     * @return The port name
     *
     * @endif
     */
    virtual const char* name()
    {
      return m_name.c_str();
    }

    
    /*!
     * @if jp
     *
     * @brief ǿǡ¸ߤ뤫ǧ
     * 
     * InPort̤ɤκǿǡ夷Ƥ뤫bool֤ͤ
     * InPort̤³ξ硢³ͥΥХåեEmpty
     * ξˤfalse֤
     *
     * @return true ̤ɤκǿǡ¸ߤ
     *         false ̤³ޤϥХåե˥ǡ¸ߤʤ
     * 
     * @else
     *
     * @brief Check whether the data is newest
     * 
     * Check whether the data stored at a current buffer position is newest.
     *
     * @return Newest data check result
     *         ( true:Newest data. Data has not been readout yet.
     *          false:Past dataData has already been readout.)
     * 
     * @endif
     */
    virtual bool isNew()
    {
      RTC_TRACE(("isNew()"));
      if (m_connectors.size() == 0)
        {
          RTC_DEBUG(("no connectors"));
          return false;
        }
      int r(m_connectors[0]->getBuffer()->readable());
      if (r > 0)
        {
          RTC_DEBUG(("isNew() = true, readable data: %d", r));
          return true;
        }
      
      RTC_DEBUG(("isNew() = false, no readable data"));
      return false;
    }

    /*!
     * @if jp
     *
     * @brief Хåեɤǧ
     * 
     * InPortΥХåեɤ bool ֤ͤ
     * ξ true, ̤ɥǡ false ֤
     *
     * @return true  Хåե϶
     *         false Хåե̤ɥǡ
     * 
     * @else
     *
     * @brief Check whether the data is newest
     * 
     * Check whether the data stored at a current buffer position is newest.
     *
     * @return Newest data check result
     *         ( true:Newest data. Data has not been readout yet.
     *          false:Past dataData has already been readout.)
     * 
     * @endif
     */
    virtual bool isEmpty()
    {
      RTC_TRACE(("isEmpty()"));

      if (m_connectors.size() == 0)
        {
          RTC_DEBUG(("no connectors"));
          return true;
        }
      int r(m_connectors[0]->getBuffer()->readable());
      if (r == 0)
        {
          RTC_DEBUG(("isEmpty() = true, buffer is empty"));
          return true;
        }
      
      RTC_DEBUG(("isEmpty() = false, data exists in the buffer"));
      return false;
    }

    /*!
     * @if jp
     *
     * @brief DataPort ͤɤ߽Ф
     *
     * InPort˽񤭹ޤ줿ǡɤߤ³0ޤϥХåե
     * ǡ񤭹ޤƤʤ֤ɤߤͤǤ롣
     * Хåեξ֤ΤȤ
     * ꤵ줿⡼ (readback, do_nothing, block) ˱ơ
     * ʲΤ褦ư򤹤롣
     *
     * - readback: Ǹͤɤߤʤ
     *
     * - do_nothing: ⤷ʤ
     *
     * - block: ֥å롣ॢȤꤵƤϡ
     *       ॢȤޤԤġ
     *
     * Хåեξ֤ǤϡInPort˥Хɤ줿ѿ֤ͤ롣
     * äơɤ߽Фˤ֤ͤǽ롣
     * δؿѤݤˤϡ
     *
     * - isNew(), isEmpty() ʻѤ˥Хåե֤å롣
     * 
     * - ɤ߽Ф֤ͤʤ褦˥Хѿ˽
     * 
     * - ReturnCode read(DataType& data) ؿѤƤ롣
     *
     * Ȥ˾ޤ
     *
     * ƥХåؿϰʲΤ褦˸ƤӽФ롣
     * - OnRead: read() ؿƤФݤɬƤФ롣
     * 
     * - OnReadConvert: ǡɤ߽Ф硢ɤߤǡ
     *       ȤOnReadConvertƤӽФ졢ͤread()
     *       Ȥ֤
     *
     * - OnEmpty: ХåեΤǡɤ߽Ф˼ԤƤӽФ롣
     *        OnEmpty ͤ read() ͤȤ֤
     *
     * - OnBufferTimeout: ǡեPushξˡɤ߽Ф
     *        ॢȤΤ˥ǡɤ߽Ф˼Ԥ˸ƤФ롣
     *
     * - OnRecvTimeout: ǡեPullξˡɤ߽Фॢ
     *        Τ˥ǡɤ߽Ф˼Ԥ˸ƤФ롣
     *
     * - OnReadError: 嵭ʳͳɤߤ˼Ԥ˸ƤФ롣
     *        ͳȤƤϡХåե硢㳰ȯʤɤͤ
     *        ̾ϵꤨʤХβǽ롣
     *
     * @return ɤ߽Ф(ɤ߽Ф:true, ɤ߽Ф:false)
     *
     * @else
     *
     * @brief Readout the value from DataPort
     *
     * Readout the value from DataPort
     *
     * - When Callback functor OnRead is already set, OnRead will be invoked
     *   before reading from the buffer held by DataPort.
     * - When the buffer held by DataPort can detect the underflow,
     *   and when it detected the underflow at reading, callback functor
     *   OnUnderflow will be invoked.
     * - When callback functor OnReadConvert is already set, the return value of
     *   operator() of OnReadConvert will be the return value of read().
     * - When timeout of reading is already set by setReadTimeout(),
     *   it waits for only timeout time until the state of the buffer underflow
     *   is reset, and if OnUnderflow is already set, this will be invoked to 
     *   return.
     *
     * @return Readout result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool read()
    {
      RTC_TRACE(("DataType read()"));

      if (m_OnRead != NULL) 
        {
          (*m_OnRead)();
          RTC_TRACE(("OnRead called"));
        }

      if (m_connectors.size() == 0)
        {
          RTC_DEBUG(("no connectors"));
          return false;
        }

      cdrMemoryStream cdr;

      ReturnCode ret(m_connectors[0]->read(cdr));
      if (ret == PORT_OK)
        {
          RTC_DEBUG(("data read succeeded"));
#ifndef ORB_IS_RTORB
          m_value <<= cdr;
#else
          m_value <<= cdr;
#endif
          if (m_OnReadConvert != 0) 
            {
              m_value = (*m_OnReadConvert)(m_value);
              RTC_DEBUG(("OnReadConvert called"));
              return true;
            }
          return true;
        }
      else if (ret == BUFFER_EMPTY)
        {
          RTC_WARN(("buffer empty"));
          return false;
        }
      else if (ret == BUFFER_TIMEOUT)
        {
          RTC_WARN(("buffer read timeout"));
          return false;
        }
      RTC_ERROR(("unknown retern value from buffer.read()"));
      return false;
    }
    

    /*!
     * @if jp
     *
     * @brief Хɤ줿 T ѿ InPort Хåեκǿͤɤ߹
     *
     * Хɤ줿 T Υǡ InPort κǿͤɤ߹ࡣ
     * 󥹥ȥ饯 T ѿ InPort ХɤƤʤФʤʤ
     * Υ᥽åɤϥݥ⡼եå˻ѤȤƤ뤿ᡢ
     * ˰¸ʤͤȤʤäƤ롣
     *
     * @else
     *
     * @brief Read the newly value to type-T variable which is bound to InPort's
     *        buffer.
     *
     * Read the newly value to type-T data which is bound to InPort's buffer.
     * The type-T variable must be bound to InPort in constructor.
     * Since this method assumes to be used for polymorphic,
     * its argument and the return value do not depend on type.
     *
     * @endif
     */
    virtual void update()
    {
      this->read();
    };
    
    /*!
     * @if jp
     *
     * @brief T Υǡ InPort κǿͥǡɤ߹
     *
     * InPort ꤵƤǿǡɤ߹ߡ
     * ꤵ줿ǡѿꤹ롣
     *
     * @param rhs InPort Хåեͤɤ߹ T ѿ
     *
     * @else
     *
     * @brief Read the newly value data in InPort to type-T variable
     *
     * Read the newly data set in InPort and set to specified data variable.
     *
     * @param rhs The type-T variable to read from InPort's buffer
     *
     * @endif
     */
    void operator>>(DataType& rhs)
    {
      this->read();
      rhs = m_value;
      return;
    }
    
    /*!
     * @if jp
     *
     * @brief InPort Хåեإǡɤ߹߻ΥХå
     *
     * InPort ĥХåեǡɤ߹ޤľ˸ƤФ륳Хå
     * ֥Ȥꤹ롣
     * 
     * @param on_read OnRead&lt;DataType&gt;Υ֥
     *
     * @else
     *
     * @brief Set callback when data is read from the InPort buffer
     *
     * Set the callback object that is invoked right before data is read from 
     * the InPort's buffer
     * 
     * @param on_read OnRead&lt;DataType&gt; type object
     *
     * @endif
     */
    inline void setOnRead(OnRead<DataType>* on_read)
    {
      m_OnRead = on_read;
    }
    
    /*!
     * @if jp
     *
     * @brief InPort Хåեإǡɤ߽ФΥХå
     *
     * InPort ĥХåեǡɤ߽Фݤ˸ƤФ륳Хå
     * ֥Ȥꤹ롣Хå֥Ȥͤread()᥽å
     * θƽз̤Ȥʤ롣
     * 
     * @param on_rconvert OnReadConvert&lt;DataType&gt;Υ֥
     *
     * @else
     *
     * @brief Set callback when data is readout to the InPort buffer
     *
     * Set the callback object that is invoked when data is readout to
     * the InPort's buffer. The return value of callback object is the return
     * result of the read() method.
     * 
     * @param on_rconvert OnReadConvert&lt;DataType&gt; type object
     *
     * @endif
     */
    inline void setOnReadConvert(OnReadConvert<DataType>* on_rconvert)
    {
      m_OnReadConvert = on_rconvert;
    }
    
  private:
    std::string m_typename;
    /*!
     * @if jp
     * @brief ݡ̾
     * @else
     * @berif Port's name
     * @endif
     */
    std::string m_name;
    
    /*!
     * @if jp
     * @brief Хɤ T ѿؤλ
     * @else
     * @brief The reference to type-T value bound this OutPort
     * @endif
     */
    DataType& m_value;
    
    /*!
     * @if jp
     * @brief OnRead Хåե󥯥ؤΥݥ
     * @else
     * @brief Pointer to OnRead callback functor
     * @endif
     */
    OnRead<DataType>* m_OnRead;
    
    /*!
     * @if jp
     * @brief OnReadConvert Хåե󥯥ؤΥݥ
     * @else
     * @brief Pointer to OnReadConvert callback functor
     * @endif
     */
    OnReadConvert<DataType>* m_OnReadConvert;
   
  };
}; // End of namesepace RTM

#endif // RTC_INPORT_H
