// -*- C++ -*-
/*!
 * @file  Routing.cpp
 * @brief Network routing information handling functions
 * @date  $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2010
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#include <stdio.h>
#ifdef __T_KERNEL__
#include "pmc_sock.h"
#else
#include <netdb.h>      // gethostbyname
#include <arpa/inet.h>  // inet_ntoa
#include <netinet/in.h> // sockaddr_in
#endif

#include <coil/Routing.h>
#include <coil/stringutil.h>
#include <coil/config_coil.h>

namespace coil
{
  /*!
   * @if jp
   * @brief 襢ɥ쥹Ѥ륨ɥݥȥɥ쥹
   * @else
   * @brief Getting network interface name from destination address
   * @endif
   */
  bool dest_to_endpoint(std::string dest_addr, std::string& endpoint)
  {
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    bool ret = true;
    int  err = 0;
    
    struct ::hostent *hostent = NULL;
    struct ::sockaddr_in to_addr;
    
    int  fd  = 0;
    char *ip_addr =NULL;
    struct sockaddr_in from_addr;
    int  len = 0;
    

    //襢ɥ쥹鰸IPɥ쥹μ
    
    //ͤȥɥåȵˡȽ
    int  c, n, v ;
    char  *p;
    char  *addr = NULL;
    
    addr = (char *)malloc(strlen(dest_addr.c_str()) +1);
    if(addr == NULL)
    {
      ret = false;
      goto LBL_EXIT;
    }
    strcpy(addr, dest_addr.c_str());
    
    for(p = addr, n = v = 0; (c = *p++) != 0;)
    {
      if (c >= '0' && c <= '9')
      {
        if ((v = v * 10 + (c - '0')) > 255) break;
      } else if (c == '.' && n < 3) {
        n++;
        v = 0;
      } else break;
    }

    if(c != 0 || n != 3)
    {
      hostent = gethostbyname(dest_addr.c_str());
      if(hostent == NULL)
      {
        ret = false;
        goto LBL_EXIT;
      }
      to_addr.sin_addr.s_addr = **(unsigned int **)(hostent->h_addr_list);
      dest_addr = inet_ntoa(to_addr.sin_addr);
    }
    
    //IPɥ쥹μ
    fd = socket(AF_INET,SOCK_STREAM, 0);
    if(fd == -1)
    {
      ret = false;
      goto LBL_EXIT;
    }
    
    len = sizeof(from_addr);
    err = getsockname(fd, (SOCKADDR*)&from_addr, &len);
    if( err < 0 )
    {
      ret = false;
      goto LBL_EXIT;
    }
    
    ip_addr = (char *)inet_ntoa(from_addr.sin_addr);
    if(ip_addr == NULL)
    {
      ret = false;
      goto LBL_EXIT;
    }

    //IPɥ쥹ȼIPɥ쥹
    //IPɥ쥹ȥ롦롼ץХåɥ쥹
    if( strcmp(ip_addr, dest_addr.c_str()) == 0
     || strcmp("127.0.0.1", dest_addr.c_str()) == 0 )
    {
      //ɤ餫Ʊ
      ret = false;
      goto LBL_EXIT;
    
    }else{
      //㤦
      endpoint = "addr:";
      endpoint += ip_addr;
      
    }
LBL_EXIT:
    if(fd != -1) closesocket(fd);
    if(hostent != NULL) free(hostent);
    if(addr != NULL) free(addr);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    
    return ret;
#else
    std::string dest_if;
    if (!find_dest_ifname(dest_addr, dest_if))
      {
        return false;
      }
    return ifname_to_ipaddr(dest_if, endpoint);
#endif
  }

  /*!
   * @if jp
   * @brief 襢ɥ쥹Ѥͥåȥ󥿡ե̾
   * @else
   * @brief Getting network interface name from destination address
   * @endif
   */
#ifndef __T_KERNEL__
  bool find_dest_ifname(std::string dest_addr, std::string& dest_if)
  {
    // This logic should be replaced by direct retrieving using
    // routing interface like AFROUTE or sysctl.
    struct ::hostent *hostent;
    struct ::sockaddr_in addr;
    
    hostent = gethostbyname(dest_addr.c_str());
    addr.sin_addr.s_addr = **(unsigned int **)(hostent->h_addr_list);
    dest_addr = inet_ntoa(addr.sin_addr);
    
#ifdef COIL_OS_FREEBSD
    std::string cmd("PATH=/bin:/sbin:/usr/bin:/usr/sbin "
                    "route get ");
    const char* match_str = "interface";
    const char* delimiter = ":";
    size_t ifname_pos(1);
    cmd += dest_addr;
    cmd += " 2> /dev/null";
#endif // COIL_OS_IS_FREEBSD
#ifdef COIL_OS_LINUX    
    std::string cmd("PATH=/bin:/sbin:/usr/bin:/usr/sbin "
                    "ip route get ");
    const char* match_str = "dev ";
    const char* delimiter = " ";
    size_t ifname_pos(2);
    cmd += dest_addr;
    cmd += " 2> /dev/null";
#endif // COIL_OS_IS_LINUX    
    
    FILE* fp;
    if ((fp = popen(cmd.c_str(), "r")) == NULL)
      {
        return false;
      }

    do
      {
        char str[512];
        fgets(str, 512, fp);
        std::string line(str);
        
        if (std::string::npos == line.find(match_str)) { continue; }
        
        line.erase(line.end() - 1);
        coil::vstring vs(coil::split(line, delimiter));

#ifdef COIL_OS_FREEBSD
        if (vs.size() > ifname_pos)
          {
            dest_if = vs[ifname_pos];
            pclose(fp);
            return true;
          }
#endif // COIL_OS_FREEBSD
#ifdef COIL_OS_LINUX
        for (int i(0); i < vs.size(); ++i)
          {
            if (vs[i] == "dev")
              {
                dest_if = vs[i + 1];
                return true;
              }
          }
#endif
      } while (!feof(fp));
    pclose(fp);
    return false;
  }
#endif

  /*!
   * @if jp
   * @brief ͥåȥ󥿡ե̾IPɥ쥹
   * @else
   * @brief Get IP address from a network interface name
   * @endif
   */
#ifndef __T_KERNEL__
  bool ifname_to_ipaddr(std::string ifname, std::string& ipaddr)
  {
    std::string cmd("ifconfig ");
    cmd += ifname;
    cmd += " 2> /dev/null";
    
    FILE* fp;
    if ((fp = popen(cmd.c_str(), "r")) == NULL)
      {
        return false;
      }
    
    do
      {
        char str[512];
        fgets(str, 512, fp);
        std::string line(str);
        
        if (std::string::npos == line.find("inet ")) { continue; }
        
        line.erase(line.end() - 1);
        coil::eraseHeadBlank(line);
        coil::vstring vs(coil::split(line, " "));
        if (vs.size() == 6)
          {
            ipaddr = vs[1];
            pclose(fp);
            return true;
          }
      } while (!feof(fp));
    pclose(fp);
    return false;
  }
#endif
  
}; // namespace coil
