// -*- C++ -*-
/*!
 * @file DynamicLib.cpp
 * @brief DynamicLib class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2008 Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#include <coil/DynamicLib.h>

#ifdef __T_KERNEL__
#include <stdio.h>
#include <tstring.h>
#include <btron/errcode.h>
#endif

namespace coil
{
  DynamicLib::DynamicLib(int close_handle_on_destruction)
    : m_closeflag(close_handle_on_destruction)
  {
  }

  DynamicLib::DynamicLib(const char* dynlib_name,
                         int open_mode,
                         int close_handle_on_destruction)
    : m_name(dynlib_name), m_mode(open_mode),
      m_closeflag(close_handle_on_destruction)
  {
    if (open(m_name.c_str(), m_mode, m_closeflag) != 0)
      {
        throw std::bad_alloc();
      }
  }

  DynamicLib::~DynamicLib()
  {
    close();
  }

  DynamicLib::DynamicLib(const DynamicLib& rhs)
    : m_name(""), m_mode(0), m_closeflag(0), m_handle(0)
  {
    if (!rhs.m_name.empty() &&
        open(rhs.m_name.c_str(), rhs.m_mode, rhs.m_closeflag) == 0)
      return;
//    throw std::bad_alloc();
  }

  DynamicLib& DynamicLib::operator=(const DynamicLib& rhs)
  {
    DynamicLib tmp(rhs);
    std::swap(this->m_name, tmp.m_name);
    std::swap(this->m_mode, tmp.m_mode);
    std::swap(this->m_closeflag, tmp.m_closeflag);
    std::swap(this->m_handle, tmp.m_handle);
    return *this;
  }

  int DynamicLib::open(const char* dll_name,
                   int open_mode,
                   int close_handle_on_destruction)
  {
#ifdef DEBUG
printf("DEBUG %s:%d\n", __FILE__, __LINE__);
    return -1;
#elif NO_DL
printf("NO_DL %s:%d\n", __FILE__, __LINE__);
    return -1;
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    ERR         err = ER_OK;
    TC          tmp[128];
    LINK        lnk;

    memset(tmp, 0, sizeof(tmp));
    memset(&lnk, 0, sizeof(lnk));

    eucstotcs( tmp, (const UB*)dll_name );

    err = b_dlsearch( tmp, &lnk, DL_BASE);
    if(err < ER_OK) return -1;

    err = b_dlopen( &lnk, open_mode);
    if(err < ER_OK) return -1;

    m_handle = err;
    m_name = dll_name;
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    void* handle = ::dlopen(dll_name, open_mode);
    if (handle == NULL)
      {
        return -1;
      }
    m_handle = handle;
    m_name = dll_name;
#endif
    return 0;
  }

  int DynamicLib::close(void)
  {
#ifdef DEBUG
printf("DEBUG %s:%d\n", __FILE__, __LINE__);
    return -1;
#elif NO_DL
printf("NO_DL %s:%d\n", __FILE__, __LINE__);
    return -1;
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    if(m_handle <= 0) return -1;
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif

    if(m_name.empty()) return -1;

    b_dlclose(m_handle);
    m_handle = -1;
    m_name = "";
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    if (m_handle == NULL)
      return -1;
    if (m_name.empty())
      {
        return -1;
      }
    ::dlclose(m_handle);
    m_handle = NULL;
    m_name = "";
#endif
    return 0;
  }

  void* DynamicLib::symbol(const char* symbol_name)
  {
#ifdef DEBUG
printf("DEBUG %s:%d\n", __FILE__, __LINE__);
    return NULL;
#elif NO_DL
printf("NO_DL %s:%d\n", __FILE__, __LINE__);
    return NULL;
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    void*       ret = NULL;
    ERR         err = ER_OK;

    if(m_handle <= 0) return NULL;

    err = b_dlsym( m_handle, symbol_name, (UW*)ret);
    if(err < ER_OK) return NULL;
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif

    return ret;
#else
    if (m_handle == NULL) return NULL;
    return ::dlsym(m_handle, symbol_name);
#endif
  }

  const char* DynamicLib::error(void) const
  {
#ifdef DEBUG
printf("DEBUG %s:%d\n", __FILE__, __LINE__);
    return NULL;
#elif NO_DL
printf("NO_DL %s:%d\n", __FILE__, __LINE__);
    return NULL;
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    return NULL;
#else
    return ::dlerror();
#endif
  }
};

/*!
 * for unit test.
 */
extern "C"
{
  int ForExternTest(void) { return coil::DynamicLib::ForExternTest(); }
}
