/**
 * @file nes_posix_proc.c
 * @brief implement of process operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <tstring.h>
#include <btron/proctask.h>
#include <btron/errcode.h>
#include <btron/message.h>
#include <btron/file.h>
#include <btron/clk.h>

#include "common.h"
#include "nes_posix_proc.h"
#include "nes_posix_error.h"


/* define the global buffer for process to restore its child process ID */
int gpidbuf[PROCESS_ID_NUMBER] = {0};


/**********************************************************************/ 
/* Function name: nes_posix_exit                                      */ 
/* Description:   This function terminate a process                   */ 
/* Return value:   void                                               */
/* Argument -     status : process finishment code                    */
/*                                                                    */ 
/**********************************************************************/ 
void nes_posix_exit(int status)
{
	b_ext_prc(status);

}

/**********************************************************************/ 
/* Function name: nes_posix_sleep                                     */ 
/* Description:   This function cause the calling programme to be     */
/*                suspended from execution until either the number of */
/*                realtime seconds specified by the argument seconds  */
/*                has elapsed or a signal is delivered to the calling */
/*                thread and its action is to invoke a signal-catching*/
/*                function or to terminate the process.               */ 
/* Return value:   0:  the requested time has elapsed                 */
/*                erro:   other value  "unslept" seconds(the requested*/
/*                        time minus the time actually slept)         */
/* Argument -     seconds: process's suspending time                  */
/*                                                                    */
/**********************************************************************/
unsigned int nes_posix_sleep(unsigned int seconds)
{
	STIME stime, etime;
	ERR eRet = 0;
	int time = 0;   

	//get now time
	b_get_tim(&stime, NULL);

	if((int)seconds >= 0)
	{    
		time = (int)seconds*1000;
		eRet = b_slp_tsk(time);	
		switch(eRet)
		{
			case ER_NONE:
				//process sleep until seconds have elapsed 
				eRet = 0;
				break;
			case ER_OK:
			case ER_MINTR:
				/*
				 * when is sleeping,process is interrupted by
				 * other task or signal
				 */ 
				//get now time
				b_get_tim(&etime, NULL);
				eRet = seconds - (etime - stime);
				break;
			default:
				eRet = EINVAL;
				break;		
		} 
	}
	else
	{
		DP(("process will sleep forever\n"));
		//wait forever
		eRet = b_slp_tsk(-1);
		switch(eRet)
		{
			case ER_OK:
			case ER_MINTR:
				/*
				 * when is sleeping,process is interrupted by
				 * other task or signal
				 */ 
				//get now time
				b_get_tim(&etime, NULL);
				eRet = seconds - (etime - stime);
				break;
			default:
				eRet = EINVAL;
				break;		
		}          
	}        

	return eRet;	
}

/**********************************************************************/ 
/* Function name: nes_posix_nanosleep                                 */ 
/* Description:   This function delay the excution of the program     */
/*                from execution until specified time by req has      */
/*                elapsed or a signal is delivered to the calling     */
/*                process and its action is to invoke a signalcatching*/
/*                function or to terminate the process.if rem is not  */
/*                null,rem will save the least time.                  */ 
/* Return value:  0: on Successful,the requested time has elapsed     */
/*                -1: Failed                                          */
/*                   EINTR: the pause is interrupted by not-blocked   */
/*                       signal.                                      */
/*                EINVAL:parameter is invalid                         */
/*                EFAULT:other erro                                   */
/* Argument -     seconds: process's suspending time                  */
/**********************************************************************/
int nes_posix_nanosleep(const struct timespec *req, struct timespec *rem)
{
	STIME stime, etime;
	ERR eRet = 0;
	STIME slptime;

	if ((req->tv_nsec >= 1000000000) || (req->tv_nsec < 0) || (req->tv_sec < 0))	
	{
		eRet = -1;
		errno = EINVAL;	
		return eRet;
	}

	slptime = req->tv_nsec/1000000 + req->tv_sec*1000;
	//get now time
	b_get_tim(&stime, NULL);

	eRet = b_slp_tsk(slptime);
	switch(eRet)
	{
		case ER_NONE:
			//process sleep until seconds have elapsed 
			eRet = 0;
			break;
		case ER_OK:
		case ER_MINTR:
			/*
			 * when is sleeping,process is interrupted by
			 * other task or signal
			 */ 				
			//get now time
			b_get_tim(&etime, NULL);
			if(rem)
			{
				rem->tv_sec = slptime/1000 - (etime - stime);
				rem->tv_nsec = 0;
			}
			eRet = -1;
			errno = EINTR;
			break;
		default:
			eRet = -1;
			errno = EFAULT;
			break;		
	} 
	return eRet;	
}

/*********************************************************************/ 
/* Function name: _createprocess                                      */ 
/* Description:   This function create a new process incording to the */
/*                value of pri and path.                              */ 
/* Return value:  on Suceess: created process ID                      */
/*                erro: -1                                            */
/*                    EINVAL:  parameter is invalid                   */
/*                    EPERM:   Operation not permitted                */
/*                    EIO:     I/O error                              */
/*                    ENOMEM:  out of memeroy                         */
/*                    EBUSY:   file link is opened by other process   */
/*                    EINVAL:  other erro                             */
/* Argument -     pri: process's priority                             */
/*                path:process execution programme                    */
/**********************************************************************/ 

int _createprocess(int pri, const char *path)
{
	WERR wRet = 0;
	TC chpath[20];
	LINK lnk;
	MESSAGE msg;

	//tranfer the path to TC type
	memset(chpath, 0, sizeof(chpath));
	eucstotcs(chpath, path);	

	/*
	 *  get link
	 */
	lnk.atr1 = 0;
	lnk.atr2 = 0;
	lnk.atr3 = 0; 
	lnk.atr4 = 0;
	lnk.atr5 = 0;
	wRet = b_get_lnk(chpath, &lnk, F_DIRECT);
	switch(wRet)
	{
		case   0:
		case   1:
		case   2:
			//get link successfully
			wRet = 0;
			break;
		case   ER_ACCES:
		case   ER_ADR:  
		case   ER_NOFS:
		case   ER_NOEXS:
			//operation not permitted
			wRet = -1;
			errno = EPERM;
			goto exit;

		case   ER_IO:
			//I/O error 
			wRet = -1;
			errno = EIO;
			goto exit;

		case   ER_FNAME:
		case   ER_PAR:
			//Path or mode is invalid
			DP(("b_get_lnk Failed: [%d]\n", EINVAL));
			wRet = -1;
			errno = EINVAL;
			goto exit;

		case   ER_NOSPC:
			//out of memory 
			wRet = -1;
			errno = ENOMEM;
			goto exit;

		default :
			wRet = -1;
			errno = EINVAL;
			goto exit;


	}

	/*
	 * create process 
	 */
	msg.msg_size = 0;
	wRet = b_cre_prc(&lnk, pri, &msg);
	if(wRet <= 0)
	{

		DP(("created process Failed: [%d]\n", wRet));
		switch(wRet)
		{
			case ER_ACCES:
			case ER_ADR:
				//operation not permitted
				errno = EPERM;
				wRet = -1;
				goto exit;

			case ER_BUSY:
				//file link is already opened by other process 
				errno = EBUSY;
				wRet = -1;
				goto exit;

			case ER_IO:
				//I/O error 
				errno = EIO;
				wRet = -1;
				goto exit;
			case ER_NOEXS:
			case ER_NOFS:
			case ER_SZOVR:
			case ER_PPRI:
				//parameter erro
				errno = EINVAL;
				wRet = -1;
				goto exit;
			case ER_NOMEM:
			case ER_NOSPC:
				//out of memory
				errno = ENOMEM;
				wRet = -1;
				goto exit;
			case ER_REC: 
				//programme record is not existing
				errno = EINVAL;
				wRet = -1;										;
				goto exit;
			default :
				errno = EINVAL;
				wRet = -1;
				goto exit;
		}
	}
exit:
	return wRet;
}

/**********************************************************************/ 
/* Function name: nes_posix_t_cre_prc                                 */ 
/* Description:   This function create a new process incording to the */
/*                value of pri and path.                              */ 
/* Return value:  on Suceess: created process ID                      */
/*                erro: -1                                            */
/*                    EINVAL:  parameter is invalid                   */
/*                    EPERM:   Operation not permitted                */
/*                    EIO:     I/O error                              */
/*                    ENOMEM:  out of memeroy                         */
/*                    EBUSY:   file link is opened by other process   */
/*                    EINVAL:  other erro                             */
/* Argument -     pri: process's priority                             */
/*                path:process execution programme                    */
/**********************************************************************/ 
int nes_posix_t_cre_prc(int pri, const char *path)
{
	WERR wRet = 0;
	pid_t pid = 0;
	int *p = NULL;
	int flg = 0;

	if( (pri >255) || (pri < 0 && pri !=-1) || (path == NULL)) 
	{
		errno = EINVAL;
		return -1;		
	}

	/*
	 * judge gpidbuf is empty or not, when flg=1, is empty to save pid
	 */
	p = gpidbuf; 
	while(p != &gpidbuf[PROCESS_ID_NUMBER-1])
	{
		if(*p == 0)
		{
			flg = 1;
			DP(("gpidbuf have space to save pid\n"));			
			//create new process
			wRet = _createprocess(pri, path);
			if(wRet == -1)
			{
				goto exit;
			}
			pid = wRet;  
			*p = pid ; 
			goto f1; 		
		}
		p ++;
	}		
	//if the last id of gpidbuf is equal to pid,then delete it
	if(*p == 0)
	{
		flg = 1;
		DP(("the last space of gpidbuf is empty,can save pid\n"));

		//create new process
		wRet = _createprocess(pri, path);
		if(wRet == -1)
		{
			goto exit;
		}
		pid = wRet;  
		*p = pid ;  
		goto f1; 	
	}

	//buff is full
	DP(("gdbuf is full not to save process ID\n"));
	wRet = -1;
	errno = ENOMEM;
	goto exit;	    


	/*
	 * set process sending message to farther or not
	 * but if it failed, we also ignore the erro.because
	 * process is created scuccessfully
	 */
f1:
	wRet = b_req_emg(pid, MM_ABORT | MM_EXIT | MM_TERM);	
	if(wRet < 0)
	{
		DP(("b_req_msg Failed: [%d]\n", wRet));
	}
	wRet = pid;

exit:	
	return wRet;
}


/**********************************************************************/ 
/* Function name: nes_posix_waitpid                                   */ 
/* Description:   This function obtain status information pertaining  */
/*                to one of the caller's child processes. Various     */
/*                options permit status information to be obtained for*/
/*                child processes that have terminated or stopped.    */
/*                (only wait the child process,and don't wait for     */
/*                grandson and itself)                                */
/* Return value:  on Suceess: the process ID of the child process for */
/*                            which status is reported                */
/*                erro: -1                                            */
/*                    EINVAL: parameter is invalid                    */
/*                    EPERM: operation is not permmited               */
/*                    ECHILD:   no process                            */
/*                    EINVAL: other erro                              */
/* Argument -     pri: process's priority                             */
/*                path:process execution programme                    */
/**********************************************************************/  
pid_t nes_posix_waitpid(pid_t pid, int *status, int options)
{
	WERR wRet = 0;
	P_STATE buff;
	pid_t callingid;
	MSG_SYS2 msg2;
	int *tmp = NULL;
	int flag = 0;


	DP(("waitpid IN -----\n"));	

	if( (options != WNOHANG) && (options != WUNTRACED) && (options != WCONTINUED))
	{
		wRet = -1;
		errno = EINVAL;
		DP(("waitpid OUT -----\n"));
		return wRet;
	}	

	/*
	 * T-Engine don't support process group,so when pid == 0
	 * or pid < -1, return EINVAL 
	 */
	if(pid == 0 || pid < -1)
	{
		wRet = -1;
		errno = EINVAL;
		goto exit;
	}

	/*
	 * only wait the process(ID = pid)
	 */
	if(pid > 0)
	{	
		DP(("pid > 0 \n"));	

		//get the calling processID
		callingid = b_prc_sts(0, NULL, NULL);		
		DP(("callingid = [%d]\n", callingid));

		//get the parent processID
		wRet = b_prc_sts(pid, &buff, NULL);

		if(wRet <= 0)		
		{
			//error mapping
			switch(wRet)
			{
				case ER_ADR:
					//operation is not permmited
					wRet = -1;
					errno = EPERM;
					goto exit;

				case ER_NOPRC:
				default:
					wRet = -1;
					errno = EINVAL;
					goto exit;

			}
		}	

		/*
		 * judge pid process is the calling process's child process or not
		 */
		if(buff.parpid != callingid)
		{
			wRet = -1;
			errno = ECHILD;
			goto exit;
		}

		//delete the child process ID from gpidbuf
		tmp = gpidbuf;
		while(tmp != &gpidbuf[PROCESS_ID_NUMBER-1])
		{
			if(*tmp == pid)
			{
				*tmp = 0;
			}
			tmp ++;
		}

		//if the last id of gpidbuf is equal to pid,then delete it
		if(*tmp == pid)
		{
			DP(("is the last id of gpidbuf\n"));
			*tmp = 0;
		}

		//get finishment message
		wRet = b_rcv_msg(MM_SYS2, (MESSAGE*)&msg2, sizeof(msg2), WAIT);
		while(wRet != pid)
		{
			wRet = b_rcv_msg(MM_SYS2, (MESSAGE*)&msg2, sizeof(msg2), WAIT);
			if(wRet < 0)
			{
				wRet = -1;
				errno = EINVAL;
				goto exit;
			}
		}
		DP(("msg2.exitmsg.code = [%d]\n", msg2.exitmsg.code));				
		if(status)
		{
			*status = msg2.exitmsg.code;
		}
		goto exit;
	}

	/*
	 * wait any child process (pid == -1)
	 */

	if(pid == -1)
	{ 			
		//judge parent process have child process or not
		tmp = gpidbuf; 

		while(tmp != &gpidbuf[PROCESS_ID_NUMBER-1])
		{
			//judge the process is alive or not
			if(*tmp != 0)
			{
				wRet = b_prc_sts(*tmp, NULL, NULL);
				if(wRet <= 0)
				{
					DP(("the process[%d] isn't alive\n", *tmp));
					*tmp = 0;
					tmp ++;
				}
				else //have child process and jumpt from while loop
				{ 
					DP(("have child process\n"));
					flag = 1;
					break;
				}
			}
			tmp++;
		}

		//judge the last process of gpidbuf is alive or not
		if(*tmp != 0)
		{
			wRet = b_prc_sts(*tmp, NULL, NULL);
			if(wRet > 0)
			{
				DP(("the last process of gpidbuf is alive\n"));
				flag = 1;
			}
		}

		//haven't child process
		if(flag == 0)
		{
			DP(("haven't child process\n"));	
			wRet = -1;
			errno = ECHILD;
			goto exit;
		}		

		wRet = b_rcv_msg(MM_SYS2, (MESSAGE*)&msg2, sizeof(msg2), WAIT);
		DP(("b_rcv_msg = [%d]\n", wRet));			
		if(wRet < 0)
		{
			wRet = -1;
			errno = EINVAL;
			goto exit;
		}

		/*
		 * delete the process ID from gpidbuf
		 */
		tmp = gpidbuf;
		while(tmp != &gpidbuf[PROCESS_ID_NUMBER-1])
		{
			if(*tmp == wRet)
			{
				*tmp = 0;
			}
			tmp ++;
		}		
		//if the last id of gpidbuf is equal to pid,then delete it
		if(*tmp == wRet)
		{
			DP(("is the last id of gpidbuf\n"));
			*tmp = 0;
		}		

		if(status)
		{
			*status = msg2.exitmsg.code;
		}		
		goto exit;
	}


exit:

	DP(("waitpid OUT -----\n"));		
	return wRet;
}

/**********************************************************************/ 
/* Function name: nes_posix_wait                                                */ 
/* Description:   This function obtain status information pertaining  */
/*                to one of the caller's child processes. Various     */
/*                options permit status information to be obtained for*/
/*                child processes that have terminated or stopped.    */
/*                (wait any of child processes                        */
/* Return value:  on Suceess: the process ID of the child process for */
/*                            which status is reported                */
/*                erro:  -1                                           */
/*                    EINVAL: parameter is invalid                    */
/* Argument -     pri: process's priority                             */
/*                path:process execution programme                    */
/**********************************************************************/ 
pid_t nes_posix_wait(int *status)
{
	return nes_posix_waitpid(-1, status, 0);	
}
