/**
 * @file nes_posix_pipe.c
 * @brief implement of process operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <sys/consio.h>
#include <errno.h>
#include "nes_posix_pipe.h"
#include "common.h"

/**********************************************************************/ 
/* Function name: nes_posix_pipe                                      */ 
/* Description:   create a pipe.(use console to simulate the pipe)    */
/* Return value:  On sucessfully: 0                                   */
/*                erro:                                               */
/*                     -1(EPERM):  operation is not permitted         */
/* Argument -     filedes[0] and filedes[1]: pipe ID                  */
/**********************************************************************/ 
int nes_posix_pipe(int filedes[2])
{
	UW arg[4] = {0};
	W wRet = 0;

	//create console
	arg[1] = CONF_BUFIO;
	arg[2] = 1024;
	arg[3] = 1024;

	wRet = console_conf(CS_CREATE, arg);
	if(wRet != 0)
	{
		DP(("console_conf Failed:[%d]\n", wRet));
		wRet = -1;
		errno = EPERM;
		goto exit;
	}
	DP(("console ID = [%d]\n", arg[0]));

	//set arg[0] to default console of calling process
	console_conf(CS_SETPORT, arg);

	//set EDIT mode to INPUT mode
	wRet = console_ctl(arg[0], INPUT, EDIT);
	if(wRet != 0)
	{
		DP(("console_ctl  Failed:[%d]\n", wRet));
		wRet = EPERM;
		goto exit;
	}
	filedes[0] = arg[0];
	filedes[1] = arg[0];
exit:
	return wRet;

}

/**********************************************************************/ 
/* Function name: nes_posix_getpipe                                             */ 
/* Description:   get the pipe that parent process has created        */
/* Return value:  on Success:   >0  pipe ID                           */
/*                erro:  0  pipe ID                                   */
/* Argument -     void                                                */
/**********************************************************************/ 
int nes_posix_getpipe(void)
{
	UW arg = 0;	

	//get console ID
	console_conf(CS_GETPORT, &arg);
	return arg;
}

/**********************************************************************/ 
/* Function name: nes_posix_pwrite                                            */ 
/* Description:   write the data to pipe. parent and child process can*/
/*                use the function to write the data to pipe.         */
/* Return value:  On sucessfully: >0     the byte of writed data      */
/*                erro:-1                                             */
/*                      EINVAL: parameter is invalid                  */
/*                      0 : don't read any data                       */
/* Argument -     flag: flag to differentiate the parent process and  */
/*                      child process .                               */
/*                cid:  pipe ID(console id)                           */
/*                buf:  the data buffer which will be writed to pipe. */      
/*                size: the data byte number which will be writed     */
/**********************************************************************/ 
int nes_posix_pwrite(int flag, int cid, B *buf, int size)
{
	W wRet = 0;
	int i=0;
	int tmpCount;
	if(!buf)
	{
		wRet = -1;
		errno = EINVAL;
		return wRet;
	}

	if(flag == 0)//child process
	{
		tmpCount=size/1024;
		if(tmpCount>0)
		{
			for(i=0;i<tmpCount;i++)
			{
				wRet = console_put((W)cid, buf+(i*1024), 1024, 0);
				if(wRet<=0)
				{
					return i*1024;
				}

			}

		}
		if((size%1024)>0)
		{
			wRet = console_put((W)cid, buf+(i*1024),(size%1024), 0);
			if(wRet<=0)
			{
				return i*1024;
			}
		}
	}
	else//parent process
	{
		tmpCount=size/1024;
		if(tmpCount>0)
		{
			for(i=0;i<tmpCount;i++)
			{
				wRet = console_out((W)cid, buf+(i*1024), 1024);
				if(wRet<=0)
				{
					return i*1024;
				}

			}

		}
		if((size%1024)>0)
		{
			wRet = console_out((W)cid, buf+(i*1024),(size%1024));
			if(wRet<=0)
			{
				return i*1024;
			}
		}	
	}

	if(wRet == 0)
	{
		errno = 0;
		wRet = -1;
	}
	return size ;	
}

/**********************************************************************/ 
/* Function name: nes_posix_pread                                             */ 
/* Description:   read the data of pipe. parent and child process can */
/*                use the function to read the data of pipe.          */
/* Return value:  On sucessfully: >0     the byte of readed out data  */
/*                erro:-1                                             */
/*                      EINVAL: parameter is invalid                  */
/*                      0 : don't read any data                       */
/* Argument -     flag: flag to differentiate the parent process and  */
/*                      child process .                               */
/*                cid:  pipe ID(console id)                           */
/*                buf:  the data buffer                               */     
/*                size: the data byte number which will be readed     */
/**********************************************************************/ 
int nes_posix_pread(int flag, int cid, char *buf, int size)
{	
	W wRet = 0;

	if(!buf)
	{
		wRet = -1;
		errno = EINVAL;
		return wRet;
	}

	if(flag == 0)//child process
	{
		wRet = console_get((W)cid, buf, size, 0);
	}
	else//parent process
	{
		//recv data from child process
		wRet = console_in(cid, buf, size);
	}
	if(wRet == 0)
	{
		errno = 0;
		wRet = 0;
	}

	return wRet;	
}


/**********************************************************************/ 
/* Function name: nes_posix_delpipe                                             */ 
/* Description:   delete the pipe.the function is used only by the    */
/*                creator.                                            */
/* Return value:  On sucessfully: =0     the byte of readed out data  */
/*                erro:                                               */
/*                     -1  operation is not permitted                 */
/* Argument -     cid: pipe ID(console id)                            */
/**********************************************************************/ 
int nes_posix_delpipe(int cid)
{
	return console_conf(CS_DELETE, &cid);
}
