/**
 * @file sigmanager.c
 * @brief internal core implemnet file
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <basic.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <tkse/errno.h>
#include <tkse/extension/message.h>
#include <tkse/extension/proctask.h>

#include "common.h"
#include "nes_posix_signal.h"
#include "sigmanager.h"
#include "nes_posix_mutex.h"
//#include "sig_internal_mutex.h"

#include "manager.h"		/* pthread manager */
#include "memory_check.h"

#define  QP(exp) ( (QUEUE*) &(exp) )

/* Global data define */
pthread_mutex_t __g_siglock; 			/* Global lock to protect all internal datas */
sigset_t __g_block_sigmask;			/* block mask */
struct sigaction __g_all_actions[_NSIG+1];	/* save all signal's sigaction */
struct sigpending __g_sigpending;		/* Pending signal info */
struct __alarm_time __g_alarm_time;		/* alarm signal record */
struct __sigwait_queue __g_sigwait_queue;	/* all thread's is call sigwait() */
QUEUE __g_tmpQueue;				/* Use by __check_pending, for save signal that thread blocked */

static void __default_term_handler(int signum, struct siginfo* info, void* param);
static void __default_ignore_handler(int signum, struct siginfo* info, void* param);
const __sigaction_handler_t __default_signal_handler_table[] = {
	0,							/* 0 */
	__default_term_handler,			/* 1 term */
	__default_term_handler,			/* 2 term */
	__default_term_handler,			/* 3 core */
	__default_term_handler,			/* 4 core */
	__default_term_handler,			/* 5 core */
	__default_term_handler,			/* 6 core */
	__default_term_handler,			/* 7 term */
	__default_term_handler,			/* 8 core */
	__default_term_handler,			/* 9 term */
	__default_term_handler,			/* 10 core */
	__default_term_handler,			/* 11 core */
	__default_term_handler,			/* 12 core */
	__default_term_handler,			/* 13 term */
	__default_term_handler,			/* 14 term */
	__default_term_handler,			/* 15 term */
	__default_term_handler,			/* 16 term */
	__default_term_handler,			/* 17 term */
	__default_ignore_handler,		/* 18 igor */
	__default_term_handler,			/* 19 term */
	__default_ignore_handler,		/* 20 igor */
	__default_ignore_handler,		/* 21 igor */
	__default_term_handler,			/* 22 term */
	__default_term_handler,			/* 23 stop */
	__default_term_handler,			/* 24 stop */
	__default_ignore_handler,		/* 25 igor */
	__default_term_handler,			/* 26 stop */
	__default_term_handler,			/* 27 stop */
	__default_term_handler,			/* 28 term */
	__default_term_handler,			/* 29 term */
	__default_term_handler,			/* 30 core */
	__default_term_handler,			/* 31 core */
};

/* Function declare */
static int __init_actions(void);			/* Init all default sig action */
static int __prepare_message(void);			/* Register reserver messages */
static void __signal_helper_thread_func(void* arg);	

/**********************************************************************/
/* Function name: __init_sig_lib                                      */
/* Description: initialize signal libary                              */
/* Return type: int                                                   */
/* Argument - void: return 0 on success                               */
/**********************************************************************/
int __init_sig_lib(void)
{
	int ret = 0;

#ifdef MEMORY_CHECK
	mc_init();
#endif

	/* Update main thread id */
	__MAIN_THREAD = tkse_get_tid();

	/* Init global lock */
	pthread_mutexattr_t mutexattr = { PTHREAD_MUTEX_RECURSIVE_NP };
	if( ( ret = nes_posix_pthread_mutex_init(&__g_siglock, &mutexattr) ) != 0 ) 
	{
		goto exit;
	}

	/* Init process's signal block mask */
	sigemptyset(&__g_block_sigmask);

	/* Init all signal default action/handler */
	if( (ret = __init_actions()) != 0 ) 
	{
		nes_posix_pthread_mutex_destroy(&__g_siglock);
		goto exit;
	}

	/* Regist reserve message */ 
	if( (ret = __prepare_message()) != 0 ) 
	{
		nes_posix_pthread_mutex_destroy(&__g_siglock);
		goto exit;
	}

	/* Create helper thread */
	ret = tkse_crs_tsk((FP)__signal_helper_thread_func, 128, NULL);
	if( ret < 0 ) 
	{
		nes_posix_pthread_mutex_destroy(&__g_siglock);
		goto exit;
	}
	
	/* Reset value */
	ret = 0;	

	/* Init pending related data */
	sigemptyset(&(__g_sigpending.mask));
	QueInit(QP(__g_sigpending.queue));
	__g_sigpending.queue_len = 0;

	/* Init thread queue */
	QueInit(QP(__g_sigwait_queue));

	/* Init temporal pending queue */
	QueInit(&__g_tmpQueue);

	/* Auto lib clear */
	atexit(__finish_sig_lib);
exit:
	return ret;
}

/**********************************************************************/
/* Function name: __finish_sig_lib                                    */
/* Description: free signal resource when process end. it auto call   */
/*              when process end                                      */
/* Return type: void                                                  */
/* Argument - void:                                                   */
/**********************************************************************/
void __finish_sig_lib(void)
{
	static int flag = 0;
	sigqueue_t* node = NULL;
	struct __sigwait_queue* node2 = NULL;
	MESSAGE sndmsg = { __MSG_END_THREAD, 0 };

	/* 
	 * Try call again ? 
	 * __finish_sig_lib() can be called manul. 
	 * so __finish_sig_lib() maybe called serveral times
	 */
	if( flag ) 
	{
		return;
	}

	/* 1. Notify helper thread to exit */
	tkse_snd_msg(getpid(), &sndmsg, NOWAIT);

	GLOBAL_SIG_LOCK();

	/* 2. Destroy pending signal queue */
	while( (node = (sigqueue_t*)QueRemoveNext(QP(__g_sigpending.queue))) != NULL ) 
	{
		MC_FREE(node);
		node = NULL;
	} 

	/* 3. Destroy temporal pending signal queue */
	while( (node = (sigqueue_t*)QueRemoveNext(&__g_tmpQueue)) != NULL ) 
	{
		MC_FREE(node);
		node = NULL;
	} 

	/* 4. Destroy all sigwait queue */
	while( (node2 = (struct __sigwait_queue*)QueRemoveNext(QP(__g_sigwait_queue))) != NULL ) 
	{
		MC_FREE(node2);
		node = NULL;
	} 

	/*5. Set excuted flag */
	flag = 1;

	GLOBAL_SIG_UNLOCK();

	/* 6. Destroy global lock */
	nes_posix_pthread_mutex_destroy(&__g_siglock);

#ifdef MEMORY_CHECK
	mc_showinfo();
#endif
	return;
}

/**********************************************************************/
/* Function name: __sig_lock                                          */
/* Description: Signal libray's global lock                           */
/* Return type: int  return 0 on sucess, otherwise return error code  */
/* Argument - void:                                                   */
/**********************************************************************/
int __sig_lock(void)
{
	return nes_posix_pthread_mutex_lock(&__g_siglock);
}

/**********************************************************************/
/* Function name: __sig_unlock                                        */
/* Description: Signal libray's global lock                           */
/* Return type: int  return 0 on sucess, otherwise return error code  */
/* Argument - void:                                                   */
/**********************************************************************/
int __sig_unlock(void)
{	
	return nes_posix_pthread_mutex_unlock(&__g_siglock);
}

/**********************************************************************/
/* Function name: __prepare_message                                   */
/* Description: Regist all MESSAGE than reserved by signal libary     */
/* Return type: int  eturn 0 on sucess, otherwise return  error code  */
/* Argument - void:                                                   */
/**********************************************************************/
static int __prepare_message(void)
{
	/* Use __MSG_TERM(MS_TYPE0) as TERM Handler */
	if( tkse_def_msg(MSGMASK(__MSG_TERM), MH_TERM) != E_OK ) 
	{
		return -1;
	} 

	/* Use __MSG_BREAK(MS_TYPE1) as TERM Handler */
	if( tkse_def_msg(MSGMASK(__MSG_BREAK), MH_BREAK) != E_OK ) 
	{
		return -1;
	}

	return 0;
}

/**********************************************************************/
/* Function name: __send_block_notify                                 */
/* Description:Notification of process signal mask's change.This      */
/*             MESSAGE will send to helper task, and when helper get  */
/*             it, helper task should check all pending signal at once*/
/* Return type: int  0 on success and -1 on failed                    */
/* Argument - void:                                                   */
/**********************************************************************/
int __send_block_notify(void)
{
	MESSAGE sndmsg = { __MSG_BLOCK, 0 };
	return tkse_snd_msg(getpid(), &sndmsg, NOWAIT) == E_OK ? 0 : -1;
}

/**********************************************************************/
/* Function name: __send_break_notify                                 */
/* Description: Notify to helper task to break main thread's          */
/*              excute(block operation)                               */
/* Return type: int  0 on success and -1 on failed                    */
/* Argument - void:                                                   */
/**********************************************************************/
int __send_break_notify(void)
{
	MESSAGE sndmsg = { __MSG_BREAK, 0 };
	return tkse_snd_msg(getpid(), &sndmsg, NOWAIT) == E_OK ? 0 : -1;
}

/**********************************************************************/
/* Function name: __send_term_notify                                  */
/* Description: Notify to helper task to call default MESSAGE term    */
/*              operation                                             */
/* Return type: int  0 on success and -1 on failed                    */
/* Argument - void:                                                   */
/**********************************************************************/
int __send_term_notify(void)
{
	MESSAGE sndmsg = { __MSG_TERM, 0 };
	return tkse_snd_msg(getpid(), &sndmsg, NOWAIT) == E_OK ? 0 : -1;
}

/**********************************************************************/
/* Function name: __default_term_handler                              */
/* Description: Default term operation of process                     */
/* Return type: void                                                  */
/* Argument - int signum: which signal cause the term operation       */
/* Argument - struct siginfo* info: No use here                       */
/* Argument - void* param: No use here                                */
/**********************************************************************/
static void __default_term_handler(int signum, struct siginfo* info, void* param)
{
	DP0(D_LEVEL_INFO, D_MODU_SIG_INTER, ("__default_term_handler %d\n", signum));

	/* Clean signal libary at last */
	__finish_sig_lib();

	/* Just send term message to process */
	__send_term_notify();
}

/**********************************************************************/
/* Function name: __default_ignore_handler                            */
/* Description: Ignore handler of signal                              */
/* Return type: void                                                  */
/* Argument - int signum: which signal is ignored                     */
/* Argument - struct siginfo* info:  No use here                      */
/* Argument - void* param: No use here                                */
/**********************************************************************/
static void __default_ignore_handler(int signum, struct siginfo* info, void* param)
{
	/* Need do nothing */
}

/**********************************************************************/
/* Function name:  __get_default_handler                              */
/* Description: Get signal corresponding dowith handler               */
/* Return type - __sigaction_handler_t:handler function pointer       */
/* Argument - const int signum:                                       */
/**********************************************************************/
static __sigaction_handler_t  __get_default_handler(const int signum)
{
	if( !__signal_is_valid(signum) ) 
	{
		return NULL;
	}

	/* Real time signal default action is TERM */
	if( signum >= SIGRTMIN ) 	
	{	
		return __default_term_handler;
	}

	return __default_signal_handler_table[signum];
}

/**********************************************************************/
/* Function name: __init_actions                                      */
/* Description:  Initialize all signal's relate data                  */
/* Return type - static int : Alway return 0                          */
/* Argument - void:                                                   */
/**********************************************************************/
static int __init_actions(void)
{
	int i = 0;

	GLOBAL_SIG_LOCK();

	memset(&__g_all_actions, 0, sizeof(__g_all_actions));

	/* Init all signal actions */
	for(i = 1; i <= _NSIG; i++ ) 
	{
		__g_all_actions[i].sa_flags |= SA_SIGINFO;
		sigaddset(&(__g_all_actions[i].sa_mask), i);
		__g_all_actions[i].sa_sigaction = (__sigaction_handler_t)SIG_DFL;
	}

	GLOBAL_SIG_UNLOCK();
	return 0;
}

/**********************************************************************/
/* Function name: __is_thread_sigwait_now                             */
/* Description: Thread is calling sigwait() now ?                     */
/* Return type - int: If dest thread is calling sigwait() return 1,   */
/*                    else return 0                                   */
/* Argument - const int thread: Dest thread id                        */
/**********************************************************************/
int __is_thread_sigwait_now(const int thread)
{
	int offset = sizeof(QUEUE);
	struct __sigwait_queue* entry = NULL;

	GLOBAL_SIG_LOCK();
	entry = (struct __sigwait_queue*)QueSearch(QP(__g_sigwait_queue), QP(__g_sigwait_queue), thread, offset);

	/* Reach end, not find target */
	if( (QUEUE*)entry == QP(__g_sigwait_queue) ) 
	{ 
		GLOBAL_SIG_UNLOCK();
		return 0;	
	}
	
	/* Not correct wait type */
	if( entry->type != SIGWAIT_WAIT_TYPE ) 
	{
		GLOBAL_SIG_UNLOCK();
		return 0;	
	}
	GLOBAL_SIG_UNLOCK();
	
	return 1;
}

/**********************************************************************/
/* Function name: __check_thread_sigwait                              */
/* Description:When receive a blocked signal, this function should be */
/*             called.__check_thread_sigwait() check dest thread's    */
/*             status, if dest thread is suspended for sigwait wakeup */
/*             dest thread and record current signal num              */
/* Return type - int: return 0 on success and <0 on failed            */
/* Argument - const int signum: the blocked signal received           */
/* Argument - const int dest_thread: Dest thread id                   */
/* Argument - MESSAGE* msg: original MESSAGE                          */
/**********************************************************************/
int __check_thread_sigwait(const int signum, const int thread, MESSAGE* msg)
{
	int dest_thread = thread;
	int offset = sizeof(QUEUE);
	__pthread_desr* desr = NULL;
	struct __sigwait_queue* entry = NULL;

	/* If thread == 0, means signal send to main thread */
	if( dest_thread == 0 ) 
	{		
		dest_thread = __MAIN_THREAD;
	}

	if( !__signal_is_valid(signum) || dest_thread <= 0 )
	{
		return -1;
	}

	GLOBAL_SIG_LOCK();
	entry = (struct __sigwait_queue*)QueSearch(QP(__g_sigwait_queue), QP(__g_sigwait_queue), dest_thread, offset);

	/* Dest thread not call sigwait() */
	if( (QUEUE*)entry == QP(__g_sigwait_queue) ) 
	{ 
		GLOBAL_SIG_UNLOCK();
		return -2;	
	}

	/* Not expected wait type */
	if( entry->type != SIGWAIT_WAIT_TYPE ) 
	{
		GLOBAL_SIG_UNLOCK();
		return -3;
	}

	/* Is sigwait() wanted signal ? */
	if( !sigismember(&(entry->waitset), signum) ) 
	{
		GLOBAL_SIG_UNLOCK();
		return -4;	
	}

	/* Sigwait() can't be wakeup for those signal that handler is SIG_IGN */
	if( __g_all_actions[signum].sa_handler == SIG_IGN ) 
	{
		GLOBAL_SIG_UNLOCK();
		return -5;
	}

	/* Release unused resource */
	QueRemove((QUEUE*)entry);
	MC_FREE(entry);

	GLOBAL_SIG_UNLOCK();

	/*
	 * Record current signal to TCB, for sigwait() need this 
	 */
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(dest_thread);
	if( !desr ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		return -6;
	}
	desr->__sigwait_signum = signum;
	GLOBAL_THREADMGR_UNLOCK();

	/*
	 * At last, interrupt(wakeup) dest thread 
	 * Normal child only use tkse_wup_tsk,
	 * But main thread can use MH_BREAK
	 */
	if( dest_thread != __MAIN_THREAD ) 
	{
		tkse_wup_tsk(dest_thread);	
	} 
	else 
	{
		__send_break_notify();
	}

	return 0;
}


/**********************************************************************/
/* Function name: __add_pending                                       */
/* Description:When signal blocked, save it to signal pending queue   */
/* Return type int: return 0 on success, otherwise error code         */
/* Argument - const int signum: signal blocked                        */
/* Argument - const int sendpid:  signal sender                       */
/* Argument - MESSAGE* msg: original MESSAGE info                     */
/**********************************************************************/
static int __add_pending(const int signum, const int sendpid, const int thread, MESSAGE* msg)
{
	/* Needn't lock @ this function */


	/* 
	 * Queue overflow.
	 * We may ignore if the signal was rt and sent by user
	 */
	if( signum > SIGRTMIN && __g_sigpending.queue_len >= __SIG_QUEUE_MAX ) 
	{
		return ERANGE;  /* For not define EAGAIN */;
	}

	/* 
	 * 1. If signal is and haven't same signal in the queue, push it to queue
	 * Else ingore this signal dirctly.
	 */
	if( signum < SIGRTMIN && sigismember(&(__g_sigpending.mask), signum) ) 
	{
		return 0;
	}

	/* 2. If rt-signal, save all info of the signal to queue */
	sigqueue_t *entry = (sigqueue_t*)MC_MALLOC(sizeof(sigqueue_t));
	if( !entry ) 
	{
		return ENOMEM;
	}

	entry->signum = signum;
	entry->sender = sendpid;
	entry->thread = thread;
	entry->msg = *msg;
	
	QueInsert((QUEUE*)entry, QP(__g_sigpending.queue));
	__g_sigpending.queue_len++;

	/* 3. Set flag to __g_sigpending.mask */
	sigaddset(&(__g_sigpending.mask), signum);

	return 0;
}

/**********************************************************************/
/* Function name: __find_pending_signal                    */
/* Description: Get pending signal's info by signal                   */
/* Return type - sigqueue_t*: If found related node, return valid     */
/*                            pointer, otherwise NULL                 */
/* Argument - const int signum:  dest signal number                   */
/**********************************************************************/
sigqueue_t* __find_pending_signal(const int signum)
{
	sigqueue_t* entry = NULL;
	int offset = sizeof(QUEUE);
	entry = (sigqueue_t*)QueSearch(QP(__g_sigpending.queue), QP(__g_sigpending.queue), signum, offset);
	
	/* Reach end, not find target */
	if( (QUEUE*)entry == QP(__g_sigpending.queue) ) 
	{ 
		return NULL;	
	}

	return entry;
}

/**********************************************************************/
/* Function name: __remove_pending_signal   */
/* Description:  remove entry form pending queue  */
/* Return type - void :     */
/* Argument - sigqueue_t* entry:  */
/**********************************************************************/
void __remove_pending_signal(sigqueue_t* entry)
{
	QueRemove((QUEUE*)entry);
	__g_sigpending.queue_len--;
}

/**********************************************************************/
/* Function name: __find_and_remove_pending_signal                    */
/* Description: Get pending signal's info by signal and remove it from pending queue                */
/* Return type - sigqueue_t*: If found related node, return valid     */
/*                            pointer, otherwise NULL                 */
/* Argument - const int signum:  dest signal number                   */
/**********************************************************************/
sigqueue_t* __find_and_remove_pending_signal(const int signum)
{
	sigqueue_t* entry = __find_pending_signal(signum);
	if( entry ) 
	{
		__remove_pending_signal(entry);
	}
	return entry;
}


/**********************************************************************/
/* Function name: __fill_siginfo                                      */
/* Description:Use originalMESSAGE info to build siginfo_t data struct */
/* Return type - int: Return 0 on success, otherwise return -1        */
/* Argument - siginfo_t* siginfo: siginfo_t data struct wait to build */
/* Argument - const int signum:                                       */
/* Argument - const int sendpid: signal sender                        */
/* Argument - const MESSAGE* msg: Original MESSAGE info               */
/**********************************************************************/
static int __fill_siginfo(siginfo_t* siginfo, const int signum, const int sendpid, const MESSAGE* msg)
{
	/*
	 * msg->msg_body.ANYMSG.msg_str[32]
	 *
	 * 0       4         8       12       16       20       24        28       32  bytes
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 * |signum  |sigdata | sender | thread |        |        |        |        |     
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 */
	void* sigdata = (void*)*(int*)(msg->msg_body.ANYMSG.msg_str + 4);
	int sender = *(int*)(msg->msg_body.ANYMSG.msg_str + 8);

	/* Gernal info */
	siginfo->si_signo = signum;
	siginfo->si_errno = 0;
	siginfo->si_code = sender;

	if( signum >=  SIGRTMIN ) 
	{
		siginfo->_sifields._rt._pid = sendpid;
		siginfo->_sifields._rt._sigval.sival_ptr = sigdata;
		return 0;
	}

	switch(signum)
	{
	case SIGCHLD:
		siginfo->_sifields._sigchld._pid = sendpid;
		if( msg->msg_type == MS_ABORT ) siginfo->_sifields._sigchld._status = msg->msg_body.ABORT.code;
		if( msg->msg_type == MS_EXIT ) siginfo->_sifields._sigchld._status = msg->msg_body.EXIT.code;
		if( msg->msg_type == MS_TERM ) siginfo->_sifields._sigchld._status = msg->msg_body.TERM.code;
		break;
	default:
		return -1;
	    break;
	}
	
	return 0;
}

/**********************************************************************/
/* Function name: __dowith_message                                    */
/* Description: Core function of signal libary.                       */
/*		     1. Transfer from MESSAGE to signal               */
/*		     2. Dowith signal need block or not               */
/*		     3. Call signal corresponding handler             */
/*		     4. Dowith signal's kinds of flag                 */
/* Return type - int: Return 0 on success, otherwise return -1        */
/* Argument - const int sendpid: Signal sender                        */
/* Argument - MESSAGE* msg: Original MESSAGE info                     */
/**********************************************************************/
static int __dowith_message(const int sendpid, MESSAGE* msg, int ispending)
{
	int signum = 0;			/* Linux siganl num */
	void* sigdata = NULL;		/* Linux siganl data */
	int tid = 0;
	siginfo_t siginfo;			/* Handler param */
	int break_main_thread = 0;	/* Need break main thread ? */

	sigset_t thread_block_mask;
	__pthread_desr* desr = NULL;

	/* Local var init */
	memset(&siginfo, 0, sizeof(siginfo_t));
	sigemptyset(&thread_block_mask);

        /* Check MSG invalid */
	if( msg->msg_type != __MSG_UNREALTIME && msg->msg_type != __MSG_REALTIME ) 
	{
		return -1;
	}

	/*
	 * msg->msg_body.ANYMSG.msg_str[32]
	 *
	 * 0       4         8       12       16       20       24        28       32  bytes
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 * |signum  |sigdata | sender | thread |        |        |        |        |     
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 */
	signum	= *(int*)(msg->msg_body.ANYMSG.msg_str);
	sigdata	= (void*)*(int*)(msg->msg_body.ANYMSG.msg_str + 4);
	tid	= *(int*)(msg->msg_body.ANYMSG.msg_str + 12);

	if( !__signal_is_valid(signum) ) 
	{
		return -2;
	}

	/* Get thread's signal block mask */
	if( tid && tid != __MAIN_THREAD ) 
	{
		/* 
		 * Get real signal block mask 
		 * Main thread's is Process block mask
		 */
		GLOBAL_THREADMGR_LOCK();
		desr = __pthread_find_desr(tid);
		if( !desr ) /* Invalid thread */
		{		
			GLOBAL_THREADMGR_UNLOCK();
			return -3;		
		}
		thread_block_mask = desr->__block_sigmask;
		GLOBAL_THREADMGR_UNLOCK();
	}

	GLOBAL_SIG_LOCK();

	/*
	 * If signal is blocked, just push it to queue 
	 * When the signal is unblock, then do it's handler 
	 */
	if( sigismember(&__g_block_sigmask, signum ) ) /* Signal is blocked */
	{		
		if( !ispending) __add_pending(signum, sendpid, tid, msg);
		__check_thread_sigwait(signum, tid, msg);
		GLOBAL_SIG_UNLOCK();
		return 1;
	}

	break_main_thread = __g_all_actions[signum].sa_flags & SA_NOTINTR;

 	if( tid && tid != __MAIN_THREAD ) 
	{
		if (sigismember(&thread_block_mask, signum ) ) 		/* Signal is blocked by thread mask */
		{	
			if( !ispending) __add_pending(signum, sendpid, tid, msg);
 			GLOBAL_SIG_UNLOCK();
 			return 1;
		}
 	}

	/* Reset alarm info */
	if( signum == SIGALRM ) 
	{
		__g_alarm_time.__alive_time = 0;
		__g_alarm_time.__send_time = 0;
	}

	/* Ingore. Needn't analyse MESSAGE */
	if( __g_all_actions[signum].sa_handler == SIG_IGN ) 
	{
		GLOBAL_SIG_UNLOCK();
		goto exit;
	}

        /* Default. Needn't analyse MESSAGE */
	if( __g_all_actions[signum].sa_handler == SIG_DFL ) 
	{
		__sigaction_handler_t default_handler =  __get_default_handler(signum);
		GLOBAL_SIG_UNLOCK();
		default_handler(signum, &siginfo, NULL);
		goto exit;
	}

	/* User Application handler */
	/* Old type handler(sa_handler type) without sig param */
        if( !(__g_all_actions[signum].sa_flags & SA_SIGINFO) ) 
	{   
		__sighandler_t tmp_sighandler = __g_all_actions[signum].sa_handler;
		GLOBAL_SIG_UNLOCK();
		tmp_sighandler(signum);      
		goto exit;
	}
	
	/* New type handler(sa_sigaction type) without sig param */
	__sigaction_handler_t tmp_sigaction_handler = __g_all_actions[signum].sa_sigaction;
	GLOBAL_SIG_UNLOCK();

	__fill_siginfo(&siginfo, signum, sendpid, msg);
	tmp_sigaction_handler(signum, &siginfo, NULL);       /* Call handler with param*/

exit:
	/* Reset handler if needed*/
	if( __g_all_actions[signum].sa_flags & SA_ONESHOT) 
	{ 
		GLOBAL_SIG_LOCK();
		__g_all_actions[signum].sa_handler = SIG_DFL;
		GLOBAL_SIG_UNLOCK();
	}

	/*
	 * Notify Process or thread
	 */
	if( tid > 0 && tid != __MAIN_THREAD ) 	/* Normal thread */
	{	
		if( __is_thread_sigwait_now(tid) ) 
		{
			return 0;
		}
		tkse_wup_tsk(tid);

	} else {

		if( __is_thread_sigwait_now(__MAIN_THREAD) ) 
		{
			return 0;
		}
		/* 
		* Anyway, process(main thread) should be interrupted 
		* Interrupt main thread's block operation ( If exist ) 
		*/
		if( !break_main_thread ) 		/* SA_NOTINTR is set */
		{
			__send_break_notify();
		}
	}

	return 0;
}

/**********************************************************************/
/* Function name: __check_pending                                     */
/* Description:When process block mask update, check allpending signal*/
/* Return type - int: Alway return 0                                  */
/* Argument - void:                                                   */
/**********************************************************************/
static int __check_pending(void)
{
	int signum = 1;
	int clear_flag = 1;			
	sigqueue_t* entry = NULL;
	sigset_t block_mask;	

	if( isQueEmpty(QP(__g_sigpending.queue)) ) 
	{
		goto exit;
	}

	/* __g_block_sigmask need protect to access */
	GLOBAL_SIG_LOCK();
	block_mask = __g_block_sigmask;
	GLOBAL_SIG_UNLOCK();

	QueInit(&__g_tmpQueue);

	/* Get unblock signals */
	for( signum = 1; signum <= _NSIG; signum++ ) 
	{
		/* Reset flag when new signal */
		clear_flag = 1;	

		/* 
		 * If a signal is not blocked and it has some node in pending queue,
		 * Then need to call it's hander to dowith it
		 */
		if( !sigismember(&block_mask, signum) && sigismember(&(__g_sigpending.mask), signum) ) 
		{
			while( 1 ) 
			{
				entry = __find_pending_signal(signum);
				if( !entry ) break;

				/*
				 * If __dowith_message() not return. means signal term current proccess, 
				 * Under this case,  has no change to FREE entry.
				 * And memory leak will happen, for prevent this case,
				 * We leave the entry on pending queue temporally, for __finish_sig_lib() can
				 * free all pending signal in the pending queue.
				 */
				if( __dowith_message(entry->sender, &entry->msg, 1) == 1 )  /* Block still */
				{
					__remove_pending_signal(entry);
					QueInsert((QUEUE*)entry, &__g_tmpQueue);			   /* Save to temporay queue */
					clear_flag = 0;							   /* Can't clear all from pending queue */
					continue;
				}

				__remove_pending_signal(entry);

				MC_FREE(entry);
				entry = NULL;
			}

			/* Remove signal from pending mask */
			if( clear_flag ) 
			{
				sigdelset(&(__g_sigpending.mask), signum);
			}

		} /* if */

	} /* for */

	/* Recover all blocked signal from temp queue to pending queue*/
	while( ((QUEUE*)entry = QueRemoveNext(&__g_tmpQueue)) != NULL ) 
	{
		QueInsert((QUEUE*)entry, QP(__g_sigpending.queue));
		__g_sigpending.queue_len++;		/* Remeber update pending queue len */
	}

exit:
	return 0;
}

/**********************************************************************/
/* Function name: __signal_helper_thread_func                         */
/* Description:Signal libary logic's implement.Shuold be create when  */
/*             process begin                                          */
/* Return type - void:                                                */
/* Argument - void* arg: No use now                                   */
/**********************************************************************/
static VOID __signal_helper_thread_func(void* arg)
{
	int ret = 0;
	unsigned int mask;
	MESSAGE msg;

	mask = __ALL_RECEIVE_MASK | MM_ABORT | MM_TERM | MM_EXIT | MM_TMOUT | MM_SYSEVT;

	while( 1 ) 
	{
		memset(&msg, 0, sizeof(msg));
		ret = tkse_rcv_msg(mask, &msg, sizeof(msg),  WAIT|CLR);

		if( ret < 0 ) 
		{
			continue;
		}

		if( ret == 0 )  				/* Must be self process sended */
		{	
			//__check_pending();		/* Check pending when receive one message */
		}

		switch(msg.msg_type)
		{
		case __MSG_BLOCK:			/* Block update notify*/
			if( ret == 0 )  
			{
				__check_pending();	/* Must be self process sended */
			}
			break;
		case __MSG_END_THREAD:		/* Exit notify */
			if( ret == 0 )  
			{
				goto exit;			/* Must be self process sended */
			}
			break;
		case MS_TMOUT:				/* SIGALRM */
			if( ret > 0 ) 
			{			/* Only self process can send timeout siganl */
				break;
			}
			/* Change to SIGALRM message and call __dowith_message() */
			msg.msg_type = __MSG_UNREALTIME;
			msg.msg_size = 8;
			*(int*)(msg.msg_body.ANYMSG.msg_str) = SIGALRM;
			*(int*)(msg.msg_body.ANYMSG.msg_str + 4) = 0;

		case __MSG_UNREALTIME:		/* RM signal */
		case __MSG_REALTIME:		/* UnRM signal */
			__dowith_message((ret ? ret : getpid()), &msg, 0);
			break;

		case MS_EXIT:				/* SIGCHLD SubProcess exited */
		case MS_TERM:				/* SIGCHLD SubProcess term */
		case MS_ABORT:				/* SIGCHLD SubProcess abort */
			msg.msg_type = __MSG_UNREALTIME;
			msg.msg_size = 8;
			*(int*)(msg.msg_body.ANYMSG.msg_str) = SIGCHLD;
			*(int*)(msg.msg_body.ANYMSG.msg_str + 4) = 0;
			__dowith_message((ret ? ret : getpid()), &msg, 0);
			break;
		default:					/* Unknow message */
			break;
		}
	}

exit:
	tkse_ext_tsk();	
}


