/**
 * @file nes_posix_mutex.c
 * @brief implement of mutex operation 
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "nes_posix_error.h"
#include "nes_posix_mutex.h"
#include "nes_posix_pthread.h"
#include "common.h"
#include "internal_lock.h"

/**********************************************************************/
/* Function name: nes_posix_pthread_mutex_init                                  */
/* Description:  create mutex                                         */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EINVAL: param invalid                             */
/* Argument - pthread_mutex_t * mutex:                                */
/* Argument - mutex_attr:  init param, only one attribute : type      */
/**********************************************************************/
int nes_posix_pthread_mutex_init(pthread_mutex_t* mutex, const pthread_mutexattr_t *attr) 
{
	int nRet;
	int nKind = PTHREAD_MUTEX_ADAPTIVE_NP;	/* Set default lock type */

	if( !mutex )
	{
		nRet = EINVAL;
		goto exit;
	}

	/* 
	 * Check mutex type
	 */
	if( attr )
	{
		if( attr->__mutexkind != PTHREAD_MUTEX_ADAPTIVE_NP &&
		    attr->__mutexkind != PTHREAD_MUTEX_RECURSIVE_NP &&
		    attr->__mutexkind != PTHREAD_MUTEX_ERRORCHECK_NP )
		{
			nRet = EINVAL;
			goto exit;
		} 
		nKind = attr->__mutexkind;
	}

    	memset(mutex, 0, sizeof(pthread_mutex_t));
	mutex->__m_kind = nKind;
	
	/* Here, for PMC Extention not support mutex, 
	 * So use semaphore to simulate mutex, make sure the 
	 * semaphore's resouce count must be 1
	 */
	nRet = __internal_lock_init(&(mutex->__m_lock));

exit:
	return nRet;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutex_destroy                               */
/* Description:  Destroy MUTEX.                                       */
/* Return type: int  0 : successfully, not zero value when error      */
/* Argument - pthread_mutex_t *mutex:  mutex handle                   */
/**********************************************************************/
int nes_posix_pthread_mutex_destroy(pthread_mutex_t *mutex)  
{
	/* 
	 * At linux, pthread_mutex_destroy may be return EBUSY,
	 * But here sem_destroy never return some relate errorcode
	 * So, here we just return the source error code.
	 */
	return __internal_lock_destroy(&(mutex->__m_lock));
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutex_trylock                               */
/* Description:  Try to lock MUTEX.                                   */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EBUSY: try lock failed  */
/*                  EDEADLK: a ERRORCHECK lock want to lock self again */
/*                  EINVAL: invalid lock type */
/* Argument - pthread_mutex_t *mutex:  mutex handle                   */
/**********************************************************************/
int nes_posix_pthread_mutex_trylock(pthread_mutex_t *mutex) 
{
	int ret = 0;
	pthread_t wSelf = nes_posix_pthread_self();	/* Get current thread's id */

	if( !mutex ) 
	{
		return EINVAL;
	}

	switch(mutex->__m_kind) 
	{
	case PTHREAD_MUTEX_ADAPTIVE_NP:			/* Normal lock( Fast lock) */
	case PTHREAD_MUTEX_TIMED_NP:			/* Normal lock( Fast lock) with timeout, not support now*/
		ret = __internal_try_lock(&(mutex->__m_lock));
		// if( ret ) return EBUSY;			/* Try failed */
		return ret;

	case PTHREAD_MUTEX_RECURSIVE_NP:	       	/* Recursive lock, Lock again in one thread always ok */
		if (mutex->__m_owner == wSelf ) 	/* Did self thread want to lock again ?*/
		{
			mutex->__m_count++;		/* Record recursive depth use by unlock */
			return 0;			/* Lock successfully */
		}
		ret = __internal_try_lock(&(mutex->__m_lock));	/* Another thread( or first use mutex )want to get lock, just to try*/
		if( ret ) return ret;			/* Try failed */
		mutex->__m_owner = wSelf;		/* Lock ok, mutex owner is changed */
		mutex->__m_count = 1;		      	/* Init recursive depth is : 1 */
		return 0;				/* Lock successfully */

	case PTHREAD_MUTEX_ERRORCHECK_NP:	     	/* Errorcheck lock, return EDEADLK when lock again in one thread */
		if (mutex->__m_owner == wSelf) return EDEADLK;   /* lock agin is in self thread, so return error */
		ret = __internal_try_lock(&(mutex->__m_lock));   /* Another thread want to get lock, just to try*/ 
		if( ret ) return ret;			/* Try failed */
		mutex->__m_owner = wSelf;		/* Lock ok, mutex owner is changed  */
		return 0;

	default:
		return EINVAL;				/* Unknow lock , do nothing */
	};

	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutex_lock                                  */
/* Description:Wait until lock for MUTEX becomes available and lock it*/
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*                  EDEADLK: a ERRORCHECK lock want to lock self again */
/*                  EINVAL: invalid lock type */
/* Argument - pthread_mutex_t *mutex:  mutex handle                   */
/**********************************************************************/
int nes_posix_pthread_mutex_lock(pthread_mutex_t *mutex) 
{
	int ret = 0;
	pthread_t wSelf = nes_posix_pthread_self();	/* Get current thread's id */
	
	if( !mutex ) 
	{
		return EINVAL;
	}

	switch(mutex->__m_kind) 
	{
	case PTHREAD_MUTEX_ADAPTIVE_NP:			/* Normal lock( Fast lock) */
	case PTHREAD_MUTEX_TIMED_NP:			/* Normal lock( Fast lock) with timeout, not support now*/
		ret = __internal_lock(&(mutex->__m_lock));	/* Just wait one by one */
		if( ret ) return ret;				/* Internal error happened, just return */
		// mutex->__m_owner = wSelf;				/* New owner */
		return 0;

	case PTHREAD_MUTEX_RECURSIVE_NP:	       	/* Recursive lock, Lock again in one thread always ok */
		if (mutex->__m_owner == wSelf ) 	/* Did self thread want to lock again ?*/
		{
			mutex->__m_count++;		/* Record recursive depth use by unlock */
			return 0;			/* Lock successfully */
		}
		ret = __internal_lock(&(mutex->__m_lock));	/* Another thread( or first use mutex )want to get lock, just to wait*/
		if( ret ) return ret;				/* Internal error happened, just return */
		mutex->__m_owner = wSelf;		/* Lock ok, mutex owner is changed */
		mutex->__m_count = 1;		      	/* Init recursive depth is : 1 */
		return 0;				/* Lock successfully */

	case PTHREAD_MUTEX_ERRORCHECK_NP:	     	/* Errorcheck lock, return EDEADLK when lock again in one thread */
		if (mutex->__m_owner == wSelf) return EDEADLK;   /* lock agin is in self thread, so return error */
		ret = __internal_lock(&(mutex->__m_lock));	/* Another thread want to get lock, just to wait*/ 
		if( ret ) return ret;				/* Internal error happened, just return */
		mutex->__m_owner = wSelf;		/* Lock ok, mutex owner is changed  */
		return 0;

	default:
		return EINVAL;				/* Unknow lock , do nothing */
	};

	return 0;
}


/**********************************************************************/
/* Function name: nes_posix_pthread_mutex_unlock                                */
/* Description:   Unlock MUTEX.                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*  EPERM: a ERRORCHECK or RECURSIVE lock want to unlock not own mutex*/
/*  EINVAL: invalid lock type                                         */
/* Argument - pthread_mutex_t *mutex:  mutex handle                   */
/**********************************************************************/
int nes_posix_pthread_mutex_unlock(pthread_mutex_t *mutex) 
{
	int ret = 0;
	pthread_t wSelf = nes_posix_pthread_self(); /* Get current thread's id */

	if( !mutex ) 
	{
		return EINVAL;
	}

	switch (mutex->__m_kind) 
	{
	case PTHREAD_MUTEX_ADAPTIVE_NP:			/* Normal lock( Fast lock) */
	case PTHREAD_MUTEX_TIMED_NP:			/* Normal lock( Fast lock) with timeout, not support now*/
		// if( mutex->__m_owner != wSelf ) 
		//	return EPERM;
		ret = __internal_unlock(&(mutex->__m_lock));	/* Just post */
		return ret;

	case PTHREAD_MUTEX_RECURSIVE_NP:		/* Recursive lock, always unlock ok if depth > 1 */
		if (mutex->__m_owner != wSelf )		/* No right to post */
			return EPERM;
		if (mutex->__m_count > 1 ) 		/* Decrease recursive depth, if depth = 1, means recursion is over */
		{
			mutex->__m_count--;
			return 0;
		}
		mutex->__m_owner = 0;			/* Lock is free, so no owner now */
		mutex->__m_count = 0;			/* Lock is free, so depth is init */
		ret = __internal_unlock(&(mutex->__m_lock));		
		return ret;

	case PTHREAD_MUTEX_ERRORCHECK_NP:		/* Errorcheck lock */
		if (mutex->__m_owner != wSelf )		/* No right to post */
			return EPERM;
		mutex->__m_owner = 0;			/* Lock is free, so no owner now */
		ret = __internal_unlock(&(mutex->__m_lock));
		return ret;

	default:
		return EINVAL;				/* Unknow lock , do nothing */
	};

	return 0;
}

/* Functions for handling mutex attributes.  */

/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_init                              */
/* Description:	Initialize mutex attribute object ATTR with           */
/*		default attributes (kind is PTHREAD_MUTEX_ADAPTIVE_NP)*/
/* Return type: int  always return zero                               */
/* Argument - pthread_mutexattr_t*attr:mutex attribute struct pointer */
/**********************************************************************/
int nes_posix_pthread_mutexattr_init(pthread_mutexattr_t *attr) 
{
	/* Defualt lock type is : PTHREAD_MUTEX_ADAPTIVE_NP */
	if( attr ) 
	{
		attr->__mutexkind = PTHREAD_MUTEX_ADAPTIVE_NP;
	}
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_destroy                           */
/* Description: Destroy mutex attribute object ATTR.                  */
/* Return type: int   always return zero                              */
/* Argument - pthread_mutexattr_t *attr:  no use here                 */
/**********************************************************************/
int nes_posix_pthread_mutexattr_destroy(pthread_mutexattr_t *attr) 
{
	/* Here, we needn't do anything */
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_gettype                        */
/* Description:  Get lock type                                        */
/* Return type: int    always zero                                    */
/* Argument -pthread_mutexattr_t*attr: mutex attribute struct pointer */
/* Argument - int* kind: the buffer to store type                     */
/**********************************************************************/
int nes_posix_pthread_mutexattr_gettype(const pthread_mutexattr_t* attr, int* type)
{
	*type = attr->__mutexkind;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_getkind_np                        */
/* Description:  Same as pthread_mutexattr_gettype                                       */
/* Return type: int    always zero                                    */
/* Argument -pthread_mutexattr_t*attr: mutex attribute struct pointer */
/* Argument - int* kind: the buffer to store type                     */
/**********************************************************************/
int nes_posix_pthread_mutexattr_getkind_np(const pthread_mutexattr_t *attr, int* kind)
{
	return nes_posix_pthread_mutexattr_gettype(attr, kind);
}

/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_settype                        */
/* Description:   Set lock type                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: invalid lock type                             */
/* Argument - pthread_mutexattr_t* attr:mutex attribute struct pointer*/
/* Argument - int kind: the type to set.                              */
/**********************************************************************/
int nes_posix_pthread_mutexattr_settype(pthread_mutexattr_t* attr, int type)
{
	/* If find unknow lock type, just return error */
	if( type != PTHREAD_MUTEX_ADAPTIVE_NP	&&
		type != PTHREAD_MUTEX_RECURSIVE_NP	&&
		type != PTHREAD_MUTEX_ERRORCHECK_NP ) 
	{
		return EINVAL;
	}

	attr->__mutexkind = type;
	return 0;
}


/**********************************************************************/
/* Function name: nes_posix_pthread_mutexattr_setkind_np                        */
/* Description:   Same as pthread_mutexattr_settype                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: invalid lock type                             */
/* Argument - pthread_mutexattr_t* attr:mutex attribute struct pointer*/
/* Argument - int kind: the type to set.                              */
/**********************************************************************/
int nes_posix_pthread_mutexattr_setkind_np(pthread_mutexattr_t *attr, int kind)
{
	return nes_posix_pthread_mutexattr_settype(attr, kind);
}
