/**
 * @file nes_posix_pthread.h
 * @brief pthread libary core header file
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __NES_POSIX_PTHREAD_H_
#define __NES_POSIX_PTHREAD_H_

#include <basic.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef NES_SE   /* Use Standard Extension */
#else           /* Use PMC Extension */
#include <btron/errcode.h>
#include <btron/proctask.h>
#endif
#include "nes_posix_error.h"
#include "nes_posix_mutex.h"
#include "nes_posix_cond.h"
#include "nes_posix_signal.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Init pthread libary, must be called at first when use pthread*/
IMPORT int __pthread_lib_init(void);
#define PTHREAD_INIT()	__pthread_lib_init()

/* Init pthread libary, must be called at end when not use pthread more*/
IMPORT void __pthread_lib_finish(void);
#define PTHREAD_FINISH()	__pthread_lib_finish()

/*
 * Thread HANDLE define
 */
typedef int pthread_t;

/* define thread canceled or not */
#define PTHREAD_CANCELED (void *) -1

/*
 * Thread scheduling policies strategy
 *
 * For linux is not a real REALTIME os, so default value is SCHED_OTHER
 * And t-kernal is realtime os, so the default value should be SCHED_FIFO or SCHED_RR
 * But, the pthread is bulid on PMC extention, we can't control anything excepte PRIORITY
 * So, Those 3 type is same here
 */
enum
{
	SCHED_RR,					/* Real time */
	SCHED_FIFO = SCHED_RR,		/* Real time */
	SCHED_OTHER = SCHED_RR,		/* Not realtime , default value */
};

/*
 *  Thread priority define
 */
enum
{
	PRIORITY_NORMAL = -1,
};

/*
 *  Thread INHERIT type define
 */
enum
{
	PTHREAD_INHERIT_SCHED = 0,
	PTHREAD_EXPLICIT_SCHED,
};

/*
 *  Thread create type define
 */
enum
{
	PTHREAD_CREATE_JOINABLE,
	PTHREAD_CREATE_DETACHED,
};

/*
 * Thread scope define
 */
enum
{
	PTHREAD_SCOPE_SYSTEM,
	PTHREAD_SCOPE_PROCESS,
};

struct sched_param 
{
	int sched_priority;
};

/*
 * Thread init attribute define
 */
typedef struct
{
	int __detachstate;
	int __schedpolicy;
	struct sched_param __schedparam;
	int __inheritsched;
	int __scope;
	size_t __guardsize;
	int __stackaddr_set;
	void *__stackaddr;
	size_t __stacksize;
} pthread_attr_t;


/* Once-only execution */
enum
{
	PTHREAD_ONCE_INIT = 0,
	PTHREAD_ONCE_DONE,
};

typedef int pthread_once_t;

/* TSD */
typedef int pthread_key_t;


/* Cancellation */
struct _pthread_cleanup_buffer			/* Cleanup buffers */
{
	void (*__routine) (void *);			/* Function to call.  */
	void *__arg;					/* Its argument.  */
	struct _pthread_cleanup_buffer *__prev; /* Chaining of cleanup functions.  */
};

enum
{
	PTHREAD_CANCEL_ENABLE ,		/* thread can be canceled:default value*/
	PTHREAD_CANCEL_DISABLE,		/* ignore the cancel request*/
};

enum
{
	PTHREAD_CANCEL_ASYNCHRONOUS,/* cancel as soon as :default value */
	PTHREAD_CANCEL_DEFERRED ,	/* delay the cancel request to the next cancelation point */                      
	
};



/*
 * Create a thread with given attributes ATTR (or default attributes
 * if ATTR is NULL), and call function START_ROUTINE with given
 * arguments ARG.
 */
IMPORT int nes_posix_pthread_create (pthread_t* thread, const pthread_attr_t* attr, void (*start_routine)(void*), void* arg);

 /*
  * Make calling thread wait for termination of the thread TH.  The
  *exit status of the thread is stored in *THREAD_RETURN, if THREAD_RETURN
  *is not NULL.
  */
IMPORT int nes_posix_pthread_join(pthread_t thread, void** thread_return);

/* Obtain the identifier of the current thread.  */
IMPORT  pthread_t nes_posix_pthread_self (void);

/* Compare two thread identifiers.  */
IMPORT int nes_posix_pthread_equal (pthread_t thread1, pthread_t thread2);

/* Terminate calling thread.  */
IMPORT void nes_posix_pthread_exit (void *retval);

 /* 
  * Indicate that the thread TH is never to be joined with PTHREAD_JOIN.
  * The resources of TH will therefore be freed immediately when it
  * terminates, instead of waiting for another thread to perform PTHREAD_JOIN
  * on it.  
  */
IMPORT int nes_posix_pthread_detach(pthread_t  th);

 /***************** Functions for handling attributes.  *************************/

  /* 
   * Initialize thread attribute *ATTR with default attributes
   * (detachstate is PTHREAD_JOINABLE, scheduling policy is SCHED_OTHER,
   * no user-provided stack).  
   */
IMPORT int nes_posix_pthread_attr_init(pthread_attr_t * attr);

 /* Destroy thread attribute *ATTR.  */
IMPORT int nes_posix_pthread_attr_destroy(pthread_attr_t * attr);

 /* Set the `detachstate' attribute in *ATTR according to DETACHSTATE.  */
IMPORT int nes_posix_pthread_attr_setdetachstate(pthread_attr_t* attr, int detachstate);

 /* Return in *DETACHSTATE the `detachstate' attribute in *ATTR.  */
IMPORT int nes_posix_pthread_attr_getdetachstate(const pthread_attr_t* attr, int* detachstate);

 /* Set scheduling parameters(priority, etc) in *ATTR according to PARAM.  */
IMPORT int nes_posix_pthread_attr_setschedparam(pthread_attr_t* attr, const struct sched_param* param);

 /* Return in *PARAM the scheduling parameters of *ATTR.  */
IMPORT int nes_posix_pthread_attr_getschedparam(const pthread_attr_t* attr, struct sched_param*  param);

 /* Set scheduling policy in *ATTR according to POLICY.  */
IMPORT int nes_posix_pthread_attr_setschedpolicy(pthread_attr_t* attr, int  policy);

 /* Return in *POLICY the scheduling policy of *ATTR.  */
IMPORT int nes_posix_pthread_attr_getschedpolicy(const pthread_attr_t* attr, int*  policy);

 /* Set scheduling inheritance mode in *ATTR according to INHERIT.  */
IMPORT int nes_posix_pthread_attr_setinheritsched(pthread_attr_t* attr, int  inherit) ;

 /* Return in *INHERIT the scheduling inheritance mode of *ATTR.  */
IMPORT int nes_posix_pthread_attr_getinheritsched(const pthread_attr_t* attr, int* inherit);

 /* Set scheduling contention scope in *ATTR according to SCOPE.  */
IMPORT int nes_posix_pthread_attr_setscope(pthread_attr_t* attr, int  scope);

 /* Return in *SCOPE the scheduling contention scope of *ATTR.  */
IMPORT int nes_posix_pthread_attr_getscope(const pthread_attr_t*  attr, int*  scope);

  /* 
   * The following two interfaces are intended to replace the last two.  They
   * require setting the address as well as the size since only setting the
   * address will make the implementation on some architectures impossible.  
   */
IMPORT int nes_posix_pthread_attr_setstack(pthread_attr_t* attr, void* stackaddr, size_t stacksize);

  /* Return the previously set address for the stack.  */
IMPORT  int nes_posix_pthread_attr_getstack(const pthread_attr_t* attr, void** stackaddr, size_t* stacksize);

 /* 
  * Set the starting address of the stack of the thread to be created.
  * Depending on whether the stack grows up or down the value must either
  * be higher or lower than all the address in the memory block.  The
  * minimal size of the block must be PTHREAD_STACK_MIN.  
  */
IMPORT int nes_posix_pthread_attr_setstackaddr(pthread_attr_t* attr, void* stackaddr) ;

 /* Return the previously set address for the stack.  */
IMPORT int nes_posix_pthread_attr_getstackaddr(const pthread_attr_t* attr, void**  stackaddr);

 /* 
  * Add information about the minimum stack size needed for the thread
  * to be started.  This size must never be less than PTHREAD_STACK_MIN
  * and must also not exceed the system limits.  
  */
IMPORT int nes_posix_pthread_attr_setstacksize(pthread_attr_t* attr, size_t stacksize);

 /* Return the currently used minimal stack size.  */
IMPORT int nes_posix_pthread_attr_getstacksize(const pthread_attr_t* attr, size_t* stacksize);

 /* 
  * Set the scheduling parameters for TARGET_THREAD according to POLICY
  * and *PARAM.  
  */
IMPORT int nes_posix_pthread_setschedparam(pthread_t target_thread, int  policy, const struct sched_param * param);

 /* Return in *POLICY and *PARAM the scheduling parameters for TARGET_THREAD.  */
IMPORT int nes_posix_pthread_getschedparam(pthread_t  target_thread, int* policy, struct sched_param* param);

/* Set the size of the guard area at the bottom of the thread.  */
IMPORT int nes_posix_pthread_attr_setguardsize(pthread_attr_t* attr, size_t guardsize);

/* Get the size of the guard area at the bottom of the thread.  */
IMPORT int nes_posix_pthread_attr_getguardsize(const pthread_attr_t* attr, size_t* guardsize);

/* 
 * Initialize thread attribute *ATTR with attributes corresponding to the
 * already running thread TH.  It shall be called on unitialized ATTR
 * and destroyed with pthread_attr_destroy when no longer needed.  
 */
IMPORT int nes_posix_pthread_getattr_np(pthread_t thread, pthread_attr_t *attr);

/* Guarantee that the initialization function INIT_ROUTINE will be called
only once, even if pthread_once is executed several times with the
same ONCE_CONTROL argument. ONCE_CONTROL must point to a static or
extern variable initialized to PTHREAD_ONCE_INIT.

The initialization functions might throw exception which is why
this function is not marked with __THROW.  */
IMPORT int nes_posix_pthread_once(pthread_once_t *__once_control, void (*__init_routine) (void));

/* Functions for handling thread-specific data.  */

/* Create a key value identifying a location in the thread-specific
data area.  Each thread maintains a distinct thread-specific data
area.  DESTR_FUNCTION, if non-NULL, is called with the value
associated to that key when the key is destroyed.
DESTR_FUNCTION is not called if the value associated is NULL when
the key is destroyed.  */
IMPORT int nes_posix_pthread_key_create(pthread_key_t*__key, void (*__destr_function) (void*));

/* Destroy KEY.  */
IMPORT int nes_posix_pthread_key_delete(pthread_key_t __key);

/* Store POINTER in the thread-specific data slot identified by KEY. */
IMPORT int nes_posix_pthread_setspecific(pthread_key_t __key,	const void* __pointer);

/* Return current value of the thread-specific data slot identified by KEY.  */
IMPORT void *nes_posix_pthread_getspecific(pthread_key_t __key);

IMPORT int nes_posix_pthread_kill(pthread_t thread, int signo);

/* Terminate all threads in the program except the calling process.
Should be called just before invoking one of the exec*() functions.  */
IMPORT void nes_posix_pthread_kill_other_threads_np(void);

/* Functions for handling cancellation.  */

/* Set cancelability state of current thread to STATE, returning old
state in *OLDSTATE if OLDSTATE is not NULL.  */
IMPORT int nes_posix_pthread_setcancelstate(int state, int *oldstate);

/* Set cancellation state of current thread to TYPE, returning the old
type in *OLDTYPE if OLDTYPE is not NULL.  */
IMPORT int nes_posix_pthread_setcanceltype(int type, int *oldtype);

/* Cancel THREAD immediately or at the next possibility.  */
IMPORT int nes_posix_pthread_cancel(pthread_t cancelthread);

/* Test for pending cancellation for the current thread and terminate
the thread as per pthread_exit(PTHREAD_CANCELED) if it has been
cancelled.  */
IMPORT void nes_posix_pthread_testcancel(void);

/* same with _pthread_cleanup_push */
IMPORT void nes_posix_pthread_cleanup_push_defer_np(void (*routine)(void*), void* arg);

/* removes the most recently installed cleanup handler */
IMPORT void nes_posix_pthread_cleanup_pop_restore_np(int execute);

/*selects a pending signal from set, atomically clears
 * it from the system's set of pending signals, and  returns that signal
 * number in the location referenced by sig. If no signal in set is pending at
 * the time of the call, the thread is suspended until one or more becomes
 * pending
 */ 
IMPORT int nes_posix_sigwait(const sigset_t* set, int *sig);

/* examine and change the signal mask for the calling thread  */
IMPORT int nes_posix_pthread_sigmask(int how, const sigset_t *newmask, sigset_t *oldmask) ;

#define pthread_cleanup_push(routine,arg) \
  { _pthread_cleanup_push((routine), (arg));

/* Remove a cleanup handler installed by the matching pthread_cleanup_push.
   If EXECUTE is non-zero, the handler function is called. */
#define pthread_cleanup_pop(execute) \
    _pthread_cleanup_pop(execute); }

#ifdef __cplusplus
}
#endif

/*
 *Wangjianxiang @ NECsoft(JiNan). 2008/07/11 append
  */
#ifdef __cplusplus
  extern "C" {
#endif

IMPORT int pthread_create(pthread_t* thread, const pthread_attr_t* attr, void (*start_routine)(void*), void* arg);
IMPORT int pthread_join(pthread_t thread, void** thread_return);
IMPORT pthread_t pthread_self(void);
IMPORT int pthread_equal(pthread_t thread1, pthread_t thread2);
IMPORT void pthread_exit(void *retval);
IMPORT int pthread_detach(pthread_t  th);
IMPORT int pthread_attr_init(pthread_attr_t * attr);
IMPORT int pthread_attr_destroy(pthread_attr_t * attr);
IMPORT int pthread_attr_setdetachstate(pthread_attr_t* attr, int detachstate);
IMPORT int pthread_attr_getdetachstate(const pthread_attr_t* attr, int* detachstate);
IMPORT int pthread_attr_setschedparam(pthread_attr_t* attr, const struct sched_param* param);
IMPORT int pthread_attr_getschedparam(const pthread_attr_t* attr, struct sched_param*  param);
IMPORT int pthread_attr_setschedpolicy(pthread_attr_t* attr, int  policy);
IMPORT int pthread_attr_getschedpolicy(const pthread_attr_t* attr, int*  policy);
IMPORT int pthread_attr_setinheritsched(pthread_attr_t* attr, int  inherit);
IMPORT int pthread_attr_getinheritsched(const pthread_attr_t* attr, int* inherit);
IMPORT int pthread_attr_setscope(pthread_attr_t* attr, int  scope);
IMPORT int pthread_attr_getscope(const pthread_attr_t*  attr, int*  scope);
IMPORT int pthread_attr_setstack(pthread_attr_t* attr, void* stackaddr, size_t stacksize);
IMPORT int pthread_attr_getstack(const pthread_attr_t* attr, void** stackaddr, size_t* stacksize);
IMPORT int pthread_attr_setstackaddr(pthread_attr_t* attr, void* stackaddr);
IMPORT int pthread_attr_getstackaddr(const pthread_attr_t* attr, void**  stackaddr);
IMPORT int pthread_attr_setstacksize(pthread_attr_t* attr, size_t stacksize);
IMPORT int pthread_attr_getstacksize(const pthread_attr_t* attr, size_t* stacksize);
IMPORT int pthread_setschedparam(pthread_t target_thread, int  policy, const struct sched_param * param);
IMPORT int pthread_getschedparam(pthread_t  target_thread, int* policy, struct sched_param* param);
IMPORT int pthread_attr_setguardsize(pthread_attr_t* attr, size_t guardsize);
IMPORT int pthread_attr_getguardsize(const pthread_attr_t* attr, size_t* guardsize);
IMPORT int pthread_getattr_np(pthread_t thread, pthread_attr_t *attr);
IMPORT int pthread_once(pthread_once_t *__once_control, void (*__init_routine) (void));
IMPORT int pthread_key_create(pthread_key_t*__key, void (*__destr_function) (void*));
IMPORT int pthread_key_delete(pthread_key_t __key);
IMPORT int pthread_setspecific(pthread_key_t __key,   const void* __pointer);
IMPORT void* pthread_getspecific(pthread_key_t __key);
IMPORT int pthread_setcanceltype(int type, int *oldtype);
IMPORT int pthread_cancel(pthread_t cancelthread);
IMPORT void pthread_testcancel(void);
IMPORT int pthread_kill(pthread_t thread, int signo);
IMPORT  void pthread_kill_other_threads_np(void);
IMPORT int pthread_setcancelstate(int state, int *oldstate);
IMPORT void pthread_cleanup_push_defer_np(void (*routine)(void*), void* arg);
IMPORT void pthread_cleanup_pop_restore_np(int execute);
IMPORT int sigwait(const sigset_t* set, int *sig);
IMPORT int pthread_sigmask(int how, const sigset_t *newmask, sigset_t *oldmask);

#ifdef __cplusplus
  }
#endif
  
#endif // _PTHREAD_H_
