// -*- C++ -*-
/*!
 * @file  rtc_hokuyoaist.cpp * @brief Hokuyo laser scanner component. * $Date$ 
 *
 * $Id$ 
 */
#include "rtc_hokuyoaist.h"

#ifdef __LOG_OUTPUT__
#include "rtc_log.h"
#define LOG_OUTPUT(...) LOGOUT(__VA_ARGS__);
#else
#define LOG_OUTPUT(...)
#endif

#include <flexiport/flexiport.h>

#ifdef __T_KERNEL__
#include <basic.h>
#include <btron/tkcall.h>
#endif // __T_KERNEL__

#ifdef __LOG_OUTPUT__
#define DEF_LOGFILEPATH		"./log/"			// OL^t@CpX
#define DEF_LOGFILENAME		"hokuyoaist"		// OL^t@C
#define DEF_LOGFILEMODE		LOGFILE_MODE_NUMBER	// Ot@C[h
#define DEF_LOGFILEMAXNUM	MAXFILENUM_MIN		// OL^t@Cő吔
#define DEF_LOGFILECOUNTER	FILECOUNTER_DEF		// Ot@CJE^
#define DEF_LOGFILEOUTPUT	OUTPUT_MODE_ON		// Ot@Co
#define DEF_CONSOLEOUTPUT	OUTPUT_MODE_OFF		// R\[o
#endif

// Module specification
// <rtc-template block="module_spec">
static const char* rtc_hokuyoaist_spec[] =
  {
    "implementation_id", "rtc_hokuyoaist",
    "type_name",         "rtc_hokuyoaist",
    "description",       "Hokuyo laser scanner component.",
    "version",           "0.1",
    "vendor",            "Geoffrey Biggs, AIST",
    "category",          "Sensor",
    "activity_type",     "PERIODIC",
    "kind",              "rtc_hokuyoaist",
    "max_instance",      "999",
    "language",          "C++",
    "lang_type",         "compile",
    // Configuration variables
#ifdef __T_KERNEL__
    "conf.default.port_opts", "type=serial,device=rsa,baud=19200,timeout=5",
#else
    "conf.default.port_opts", "type=serial,device=/dev/ttyACM0,timeout=1",
#endif
    // Widget
    "conf.__widget__.port_opts", "text",
    //"conf.__widget__.", "spin",
    // Constraints
    ""
  };

// </rtc-template>

rtc_hokuyoaist::rtc_hokuyoaist(RTC::Manager* manager)
    // <rtc-template block="initializer">
  : RTC::DataFlowComponentBase(manager),
    m_ranges_Out("ranges_", m_ranges_),
    m_intensities_Out("intensities_", m_intensities_),
#ifdef __T_KERNEL__
    m_svc_prov_(),
    svc_port_("ranger"),
    port_opts_("type=serial,device=rsa,baud=19200,timeout=5"),
#else
    m_RangerPort("Ranger"),
    m_HokuyoAistPort("HokuyoAist"),
    port_opts_("type=serial,device=/dev/ttyACM0,timeout=1"),
#endif
    start_angle_(0.0), end_angle_(0.0), cluster_count_(1),
    enable_intns_(false), high_sens_(false), pull_mode_(false),
    new_data_mode_(false), error_time_(5),
    x_(0.0), y_(0.0), z_(0.0), roll_(0.0), pitch_(0.0), yaw_(0.0),
    base_ang_res_(0.0), last_error_time_(0)
    // </rtc-template>
{
#ifdef __LOG_OUTPUT__
  LOG_MODE logMode;
  memset(&logMode, 0, sizeof(LOG_MODE));
  
  // Ot@CpXƃt@C
  strcpy( logMode.fileDirPath, DEF_LOGFILEPATH );
  strcpy( logMode.fileName, DEF_LOGFILENAME);
  // Oo̓[h
  logMode.mode = DEF_LOGFILEMODE;
  // Ot@Cő吔
  logMode.maxNum = DEF_LOGFILEMAXNUM;
  // Ot@CJE^
  logMode.counter = DEF_LOGFILECOUNTER;
  // Ot@Co
  logMode.logFile = DEF_LOGFILEOUTPUT;
  // R\[o
  logMode.console = DEF_CONSOLEOUTPUT;

  // Oo͋@\
  initLog(logMode);
  
#endif
}

rtc_hokuyoaist::~rtc_hokuyoaist()
{
#ifdef __LOG_OUTPUT__
  
  // Oo͋@\I
  finishLog();
  
#endif

}


RTC::ReturnCode_t rtc_hokuyoaist::onInitialize()
{
  LOG_OUTPUT("onInitialize (rtc_hokuyoaist) Start \n");

#ifdef __T_KERNEL__
  bindParameter("port_opts", port_opts_,
          "type=serial,device=rsa,baud=19200,timeout=5");
#else
  bindParameter("port_opts", port_opts_,
          "type=serial,device=/dev/ttyACM0,timeout=1");
#endif
  bindParameter("start_angle", start_angle_, "0.0");
  bindParameter("end_angle", end_angle_, "0.0");
  bindParameter("cluster_count", cluster_count_, "1");
  bindParameter("enable_intensity", enable_intns_, "0");
  bindParameter("high_sensitivity", high_sens_, "0");
  bindParameter("pull_mode", pull_mode_, "0");
  bindParameter("new_data_mode", new_data_mode_, "0");
  bindParameter("error_time", error_time_, "5");

  bindParameter("x", x_, "0.0");
  bindParameter("y", y_, "0.0");
  bindParameter("z", z_, "0.0");
  bindParameter("roll", roll_, "0.0");
  bindParameter("pitch", pitch_, "0.0");
  bindParameter("yaw", yaw_, "0.0");
  
  // Registration: InPort/OutPort/Service
  // <rtc-template block="registration">
  // Set InPort buffers

  // Set OutPort buffer
  addOutPort("ranges_", m_ranges_Out);
  addOutPort("intensities_", m_intensities_Out);

  // Set service provider to Ports
#ifdef __T_KERNEL__
  svc_port_.registerProvider("ranger", "Ranger", m_svc_prov_);
  svc_port_.registerProvider("hokuyoaist", "HokuyoAist",m_specialist_prov_);
#else
  m_RangerPort.registerProvider("svc_prov_", "Ranger", m_svc_prov_);
  m_HokuyoAistPort.registerProvider("specialist_prov_", "HokuyoAist", m_specialist_prov_);
#endif

  // Set service consumers to Ports

  // Set CORBA Service Ports
#ifdef __T_KERNEL__
  addPort(svc_port_);
#else
  addPort(m_RangerPort);
  addPort(m_HokuyoAistPort);
#endif

#ifndef __T_KERNEL__
    m_svc_prov_.setup(this);
    m_specialist_prov_.setup(this);
#endif

  // </rtc-template>

  // <rtc-template block="bind_config">
  // Bind variables and configuration variable
  
  LOG_OUTPUT("onInitialize (rtc_hokuyoaist) End \n");

  // </rtc-template>
  return RTC::RTC_OK;
}

/*
RTC::ReturnCode_t rtc_hokuyoaist::onFinalize()
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onStartup(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onShutdown(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

RTC::ReturnCode_t rtc_hokuyoaist::onActivated(RTC::UniqueId ec_id)
{
    LOG_OUTPUT("onActivated (rtc_hokuyoaist) START \n");
    
    std::cout << "port_opts_:" << port_opts_ << '\n' ;
    std::cout << "start_angle_:" << start_angle_ << '\n' ;
    std::cout << "end_angle_:" << end_angle_ << '\n' ;
    std::cout << "cluster_count_:" << cluster_count_ << '\n' ;
    std::cout << "enable_intns_:" << enable_intns_ << '\n' ;
    std::cout << "high_sens_:" << high_sens_ << '\n' ;
    std::cout << "pull_mode_:" << pull_mode_ << '\n' ;
    std::cout << "new_data_mode_:" << new_data_mode_ << '\n' ;
    std::cout << "error_time_:" << error_time_ << '\n' ;
    std::cout << "x_:" << x_ << '\n' ;
    std::cout << "y_:" << y_ << '\n' ;
    std::cout << "z_:" << z_ << '\n' ;
    std::cout << "roll_:" << roll_ << '\n' ;
    std::cout << "pitch_:" << pitch_ << '\n' ;
    std::cout << "yaw_:" << yaw_ << '\n' ;
    
    coil::Guard<coil::Mutex> guard(mutex_);
    try
    {
        open_laser();
    }
    catch(hokuyo_aist::BaseError &e)
    {
        std::cerr << "Error setting up laser: " << e.what() << '\n';
        
        LOG_OUTPUT("Error setting up laser: %s\n", e.what());
        LOG_OUTPUT("onActivated (rtc_hokuyoaist) End \n");
        
        return RTC::RTC_ERROR;
    }

    sensor_geom_.geometry.pose.position.x = x_;
    sensor_geom_.geometry.pose.position.y = y_;
    sensor_geom_.geometry.pose.position.z = z_;
    sensor_geom_.geometry.pose.orientation.r = roll_;
    sensor_geom_.geometry.pose.orientation.p = pitch_;
    sensor_geom_.geometry.pose.orientation.y = yaw_;

    LOG_OUTPUT("onActivated (rtc_hokuyoaist) End \n");
    
    return RTC::RTC_OK;
}


RTC::ReturnCode_t rtc_hokuyoaist::onDeactivated(RTC::UniqueId ec_id)
{
    LOG_OUTPUT("onDeactivated (rtc_hokuyoaist) Start \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);

    try
    {
        close_laser();
    }
    catch(hokuyo_aist::BaseError &e)
    {
        std::cerr << "Error shutting down laser: " << e.what() << '\n';
        
        LOG_OUTPUT("Error shutting down laser: %s \n", e.what());
        LOG_OUTPUT("onDeactivated (rtc_hokuyoaist) End \n");
        
        return RTC::RTC_ERROR;
    }

    LOG_OUTPUT("onDeactivated (rtc_hokuyoaist) End \n");
    return RTC::RTC_OK;
}


RTC::ReturnCode_t rtc_hokuyoaist::onExecute(RTC::UniqueId ec_id)
{
    LOG_OUTPUT("onExecute (rtc_hokuyoaist) Start \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);
    try
    {
        LOG_OUTPUT("pull_mode_ %d", pull_mode_);
        if(!pull_mode_)
        {
            get_scan();
        }
    }
    catch(hokuyo_aist::BaseError &e)
    {
        std::cerr << "Error getting laser scan: " << e.what() << '\n';
        
        LOG_OUTPUT("Error getting laser scan: %s \n", e.what());
        
        time_t now = time(NULL);
        if (now - last_error_time_ <= error_time_)
        {
            last_error_time_ = now;
            return RTC::RTC_ERROR;
        }
        else
        {
            last_error_time_ = now;
            try
            {
                std::cerr << "Attempting to reset laser.\n";
                
                LOG_OUTPUT("Attempting to reset laser.\n");
                
                reset_laser();
            }
            catch(hokuyo_aist::BaseError &e)
            {
                std::cerr << "Reset failed: " << e.what() << '\n';
                
                LOG_OUTPUT("Reset failed: %s \n", e.what());
                LOG_OUTPUT("onExecute (rtc_hokuyoaist) End \n");
                
                return RTC::RTC_ERROR;
            }
            catch(flexiport::PortException &e)
            {
                std::cerr << "Reset failed: " << e.what() << '\n';
                
                LOG_OUTPUT("Reset failed: %s \n", e.what());
                LOG_OUTPUT("onExecute (rtc_hokuyoaist) End \n");
                
                return RTC::RTC_ERROR;
            }
        }
    }
    
    LOG_OUTPUT("onExecute (rtc_hokuyoaist) End \n");
    return RTC::RTC_OK;
}

/*
RTC::ReturnCode_t rtc_hokuyoaist::onAborting(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onError(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onReset(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onStateUpdate(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/
/*
RTC::ReturnCode_t rtc_hokuyoaist::onRateChanged(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/


void rtc_hokuyoaist::set_power(bool enable)
{
    LOG_OUTPUT("set_power (rtc_hokuyoaist) Start \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);
    laser_.set_power(enable);
    
    LOG_OUTPUT("set_power (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::enable_intensity_data(bool enable)
{
    LOG_OUTPUT("enable_intensity_data (rtc_hokuyoaist) Start \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);
    enable_intns_ = enable;
    
    LOG_OUTPUT("enable_intensity_data (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::set_config(RTC::RangerConfig const& config)
{
    LOG_OUTPUT("set_config (rtc_hokuyoaist) Start \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);

    start_angle_ = sensor_config_.minAngle = config.minAngle;
    end_angle_ = sensor_config_.maxAngle = config.maxAngle;
    sensor_config_.angularRes = config.angularRes;
    cluster_count_ = config.angularRes / base_ang_res_;
    if(cluster_count_ < 1)
        cluster_count_ = 1;
    
    LOG_OUTPUT("set_config (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::request_scan()
{
    LOG_OUTPUT("request_scan (rtc_hokuyoaist) End \n");
    
    coil::Guard<coil::Mutex> guard(mutex_);
    get_scan();
    
    LOG_OUTPUT("request_scan (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::open_laser()
{
    LOG_OUTPUT("open_laser (rtc_hokuyoaist) Start \n");
    
    laser_.open(port_opts_);

    hokuyo_aist::SensorInfo info;
    laser_.get_sensor_info(info);
    if(start_angle_ == 0.0)
    {
        sensor_config_.minAngle = info.min_angle;
    }
    else
    {
        sensor_config_.minAngle = start_angle_;
    }
    if(end_angle_ == 0.0)
    {
        sensor_config_.maxAngle = info.max_angle;
    }
    else
    {
        sensor_config_.maxAngle = end_angle_;
    }
    base_ang_res_ = info.resolution;
    sensor_config_.angularRes = info.resolution * cluster_count_;
    sensor_config_.minRange = info.min_range;
    sensor_config_.maxRange = info.max_range;
    sensor_config_.rangeRes = 0.01;
    sensor_config_.frequency = info.speed / 60.0;

    try
    {
        laser_.set_high_sensitivity(high_sens_);
    }
    catch(hokuyo_aist::ResponseError &e)
    {
        if(e.desc_code() != 30)
        {
            throw;
        }
        std::cerr << "Warning: could not alter high-sensitivity setting "
            "(possibly not supported): " << e.what() << '\n';
        
        LOG_OUTPUT("arning: could not alter high-sensitivity setting (possibly not supported): %s \n", e.what() );
    }

    laser_.calibrate_time();
    if(pull_mode_)
    {
        laser_.set_power(false);
    }
    else
    {
        laser_.set_power(true);
    }

    LOG_OUTPUT("open_laser (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::close_laser()
{
    LOG_OUTPUT("close_laser (rtc_hokuyoaist) Start \n");
    
    laser_.set_power(false);
    laser_.close();
    scan_data_.clean_up();
    
    LOG_OUTPUT("close_laser (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::reset_laser()
{
    LOG_OUTPUT("reset_laser (rtc_hokuyoaist) Start \n");
    
    laser_.reset();
    close_laser();
    open_laser();
    
    LOG_OUTPUT("reset_laser (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::get_scan()
{
    LOG_OUTPUT("get_scan (rtc_hokuyoaist) Start \n");
    
    if(start_angle_ == 0.0 and end_angle_ == 0.0)
    {
        // Get a full scan
        if(new_data_mode_)
        {
            if(enable_intns_)
            {
                laser_.get_new_ranges_intensities(scan_data_, -1, -1,
                        cluster_count_);
            }
            else
            {
                laser_.get_new_ranges(scan_data_, -1, -1, cluster_count_);
            }
        }
        else
        {
            if(enable_intns_)
            {
                laser_.get_ranges_intensities(scan_data_, -1, -1,
                        cluster_count_);
            }
            else
            {
                laser_.get_ranges(scan_data_, -1, -1, cluster_count_);
            }
        }
    }
    else
    {
        // Get a partial scan
        if(new_data_mode_)
        {
            if(enable_intns_)
            {
                laser_.get_new_ranges_intensities_by_angle(scan_data_,
                        start_angle_, end_angle_, cluster_count_);
            }
            else
            {
                laser_.get_new_ranges_by_angle(scan_data_, start_angle_,
                        end_angle_, cluster_count_);
            }
        }
        else
        {
            if(enable_intns_)
            {
                laser_.get_ranges_intensities_by_angle(scan_data_,
                        start_angle_, end_angle_, cluster_count_);
            }
            else
            {
                laser_.get_ranges_by_angle(scan_data_, start_angle_,
                        end_angle_, cluster_count_);
            }
        }
    }

    write_scan();
    
    LOG_OUTPUT("get_scan (rtc_hokuyoaist) End \n");
}


void rtc_hokuyoaist::write_scan()
{
    LOG_OUTPUT("write_scan (rtc_hokuyoaist) Start \n");
    
#if 1
    /* todo: use sensor's time stamp */
    {
        ER ercd = 0;
        SYSTIM tim = {0};
        ercd = tk_get_otm(&tim);
        if (ercd < 0) {
            LOG_OUTPUT("ERROR: tk_get_tim failed, err=%d\n", ercd);
        }
        m_ranges_.tm.sec = ((longlong)tim.hi << 32 | tim.lo) / 1000;
        m_ranges_.tm.nsec = (tim.lo % 1000) * 1000000;
    }
#else
    m_ranges_.tm.sec = scan_data_.system_time_stamp() / 1000000000;
    m_ranges_.tm.nsec = scan_data_.system_time_stamp() % 1000000000;
#endif
    m_ranges_.ranges.length(scan_data_.ranges_length());
    for(unsigned int ii = 0; ii < scan_data_.ranges_length(); ii++)
    {
        if (scan_data_[ii] < 20)
            m_ranges_.ranges[ii] = 0.0;
        else
            m_ranges_.ranges[ii] = scan_data_[ii] / 1000.0;
    }
    m_ranges_.geometry = sensor_geom_;
    m_ranges_.config = sensor_config_;

    if(scan_data_.intensities_length() > 0)
    {
        m_intensities_.tm.sec = scan_data_.system_time_stamp() / 1000000000;
        m_intensities_.tm.nsec = scan_data_.system_time_stamp() % 1000000000;
        m_intensities_.ranges.length(scan_data_.intensities_length());
        for(unsigned int ii = 0; ii < scan_data_.intensities_length(); ii++)
        {
            m_intensities_.ranges[ii] = scan_data_.intensities()[ii];
        }
        m_intensities_.geometry = sensor_geom_;
        m_intensities_.config = sensor_config_;
    }

    LOG_OUTPUT("write_scan (rtc_hokuyoaist) RangeData Write Start \n");
    m_ranges_Out.write();
    LOG_OUTPUT("write_scan (rtc_hokuyoaist) RangeData Write End \n");
    
    LOG_OUTPUT("write_scan (rtc_hokuyoaist) IntensityData Write Start \n");
    m_intensities_Out.write();
    LOG_OUTPUT("write_scan (rtc_hokuyoaist) IntensityData Write End \n");
    
    LOG_OUTPUT("write_scan (rtc_hokuyoaist) End \n");
}

extern "C"
{
 
  void rtc_hokuyoaistInit(RTC::Manager* manager)
  {
    coil::Properties profile(rtc_hokuyoaist_spec);
    manager->registerFactory(profile,
                             RTC::Create<rtc_hokuyoaist>,
                             RTC::Delete<rtc_hokuyoaist>);
  }
  

};



